/***************************************************************************
                         mybarcode.h  -  description
                             -------------------
    begin                : Die Apr 23 2002
    copyright            : (C) 2002 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef MY_BARCODE_H
#define MY_BARCODE_H

#include <qpixmap.h>
#include <qobject.h>

#include <qvaluelist.h>

void init();

typedef union u_settings {
    struct {
        int row;
        int col;
        int err;
    } pdf417;

    struct {
        int size;
    } datamatrix;

    struct {
        double modulewidth;
        bool escape;
    } tbarcode;
};

class QColor;
class QString;
typedef struct barcodeData {
    QString value;
    QString type;
    int margin;
    int rotation;
    double scale;
    double cut;
    bool text;
    bool valid;

    struct _pdf417 {
        int row;
        int col;
        int err;

        bool operator==( const struct _pdf417 p ) const {
            return row == p.row && col == p.col && err == p.err;
        };
        
    } pdf417;

    struct _datamatrix {
        int size;

        bool operator==( const struct _datamatrix d ) const {
            return size == d.size;
        };
        
    } datamatrix;

    struct _tbarcode {
        double modulewidth;
        bool escape;
        bool above;
        bool autocorrect;
        int checksum;

        bool operator==( const struct _tbarcode t ) const {
            return modulewidth == t.modulewidth && escape == t.escape && above == t.above && checksum == t.checksum && autocorrect == t.autocorrect;
        };
        
    } tbarcode;

    struct _colors {
        bool invert;

        bool operator==( const struct _colors c ) const {
            return invert == c.invert;
        }
    } colors;
                                
    struct {
        // only important
        // when this struct
        // is read from XML
        QString caption;
        int x;
        int y;
    } xml;

    /**
      * Compare to barcodeData structs,
      * but only the fields which are used
      * as commandline parameters for
      * the genaration backends.
      * E.g. margin is not compared, because
      * the margin is created by KBarcode internally.
      */
    bool operator==( const struct barcodeData d ) const;

    struct {
        bool enabled;
        int mode;
        int step;
        int start;
    } sequence;
};

/* Serial Indexing/Sequence Modes */
enum { NUM, ALPHA, ALPHANUM };

enum { NOCUT = 0x00001, MULTILINE = 0x00002, NOTEXT = 0x00004, NOSCALE = 0x00008,
       /**
         * The different Generator Backends
         */
       GNU_BARCODE = 0x00010, PDF417 = 0x00020, TBARCODE = 0x00040,
       /**
         * Different advanced option modes.
         */
       DATAMATRIX = 0x00080, PDF417BARCODE = 0x00100, TBARCODEADV = 0x00200,
       /**
         * Checksum ID's
         */
       POSTNETCHECK = 0x01000, CODE128CHECK = 0x02000, EAN8CHECK = 0x04000, EAN13CHECK = 0x08000,
       UPCACHECK = 0x10000, UPCECHECK = 0x20000, MODULOALLCHECK = 0x40000, MODULO10CHECK = 0x80000,
       /**
         * Short cut's
         */
       BARCODE2D =  NOCUT | MULTILINE | NOTEXT };

typedef struct barcodeFormat {
    QString name;
    QString text;
    unsigned int features;
};

class KProcess;
class KTempFile;
class QPaintDevice;
/**
  * This class creates a QPixmap from a barcodeData struct.
  * Several backends are supported to generate the pixmap.
  * GNU Barcode
  * pdf417_enc
  * TBarcode (http://www.tec-it.com)
  *
  * This tools generate a postscript file. The ps file is
  * transformed to a BMP file scaled to the requested resolution
  * or screen resolution using convert (part of imagemagick)
  *
  * The generated barcodes are cached using BarcodeCache.
  * To fill a barcodeData struct with data from the user, you
  * might want to use BarcodeWidget.
  *
  * A small example, of how to generate a valid Barcode in your application
  * without the use of BarcodeWidget:
  * <pre>
  *  barcodeData data;
  *  BarCode::fillDefault( &data ); // setup a barcode object with valid data
  *  data.value = "KBARCODE";
  *  data.type = "code39";
  *  BarCode b( &data ); // create a barcode object
  *  QPixmap p = b.pixmap(); // get a pixmap of the barcode
  * </pre>
  *
  * @see BarcodeCache @see BarcodeWidget
  * @author Dominik Seichter  
  */
class BarCode {
    public:
        /** Create a BarCode object from the data const barcodeData* @p data.
          * You may delete @p data afterwards. Be sure to call at least once
          * BarCode::fillDefault() on @p data. Otherwise you rsik invalid barcode data.
          */
        BarCode( const barcodeData* data );
        /** Create a BarCode object filled with default data.
          */
        BarCode();
        ~BarCode();

        /** Get a QPixmap of the current barcodeData in
          * screen resolution-
          */
        const QPixmap pixmap();
        /** Get a QPixmap of the barcode in the same resolution as the
          * QPaintDevice* @p device. This is necessary if you want to print
          * your barcodes, as they won't be scanable in screen resolution.
          */
        const QPixmap printerPixmap( QPaintDevice* device );

        /** Set the index of the barcodes for barcode sequences.
          */
        void setIndex( int i ) { m_index = i; }
        /** Change the barcodeData to @p d, call redraw() afterwards.
          */        
        void setData( barcodeData* d ) { barcode = *d; }
        bool validBarcode() {return !p.isNull();}

        barcodeData data() const { return barcode; }

        QString sequenceValue();

        /** Fill @p barcodeData* data with its default parameters.
          * Always call this function, after you have created a new
          * barcodeData struct.
          */
        static void fillDefault( barcodeData* data );

        /** Test if the encodingType @p type has a certain 
          * @p feature, as defined in the codes QValueList.
          */
        static bool hasFeature( const QString & type, unsigned int feature );

        /** get the meximum length of value for the encoding type @p name
          * from the SQL databases.
          */
        static const QString getMaxLength( const QString & name );

        /** This function has to be called befor barcode support is avaible.
          * It searches for convert and barcode generation backends
          * (gnu barcode, tbarcode and pdf417) and fills the QValueList codes
          * with data and information about the avaible barcode encoding types.
          * You only have to call this function once.
          * The results can be retrieved with the following functions:
          * @see haveBarcode()
          * @see haveTBarcode()
          * @see havePDFBarcode()
          */
        static void setHaveBarcode();
        static bool haveConvert() {
            return m_haveConvert;
        }
        static bool haveBarcode() {
            return (m_haveConvert && ( m_haveGnuBarcode || m_havePdfEncode || m_haveTBarcode ));
        }
        static bool haveGNUBarcode() {
            return m_haveGnuBarcode;
        }
        static bool haveTBarcodeInstalled() {
            return m_haveTBarcode;
        }
        static bool havePDFInstalled() {
            return m_havePdfEncode;
        }
        static bool haveTBarcode() {
            return (m_haveConvert && m_haveTBarcode);
        }
        static bool havePDFBarcode() {
            return (m_haveConvert && m_havePdfEncode);
        }

        /** call this, always after you changed the barcodeData using setData,
          * so that the pixmaps are recreated.
          */
        void redrawBarcode();
    private:
        bool createPixmap( QPixmap* target, int res );
        void createBarcode( QPixmap* target, QPaintDevice* device );
        bool createPdf417( KTempFile* output );
        bool createTBarcode( KTempFile* output );
        QString createSequence( const QString & value );
        
        void cleanUp( KTempFile* file, KTempFile* file2, QPixmap* target );
        static barcodeFormat fillStruct( const QString & name, const QString & text, const int feature );

        /**
          * Cut the barcode either on the top or
          * on the bottom, depending on the text position.
          *
          * @param pic the QPixmap that will be cutted
          * @param cut a value between 0.0 and 1.0. If cut = 1.0
          *        then no cut will happen, if cut = 0.5, half of
          *        the barcode is going to be cutted away.
          */
        QPixmap cut( QPixmap* pic, double cut );
        
        /**
          * Add a margin to the QPixmap pic.
          * @param pic the QPixmap pic which should get a margin
          * @param margin the margin size in pixels.
          */
        QPixmap addMargin( QPixmap* pic, int margin );


        /**
          * The barcode is generated from this value.
          * This is necessary so that sequences will work.
          */
        QString m_value;

        /** index for barcode sequences;
          */
        int m_index;
                
        QPixmap p;
        QPixmap pp;
        barcodeData barcode;
        static bool m_haveGnuBarcode;
        static bool m_havePdfEncode;
        static bool m_haveTBarcode;
        static bool m_haveConvert;      
};

#endif
