/***************************************************************************
                          mycanvasitem.cpp  -  description
                             -------------------
    begin                : Die Apr 23 2002
    copyright            : (C) 2002 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "mycanvasitem.h"
#include "labelutils.h"
#include "labeleditor.h"

// KDE includes
#include <kapplication.h>

// QT includes
#include <qbitmap.h>
#include <qpainter.h>

SpotProvider::SpotProvider()
{
    m_spot = new QPixmap( SPOTSIZE, SPOTSIZE );
    m_spot->fill( Qt::green );

    QPainter p( m_spot );
    p.drawRect( 0, 0, SPOTSIZE, SPOTSIZE );
    p.end();
}

SpotProvider::~SpotProvider()
{
    delete m_spot;
}

SpotProvider* SpotProvider::instance = 0;
QPixmap* SpotProvider::m_spot = 0;

SpotProvider* SpotProvider::getInstance()
{
    if( !instance )
        instance = new SpotProvider();

    return instance;
}

/**********************************************/
MyCanvasRectangle::MyCanvasRectangle( QCanvas* c, double rotate )
    : QCanvasRectangle( c )
{
    m_autosize = true;
    m_color = Qt::black;
    rot = rotate;
    setSelected( false );
    setSize( 0, 0 );
}

MyCanvasRectangle::~MyCanvasRectangle()
{
}

void MyCanvasRectangle::setPixmap( QPixmap* p )
{
    pix = (*p);
    setSize( pix.width(), pix.height() );
    show();
}
void MyCanvasRectangle::setPicture( QPicture* p, int w, int h )
{
    picture = (*p);
    setSize( w, h );
    show();
}

void MyCanvasRectangle::setRot( double r )
{
    rot = r ;
}

void MyCanvasRectangle::setColor( const QColor & c )
{
    m_color = c;
    redraw();
}

void MyCanvasRectangle::draw( QPainter & p )
{
    p.save();

    // draw the pixmap
    QWMatrix wm = p.worldMatrix();
    if( picture.isNull() ) {
        QPixmap picture = getTransformedPixmap(); //.xForm( wm );

        if( m_autosize )
            setSize( (int)picture.width(), (int)picture.height() );

        QPoint point( (int)x(), (int)y() );
        point = wm * point;
        QPixmap pixtmp = picture.xForm( wm );
        bitBlt( p.device(), point.x(), point.y(), &pixtmp );
    } else {
        p.save();
        p.rotate( rot );
        p.drawPicture( (int)x(), (int)y(), picture );
        p.restore();
    }
    
    // draw edges
    if( isSelected() ) {
        const QPixmap* spot = SpotProvider::getInstance()->spot();
        p.translate( x(), y() );

        // top left
        p.drawPixmap( 0, 0, *spot );
        // bottom left
        p.drawPixmap( 0, height()-SPOTSIZE, *spot );
        // top right
        p.drawPixmap( width()-SPOTSIZE, 0, *spot );
        // bottom left
        p.drawPixmap( width()-SPOTSIZE, height()-SPOTSIZE, *spot );

        if( (width() - 2 * SPOTSIZE ) / 2 > SPOTSIZE ) {
            // top middle
            p.drawPixmap( (width()-SPOTSIZE)/2, 0, *spot );
            // bottom middle
            p.drawPixmap( (width()-SPOTSIZE)/2, height()-SPOTSIZE, *spot );
        }

        if( (height() - 2 * SPOTSIZE ) / 2 > SPOTSIZE ) {
            // left middle
            p.drawPixmap( 0, (height()-SPOTSIZE)/2, *spot );
            // right middle
            p.drawPixmap( width() - SPOTSIZE, (height()-SPOTSIZE)/2, *spot );
        }
    }
                                   
    p.restore();
}

QPixmap MyCanvasRectangle::getTransformedPixmap()
{
    if( rot != 0.0 && m_autosize ) {
        QWMatrix m;
        m.rotate( rot );
        return pix.xForm( m );
    } else
        return pix;
}

/*****************************************************************/

CanvasBarcode::CanvasBarcode( QCanvas* c )
    : MyCanvasRectangle( c )
    , BarCode()
{ }

CanvasBarcode::CanvasBarcode( QCanvas* c, barcodeData* bcode )
    : MyCanvasRectangle( c )
    , BarCode( bcode )
{ }

CanvasBarcode::~CanvasBarcode()
{ }

void CanvasBarcode::updateBarcode()
{
    redrawBarcode();
    QPixmap tmp = pixmap();
    if(!tmp.isNull())
        setPixmap( &tmp );
}

/*****************************************************************/
CanvasText::CanvasText( QCanvas* c, LabelEditor* editor )
    : MyCanvasRectangle( c ),
      TokenProvider( KApplication::desktop() ),
      stext( QString::null ),
      m_editor( editor )
{
    init();
}

CanvasText::CanvasText( const QString & t, QCanvas* c, LabelEditor* editor )
    : MyCanvasRectangle( c ),
      TokenProvider( KApplication::desktop() ),
      stext( t ),
      m_editor( editor )      
{
    init();
}

CanvasText::~CanvasText()
{ }

void CanvasText::init()
{
    setAutoSize( false );
    draw();
}

void CanvasText::draw()
{
    // to make sure that [filename] is always displayed correctly
    setLabelName( m_editor->fileName() );

    QPicture picture;
    QString text = parse( stext );
    QSize s = LabelUtils::stringSize( text );
    int ww =  width() ? width() : s.width();
    int hh =  height() ? height() : s.height() + 4; // DS Rich text has a different line spacing than QRichText

    
    if( getRotation() == 0.0 ) {
        QPainter painter( &picture );

        LabelUtils::renderString( &painter, text, width(), height(), 1.0, 1.0 );
        painter.end();

        QPixmap pix;
        setPixmap( &pix );   
        setPicture( &picture, ww, hh );
    } else {
        QPixmap* pixmap = LabelUtils::drawString( text, ww, hh, getRotation() );

        setPicture( &picture, ww, hh );
        setPixmap( pixmap );
        delete pixmap;
    }
}

/*****************************************************************/
DrawingRect::DrawingRect( QCanvas* c )
    : MyCanvasRectangle( c )
{
    init();
    
    setColor( Qt::black );
    draw();
}

DrawingRect::DrawingRect( const QColor & cl, QCanvas* c )
    : MyCanvasRectangle( c )
{
    init();

    setColor( cl );
    draw();
}

DrawingRect::~DrawingRect()
{ }

void DrawingRect::init()
{
    m_circle = false;
    m_borderwidth = 1;
    m_borderstyle = QPen::SolidLine;
    m_bordercolor = Qt::black;

    setSize( 60, 60 );
    setAutoSize( false );
}

void DrawingRect::draw()
{
    if( width() > 0 && height() > 0 ) {
        QPicture pix;
        QPainter p(&pix);
        int w = m_borderstyle ? m_borderwidth : 0;
        p.setPen( QPen( m_bordercolor, w, (Qt::PenStyle)m_borderstyle ) );
        p.setBrush( QBrush( color(), Qt::SolidPattern ) );

        QRect rect( 0 + w, 0 + w, width() - 2 * w, height() - 2 * w );
        if( m_circle ) {
            p.drawEllipse( rect );
        } else
            p.drawRect( rect );
            
        p.end();
        setPicture( &pix, width(), height() );
    }
}

/*****************************************************************/
PictureRectangle::PictureRectangle( QCanvas* c )
    : MyCanvasRectangle( c )
{
    setAutoSize( false );
    draw();
}

void PictureRectangle::draw()
{
    if( pic.isNull() )
        return;

    QImage img = pic.copy();
    if( getRotation() != 0.0 ) {
        QWMatrix wm;
        wm.rotate( getRotation() );
        img = img.xForm( wm );
    }

    img = img.scale( width(), height() );
    
    QPixmap pix( img );

    setPixmap( &pix );
}

void PictureRectangle::setImage( QImage* img )
{
    if( img ) {
        pic = (*img);
        setSize( pic.width(), pic.height() );
        redraw();
    }
}

MyCanvasLine::MyCanvasLine( QCanvas* c )
    : QCanvasItem( c )
{
    m_pen = QPen( Qt::black );
    move( 0, 0 );
    x1 = y1 = 0;
}

MyCanvasLine::~MyCanvasLine()
{
    hide();
}

QRect MyCanvasLine::boundingRect() const
{
    QRect r;
    r = r.unite( QRect( startPoint(), QSize( SPOTSIZE, SPOTSIZE ) ) );
    r = r.unite( QRect( endPoint(), QSize( SPOTSIZE, SPOTSIZE ) ) );

    if( r.height() < (signed int)m_pen.width() ) {
        r.setY( (int)y() - m_pen.width() / 2 );
        r.setHeight( m_pen.width() );
    }

    if( r.width() < (signed int)m_pen.width() ) {
        r.setX( (int)x() - m_pen.width() / 2 );
        r.setWidth( m_pen.width() );
    }
                
    return r;
}

void MyCanvasLine::draw( QPainter & p )
{
    p.setPen( m_pen );
    p.drawLine( (int)x(), (int)y(), x1, y1 );

    if( isSelected() ) {
        p.save();    
            const QPixmap* spot = SpotProvider::getInstance()->spot();
            p.drawPixmap( startPoint(), *spot );
            p.drawPixmap( endPoint(), *spot );
        p.restore();
    }
}

QPoint MyCanvasLine::startPoint() const
{
    return QPoint( (int)x(), (int)y() );
}

QPoint MyCanvasLine::endPoint() const
{
    return QPoint( x1, y1 );
}

void MyCanvasLine::setPoints( int xa, int ya, int xb, int yb )
{
    move( xa, ya );
    x1 = xb;
    y1 = yb;
}

void MyCanvasLine::moveBy( double dx, double dy )
{
    QCanvasItem::moveBy( dx, dy );
    x1 += (int)dx;
    y1 += (int)dy;
}

void MyCanvasLine::move( double mx, double my )
{
    moveBy( mx-x(), my-y() );
}

void MyCanvasLine::setStartPoint( int xa, int ya )
{
    move( xa, ya );
}

