/***************************************************************************
*   Copyright (C) 2004-2006 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <file.h>
#include <element.h>
#include <entry.h>
#include <macro.h>
#include <comment.h>
#include <encoderlatex.h>

#include "fileexporterbibtex.h"

namespace BibTeX
{

    FileExporterBibTeX::FileExporterBibTeX() : FileExporter(),
            m_stringOpenDelimiter( '"' ), m_stringCloseDelimiter( '"' ), m_encoding( File::encLaTeX ), cancelFlag( FALSE )
    {
        // nothing
    }


    FileExporterBibTeX::~FileExporterBibTeX()
    {
        // nothing
    }


    bool FileExporterBibTeX::save( QIODevice* iodevice, File* bibtexfile, QStringList * /*errorLog*/ )
    {
        bool result = TRUE;
        QTextStream stream( iodevice );
        if ( m_encoding == File::encUTF8 )
            stream.setEncoding( QTextStream::UnicodeUTF8 );

        writePreamble( stream, bibtexfile->preamble() );

        // first, write strings at the beginning
        for ( File::ElementList::iterator it = bibtexfile->elements.begin(); it != bibtexfile->elements.end() && result && !cancelFlag; it++ )
        {
            Macro *macro = dynamic_cast<Macro*>( *it );
            if ( macro != NULL )
                result &= writeMacro( stream, macro );
        }

        // second, write cross-referencing elements
        for ( File::ElementList::iterator it = bibtexfile->elements.begin(); it != bibtexfile->elements.end() && result && !cancelFlag; it++ )
        {
            Entry *entry = dynamic_cast<Entry*>( *it );
            if ( ( entry != NULL ) && ( entry->getField( EntryField::ftCrossRef ) != NULL ) )
                result &= writeEntry( stream, entry );
        }

        // third, write remaining elements
        for ( File::ElementList::iterator it = bibtexfile->elements.begin(); it != bibtexfile->elements.end() && result && !cancelFlag; it++ )
        {
            Entry *entry = dynamic_cast<Entry*>( *it );
            if ( entry != NULL )
            {
                if ( entry->getField( EntryField::ftCrossRef ) == NULL )
                    result &= writeEntry( stream, entry );
            }
            else
            {
                Comment *comment = dynamic_cast<Comment*>( *it );
                if ( comment != NULL )
                    result &= writeComment( stream, comment );
            }
        }

        return result && !cancelFlag;
    }

    bool FileExporterBibTeX::save( QIODevice* iodevice, Element* element, QStringList * /*errorLog*/ )
    {
        bool result = FALSE;
        QTextStream stream( iodevice );
        if ( m_encoding == File::encUTF8 )
            stream.setEncoding( QTextStream::UnicodeUTF8 );

        Entry *entry = dynamic_cast<Entry*>( element );
        if ( entry != NULL )
            result |= writeEntry( stream, entry );
        else
        {
            Macro * macro = dynamic_cast<Macro*>( element );
            if ( macro != NULL )
                result |= writeMacro( stream, macro );
            else
            {
                Comment * comment = dynamic_cast<Comment*>( element );
                if ( comment != NULL )
                    result |= writeComment( stream, comment );
            }
        }

        return result;
    }

    void FileExporterBibTeX::cancel()
    {
        cancelFlag = TRUE;
    }

    bool FileExporterBibTeX::writeEntry( QTextStream&stream, Entry* entry )
    {
        stream << "@" << entry->entryTypeString() << "{ " << entry->id();

        for ( Entry::EntryFields::const_iterator it = entry->begin(); it != entry->end(); it++ )
        {
            EntryField *field = *it;
            stream << ',' << endl << '\t' << field->fieldTypeName() << " = " << valueToString( field->value(), field->fieldType() );
        }
        stream << endl << "}" << endl << endl;
        return TRUE;
    }

    bool FileExporterBibTeX::writeMacro( QTextStream&stream, Macro *macro )
    {
        stream << "@string{ " << macro->key() << " = " << valueToString( macro->value() ) << " }" << endl << endl;

        return TRUE;
    }

    bool FileExporterBibTeX::writeComment( QTextStream&stream, Comment *comment )
    {
        if ( comment->usePercent() )
        {
            QString text = comment->text() ;
            escapeLaTeXChars( text );

            switch ( m_encoding )
            {
            case File::encLaTeX:
                text = EncoderLaTeX::currentEncoderLaTeX() ->encode( text );
                break;
            default:
                stream.setEncoding( QTextStream::UnicodeUTF8 );
            }
            QStringList commentLines = QStringList::split( '\n', text );
            for ( QStringList::Iterator it = commentLines.begin(); it != commentLines.end(); it++ )
            {
                //                 qDebug( "writeComment %s", ( *it ).latin1() );
                stream << '%' << *it << endl;
            }
            stream << endl;
        }
        else
        {
            QString text = comment->text() ;
            escapeLaTeXChars( text );

            switch ( m_encoding )
            {
            case File::encLaTeX:
                text = EncoderLaTeX::currentEncoderLaTeX() ->encode( text );
                break;
            default:
                stream.setEncoding( QTextStream::UnicodeUTF8 );
            }
            stream << "@comment{" << text << "}" << endl << endl;
        }
        return TRUE;
    }

    bool FileExporterBibTeX::writePreamble( QTextStream &stream, const QString& preamble )
    {
        if ( !preamble.isEmpty() )
        {
            QString text = preamble ;
            escapeLaTeXChars( text );

            switch ( m_encoding )
            {
            case File::encLaTeX:
                text = EncoderLaTeX::currentEncoderLaTeX() ->encode( text );
                break;
            default:
                stream.setEncoding( QTextStream::UnicodeUTF8 );
            }

            stream << "@preamble{" << text << "}" << endl << endl;
        }
        return TRUE;
    }

    void FileExporterBibTeX::setStringDelimiter( const QChar& stringOpenDelimiter, const QChar& stringCloseDelimiter )
    {
        m_stringOpenDelimiter = stringOpenDelimiter;
        m_stringCloseDelimiter = stringCloseDelimiter;
    }

    void FileExporterBibTeX::setEncoding( File::Encoding encoding )
    {
        m_encoding = encoding;
    }

    QString FileExporterBibTeX::valueToString( const Value *value, const EntryField::FieldType fieldType )
    {
        QString result;
        bool isFirst = TRUE;
        EncoderLaTeX *encoder = EncoderLaTeX::currentEncoderLaTeX();

        for ( QValueList<ValueItem*>::const_iterator it = value->begin(); it != value->end(); it++ )
        {
            if ( !isFirst )
                result.append( " # " );
            isFirst = FALSE;

            ValueItem *valueItem = *it;
            if ( !valueItem->isStringKey() )
            {
                QString text = valueItem->text() ;
                escapeLaTeXChars( text );

                switch ( m_encoding )
                {
                case File::encLaTeX:
                    text = encoder->encodeSpecialized( text, fieldType ) ;
                    break;
                default:
                    {
                        // nothing
                    }
                }

                // if the text to save contains a quote char ("),
                // force string delimiters to be curly brackets,
                // as quote chars as string delimiters would result
                // in parser failures
                QChar stringOpenDelimiter = m_stringOpenDelimiter;
                QChar stringCloseDelimiter = m_stringCloseDelimiter;
                if ( text.contains( '"' ) && ( m_stringOpenDelimiter == '"' || m_stringCloseDelimiter == '"' ) )
                {
                    stringOpenDelimiter = '{';
                    stringCloseDelimiter = '}';
                }

                result.append( stringOpenDelimiter );
                result.append( text );
                result.append( stringCloseDelimiter );
            }
            else
                result.append( valueItem->text() );

            //             qDebug( "valueToString=%s", result.latin1() );
        }

        return result;
    }

    void FileExporterBibTeX::escapeLaTeXChars( QString &text )
    {
        text.replace( "&", "\\&" );
    }
}
