/***************************************************************************
 *   Copyright (C) 2004-2009 by Thomas Fischer                             *
 *   fischer@unix-ag.uni-kl.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <qfile.h>
#include <qapplication.h>
#include <qregexp.h>
#include <qtimer.h>
#include <qmap.h>
#include <qbuffer.h>
#include <qspinbox.h>

#include <klineedit.h>
#include <klocale.h>
#include <kdebug.h>
#include <kmessagebox.h>

#include <settings.h>
#include "webquerygooglescholar.h"

namespace KBibTeX
{
    WebQueryGoogleScholarWidget::WebQueryGoogleScholarWidget( QWidget *parent, const char *name )
            : WebQueryWidget( parent, name )
    {
        init();

        Settings *settings = Settings::self();
        QString value = settings->getWebQueryDefault( "GoogleScholar" );
        value = value == QString::null ? "" : value;
        lineEditQuery->setText( value );
        slotTextChanged( value, true );
    }

    WebQueryGoogleScholar::WebQueryGoogleScholar( QWidget* parent ): WebQuery( parent ), m_http( NULL ), m_restarting( false ), m_restartEnabled( false ), m_buffer( NULL )
    {
        m_importer = new BibTeX::FileImporterBibTeX( FALSE );
        m_importer->setIgnoreComments( TRUE );
        m_widget = new WebQueryGoogleScholarWidget( parent );
    }

    WebQueryGoogleScholar::~WebQueryGoogleScholar()
    {
        delete m_widget;
        delete m_importer;
        if ( m_buffer != NULL ) delete m_buffer;
        if ( m_http != NULL ) delete m_http;
    }

    QString WebQueryGoogleScholar::title()
    {
        return i18n( "Google Scholar" );
    }

    QString WebQueryGoogleScholar::disclaimer()
    {
        return i18n( "About Google Scholar" );
    }

    QString WebQueryGoogleScholar::disclaimerURL()
    {
        return "http://scholar.google.com/intl/en/scholar/about.html";
    }

    WebQueryWidget *WebQueryGoogleScholar::widget()
    {
        return m_widget;
    }

    void WebQueryGoogleScholar::query()
    {
        WebQuery::query();
        Settings *settings = Settings::self();
        settings->setWebQueryDefault( "GoogleScholar", m_widget->lineEditQuery->text() );

        m_searchTerm = m_widget->lineEditQuery->text().stripWhiteSpace().replace( '$', "" );
        m_searchTerm = m_searchTerm.replace( "%", "%25" ).replace( "+", "%2B" ).replace( " ", "%20" ).replace( "#", "%23" ).replace( "&", "%26" ).replace( "?", "%3F" );

        m_numberOfResults = m_widget->spinBoxMaxHits->value();
        m_currentRequestNumber = 0;
        setNumStages( m_numberOfResults + 7 );

        if ( m_searchTerm.isEmpty() )
        {
            setEndSearch( WebQuery::statusInvalidQuery );
            return;
        }

        m_http = new QHttp();
        connect( m_http, SIGNAL( done( bool ) ), this, SLOT( done( bool ) ) );
        connect( m_http, SIGNAL( responseHeaderReceived( const QHttpResponseHeader & ) ), this, SLOT( headerReceived( const QHttpResponseHeader & ) ) );

        m_cookieMap.clear();
        m_hits.clear();
        m_redirectLocation = "";
        m_state = gsInit;
        m_referer = KURL( "http://www.google.com/" );
        KURL url( "http://scholar.google.com/" );
        startRequest( url );

        m_restartEnabled = false;
        QTimer::singleShot( 2000, this, SLOT( restart() ) );
    }

    void WebQueryGoogleScholar::cancelQuery()
    {
        if ( m_http != NULL )
        {
            kdDebug() << "state= " << m_http->state() << endl;
            kdDebug() <<  "m_aborted= " << ( m_aborted ? "TRUE" : "FALSE" ) << endl;
            m_http->abort();
            setEndSearch( WebQuery::statusError );
        }
    }

    void WebQueryGoogleScholar::done( bool error )
    {
        kdDebug() << "WebQueryGoogleScholar::done ( " << error << ", " << m_restarting << " )" << endl;

        m_buffer->close();

        if ( m_restarting )
            return;

        if ( m_aborted )
        {
            setEndSearch( WebQuery::statusAborted );
        }
        else if ( error )
        {
            m_restartEnabled = false;
            KMessageBox::error( m_parent, QString( i18n( "Querying database '%1' failed." ) ).arg( title() ) );
            setEndSearch( WebQuery::statusError );
        }
        else
        {
            ++m_currentRequestNumber;
            enterNextStage();

            if ( !m_redirectLocation.isEmpty() )
            {
                KURL url( m_redirectLocation );
                startRequest( url );
            }
            else if ( m_state == gsInit &&  m_referer.host().contains( "scholar." ) )
            {
                m_state = gsGetPref;
                KURL url = KURL( QString( "http://%1/scholar_preferences" ).arg( m_referer.host() ) );
                startRequest( url );
            }
            else if ( m_state == gsGetPref )
            {
                m_state = gsSetPref;
                KURL url = KURL( QString( "http://%1/scholar_setprefs?inststart=0&hl=en&lang=all&instq=&inst=cat-oclc&num=%2&scis=yes&scisf=4&submit=Save+Preferences" ).arg( m_referer.host() ).arg( m_numberOfResults ) );
                startRequest( url );
            }
            else if ( m_state == gsSetPref )
            {
                m_state = gsQuery;
                KURL url = KURL( QString( "http://%1/scholar?q=%3&hl=en&lr=&num=%2&ie=UTF-8&btnG=Search&lr=" ).arg( m_referer.host() ).arg( m_numberOfResults ).arg( m_searchTerm ) );
                startRequest( url );
            }
            else if ( m_state == gsQuery )
            {
                m_state = gsResult;
                m_buffer->open( IO_ReadOnly );
                QTextStream ts( m_buffer );
                QString html = ts.read();
                m_buffer->close();

                int p = 0;
                while (( p = html.find( "\"/scholar.bib?", p ) ) >= 0 )
                {
                    int p2 = html.find( "\"", p + 2 );
                    QString hit = html.mid( p + 1, p2 - p - 1 );
                    m_hits.append( hit );
                    p = p2;
                }

                if ( m_hits.size() > 0 )
                {
                    QString hit = *m_hits.begin();
                    m_hits.remove( m_hits.begin() );

                    KURL url = KURL( QString( "http://%1%2" ).arg( m_referer.host() ).arg( hit ) );
                    startRequest( url );
                }
                else
                    setEndSearch( WebQuery::statusSuccess );
            }
            else if ( m_state == gsResult )
            {
                m_buffer->open( IO_ReadOnly );
                BibTeX::File *tmpBibFile = m_importer->load( m_buffer );
                m_buffer->close();

                if ( tmpBibFile != NULL )
                {
                    for ( BibTeX::File::ElementList::iterator it = tmpBibFile->begin(); it != tmpBibFile->end(); ++it )
                    {
                        BibTeX::Entry *entry = dynamic_cast<BibTeX::Entry*>( *it );

                        if ( entry != NULL )
                            emit foundEntry( new BibTeX::Entry( entry ), false );
                    }

                    delete tmpBibFile;
                }

                if ( m_hits.size() > 0 )
                {
                    QString hit = *m_hits.begin();
                    m_hits.remove( m_hits.begin() );

                    KURL url = KURL( QString( "http://%1%2" ).arg( m_referer.host() ).arg( hit ) );
                    startRequest( url );
                }
                else
                    setEndSearch( WebQuery::statusSuccess );
            }
        }
    }

    void WebQueryGoogleScholar::headerReceived( const QHttpResponseHeader &resp )
    {
        kdDebug() << "WebQueryGoogleScholar::headerReceived ( " << resp.toString() << " )" << endl;

        m_redirectLocation = "";

        QStringList keys = resp.keys();
        for ( QStringList::Iterator it = keys.begin(); it != keys.end(); ++it )
        {
            QString key = *it;
            QString value = resp.value( key );
            if ( key == "location" )
                m_redirectLocation = value;
            else if ( key == "set-cookie" )
            {
                QRegExp rx( "^([^=]+)=([^;]+)" );
                rx.search( value );
                m_cookieMap.insert( rx.cap( 1 ), rx.cap( 2 ) );
            }
        }

    }

    void WebQueryGoogleScholar::restart()
    {
        if ( m_state == gsInit && m_restartEnabled )
        {
            m_restarting = true;
            qDebug( "Nothing happens, restarting search" );
            m_http->abort();
            m_restarting = false;
            query();
        }
    }

    void WebQueryGoogleScholar::startRequest( KURL& url )
    {
        kdDebug() << "WebQueryGoogleScholar::startRequest ( " << url.prettyURL() << " )" << endl;
        QHttpRequestHeader header( "GET", url.encodedPathAndQuery() );
        QString userAgent = "Mozilla/4.61 [dk] (OS/2; U)";
        header.setValue( "Host", url.host() );
        header.setValue( "User-Agent", userAgent );
        header.setValue( "Cookie", getCookies() );
        header.setValue( "Referer", m_referer.prettyURL() );
        m_referer = url;

        if ( m_buffer != NULL )
            delete m_buffer;
        m_buffer = new QBuffer();
        m_buffer->open( IO_WriteOnly );
        m_http->setHost( url.host(), 80 );
        m_http->request( header, NULL, m_buffer );
    }

    QString WebQueryGoogleScholar::getCookies()
    {
        QStringList cookieList;
        for ( QMap<QString, QString>::Iterator it = m_cookieMap.begin(); it != m_cookieMap.end();++it )
            cookieList.append( it.key() + "=" + it.data() );
        return cookieList.join( ";" );
    }
}
