/***************************************************************************

   Copyright (C) 2007 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "backend.h"

#include <klocale.h>
#include <kmessagebox.h>

#include <kblog/blogpost.h>
#include <kblog/blogmedia.h>

#include "itemsmanager.h"
#include "kbloggerpost.h"
#include "kbloggermedia.h"
#include "kbloggerblog.h"
#include "kblogger.h"
#include "profileconfig.h"
#include "backendjobs.h"
#include "waitwidget.h"

Q_DECLARE_METATYPE(KBlogger::KBloggerPost*);
Q_DECLARE_METATYPE(KBlog::BlogMedia*);

namespace KBlogger
{

Backend *Backend::s_self = 0;

Backend *Backend::self(QObject* parent)
{
    if ( !s_self ) {
        s_self = new Backend(parent);
        s_self->initialize();
    }
    return s_self;
}

Backend::Backend( QObject* parent): QObject(parent)
{
    mKblogger = qobject_cast<Application*> (parent);
    //Q_ASSERT(mKblogger);
}

void Backend::initialize()
{
    kDebug();

    populateBlogsList();

    //ItemsManager
    mItemsManager = ItemsManager::self(this);
    mJobsQueue = new BackendJobsQueue(this);

    connect(mJobsQueue, SIGNAL(jobDone()),
            this, SLOT(slotJobDone()));
    connect(mJobsQueue, SIGNAL(jobsDone()),
            this, SLOT( slotJobsDone() ));
}

Backend::~Backend()
{
    kDebug();
    s_self = 0;
}

void Backend::runQueuedJobs()
{
    kDebug();
    mJobsQueue->run();
}

KBloggerBlog Backend::getKbloggerBlog(const QString& blogname)
{
    kDebug();
    return blogsMap[blogname];
}

void Backend::listPosts( const QString& blogname, int downloadCount, QWidget *caller)
{
    kDebug() << blogname;
    if (!caller) caller = qobject_cast<QWidget*>(mKblogger);
    QVariant n(downloadCount);
    KBloggerBlog blog = blogsMap[blogname];
    kDebug() << "blogname of blog is" << blog.blogname() << endl;

    BackendJobs *mJob = new BackendJobs(blog,
                                        BackendJobs::LIST_RECENT_POSTS,
                                        n, caller,
                                        this);
    if ( mJobsQueue->addJob(mJob) ) {
        mJobsQueue->run();
    }
}

void Backend::listCategories(const QString& blogname, QWidget *caller)
{
    kDebug();
    KBloggerBlog blog = blogsMap[blogname];
    if (!caller)
        caller = qobject_cast<QWidget*>(mKblogger);
    BackendJobs *mJob = new BackendJobs(blog,
                                        BackendJobs::FETCH_CATEGORIES,
                                        0, caller,
                                        this);
    if ( mJobsQueue->addJob(mJob) ) {
        mJobsQueue->run();
    }
}

void Backend::sendPost( KBloggerPost* kbPost, QWidget *caller )
{
    kDebug();
    Q_ASSERT(kbPost);

    if (!caller) caller = qobject_cast<QWidget*>(mKblogger);

    if ( kbPost->status() != KBlog::BlogPost::New ) {
        kError() << "This post is already uploaded, is this a bug?" << endl;
        return;
    }

    kDebug() << "\tPost Title:" << kbPost->title() << endl;
    kDebug() << "\tContent:" << kbPost->content() << endl;
    kDebug() << "\tCategories:" << kbPost->categories().join(", ") << endl;
    kDebug() << "\tDateTime:" <<  kbPost->creationDateTime().dateTime().toString() << endl;

    //Test: The post is sent only if all media in it are already uploaded.
    /*    if ( kbPost->hasPendingMedia() ) {
            KMessageBox::sorry( 0 ,
                                i18n("This post cannot be uploaded. Upload its media first"));
            return;
        }
    */

    /*    KBlog::BlogPost *post = static_cast<KBlog::BlogPost*>(kbPost);
        if (!post) {
            kError() << "post is NULL" << endl;
            return;
        }*/

    QVariant n;
    n.setValue(kbPost);

    KBloggerBlog blog = blogsMap[kbPost->getBlogName()];
    BackendJobs *mJob;
    if ( QString(kbPost->postId()).isEmpty() ) {
        mJob = new BackendJobs(blog,
                               BackendJobs::CREATE_POST,
                               n, caller,
                               this);
    } else {
        mJob = new BackendJobs(blog,
                               BackendJobs::MODIFY_POST,
                               n, caller,
                               this);
    }
    if ( mJobsQueue->addJob(mJob) )
        emit statusBarMessage( i18n("New post upload enqueued in the job list.") );
    else
        emit statusBarMessage( i18n("It is Impossible to enqueue this new post.") );
}

void Backend::fetchPost(KBloggerPost* kbPost, QWidget *caller)
{
    kDebug();
    Q_ASSERT(kbPost);
    if (!caller) caller = qobject_cast<QWidget*>(mKblogger);
    //ID TEST
    QString id = kbPost->postId();
    if (id.isEmpty()) return;

    QVariant n;
    n.setValue(kbPost);
    KBloggerBlog blog = blogsMap[kbPost->getBlogName()];
    BackendJobs *mJob = new BackendJobs(blog,
                                        BackendJobs::FETCH_POST,
                                        n, caller,
                                        this);
    mJobsQueue->addJob(mJob);
}

void Backend::removePost( KBloggerPost *kbPost, QWidget *caller )
{
    kDebug();
    Q_ASSERT(kbPost);
    if (!caller) caller = qobject_cast<QWidget*>(mKblogger);
    //ID TEST
    QString id = kbPost->postId();
    if (id.isEmpty()) return;

    QVariant n;
    n.setValue(kbPost);

    KBloggerBlog blog = blogsMap[kbPost->getBlogName()];
    BackendJobs *mJob = new BackendJobs(blog,
                                        BackendJobs::REMOVE_POST,
                                        n, caller,
                                        this);
    if ( mJobsQueue->addJob(mJob) ) {
        mJobsQueue->run();
    }
}

void Backend::sendMedia( KBloggerMedia *kbMedia, QWidget *caller )
{
    kDebug();
    if (!kbMedia) return;
    if (!caller) caller = qobject_cast<QWidget*>(mKblogger);
    KBlog::BlogMedia *media = static_cast<KBlog::BlogMedia*>(kbMedia);
    if (!media) {
        kError() << "media is NULL" << endl;
        return;
    }
    QVariant n;
    n.setValue(media);
    KBloggerBlog blog = blogsMap[ kbMedia->getBlogname() ];
    BackendJobs *mJob = new BackendJobs(  blog,
                                          BackendJobs::CREATE_MEDIA,
                                          n, caller,
                                          this);
    if ( mJobsQueue->addJob(mJob) )
        emit statusBarMessage( i18n("New Media upload enqueued in the job list.") );
    else
        emit statusBarMessage( i18n("It is impossible to enqueue this new media.") );
}

/*
void Backend::listBlogs() //void blogInfoRetrieved( const QString &id, const QString &name );
{
        connect ( mBackend, SIGNAL ( blogInfoRetrieved( const QString&, const QString& )),
                  this, SLOT ( blogInfoRetrievedSlot( const QString&, const QString& ) ));
        mBackend->listBlogs();
}
*/

void Backend::sync()
{
    kDebug();
    mJobsQueue->clear();
    //Upload Media
    mItemsManager->enqueueMediaToUpload();
    //Upload Posts
    mItemsManager->enqueuePostsToUpload();
    //TODO Update sentList?

    //run the enquenqued jobs
    mJobsQueue->run();
}


void Backend::populateBlogsList()
{
    kDebug();
    //Clear BlogsMap
    blogsMap.clear();

    //TODO move
    profileConfig *mProfileconfig = profileConfig::prefs("THIS_ACCOUNT_DOESNT_EXIST");
    QStringList groupList = mProfileconfig->config()->groupList();
    kDebug () << "ProfileConfigDialog::populateAccountComboBox()" <<  groupList << endl;
    groupList = groupList.filter( QRegExp("^Account_"));
    //TODO end move

    QStringListIterator groupListIterator(groupList);

    while ( groupListIterator.hasNext() ) {
        const QString accountName = groupListIterator.next(); //Contains "Account_yourAccountName"

        QString blogname(accountName);
        blogname.remove(0, 8); // Contains "yourAccountName"

        KBloggerBlog blog;

        blog.setBlogname( blogname );
        blog.setUsername( profileConfig::prefs(accountName)->user() );
//         blog.setFullName( profileConfig::prefs(accountName)->user() ); // FIXME edit the config dialog
        blog.setUrl( profileConfig::prefs(accountName)->url());
        blog.setApi( api_type( profileConfig::prefs(accountName)->type() ) );
        blog.setBlogId( profileConfig::prefs(accountName)->blogId() );

        //TEST
        kDebug() << "bloginfo." << endl
        << "blogname=" << blog.blogname() << endl
        << "Username=" << blog.username() << endl
        << "url=" << blog.url() << endl
        << "api=" << blog.api() << endl;
        kDebug() << "password=" << blog.password() << endl;

        blogsMap.insert(blogname, blog);
    }
}

int Backend::jobsQueued()
{
    kDebug();
    return mJobsQueue->queueSize();
}

void Backend::showStatusBarMessage(const QString& message)
{
    kDebug();
    emit statusBarMessage(message);
}

QList<KBloggerBlog> Backend::getBlogList()
{
    kDebug();
    return blogsMap.values();
}

void Backend::stopQueue()
{
    kDebug();
    if ( !mJobsQueue->stop() ) {
        return;
    }
    mJobsQueue->clear();
    statusBarMessage( i18n("All jobs have been removed from the queue.") );
}

/*
void Backend::killCurrentJob(){
    kDebug();
    if ( mJobsQueue->killCurrentJob() )
      statusBarMessage( i18n("Current Job killed."));
    else
      statusBarMessage( i18n("No job is running."));
}
*/

void Backend::blogInfoRetrievedSlot(const QString &id, const QString &name)
{
    //TODO
    Q_UNUSED(id);
    Q_UNUSED(name);

    kDebug();
    KMessageBox::sorry( 0 ,
                        "Backend::blogInfoRetrievedSlot doesn't work yet",
                        i18n("KBlogger is still working in progress..."));
}

void Backend::categoryRetrieved ( const QList<QMap<QString, QString> >& categories )
{
    kDebug();
    emit categoryInfoRetrieved ( categories );
}

void Backend::slotJobDone()
{
    WaitWidget::self()->jobDone();
    //emit statusBarMessage(i18n("Job Done."));
}

void Backend::slotJobsDone()
{
    emit statusBarMessage(i18n("All Jobs Done."));
    emit jobsFinished();
}

void Backend::blogInfoRetrieved( const QString &id, const QString &name )
{
    kDebug();
    emit blogInfoRetrieved( id, name );
}

void Backend::slotError( const KBlog::Blog::ErrorType type, const QString &errorMessage )
{
//  kDebug() << "Backend::error" <<endl;
    //Disconnect all signals fromBackend.
//     mBackend->disconnect();
    //Reconnect the slot that are connected in the constructor.
//     connect(mBackend, SIGNAL( error( KBlog::Blog::ErrorType, const QString& ) ),
//             this, SLOT( error( KBlog::Blog::ErrorType, const QString& ) ) );
//     connect(mBackend, SIGNAL( createdPost( const QString &) ),
//             this, SLOT( createdPost( const QString & ) ) );
    KLocalizedString errType;

    switch (type) {
    case KBlog::Blog::XmlRpc:
        errType = ki18n("XML RPC Error: %1");
        break;
    case KBlog::Blog::Atom:
        errType = ki18n("Atom API Error: %1");
        break;
    case KBlog::Blog::ParsingError:
        errType = ki18n("KBlog Parsing Error: %1");
        break;
    case KBlog::Blog::AuthenticationError:
        errType = ki18n("Authentication Error: %1");
        break;
    case KBlog::Blog::NotSupported:
        errType = ki18n("Not Supported Error: %1");
        break;
    default:
        errType = ki18n("Unknown Error type: %1");
    };
    QString errorString = errType.subs(errorMessage).toString();
    kDebug() << "Backend::error " << errorString << endl;
    KMessageBox::error( 0 , errorString, i18n("KBlogger backend error"));
}

} //namespace

#include "backend.moc"

