/***************************************************************************

   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>
   Copyright (C) 2008 Christian Weilbach <christian_weilbach@web.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "media/importmediawidget.h"

#include <QTreeWidgetItem>
#include <QTreeWidget>
#include <QFileInfo>

#include <kdebug.h>
#include <kicon.h>
#include <kio/job.h>
#include <kmessagebox.h>
#include <kurl.h>

#include "media/media.h"
#include "itemsmanager.h"
#include "backend/backend.h"
#include "backend/blogserver.h"

namespace KBlogger
{

ImportMediaWidget::ImportMediaWidget(QWidget *parent): QWidget(parent),
        blackListedUrls(0)
{
    kDebug();
    setupUi(this);
    setAttribute(Qt::WA_DeleteOnClose);
    mParent = qobject_cast<KDialog*>(parent);
    mediaTreeWidget->setColumnHidden(url_coloumn,true);
}

ImportMediaWidget::~ImportMediaWidget()
{
    kDebug();
    // clean things up
    QList<QTreeWidgetItem*>::ConstIterator it = treeItemList.begin();
    QList<QTreeWidgetItem*>::ConstIterator end = treeItemList.end();
    for( ; it != end; it++ ){
        mediaTreeWidget->removeItemWidget( *it, 0 );
        delete( *it );
    }
}

void ImportMediaWidget::clearList(){
    kDebug();
    mediaTreeWidget->clear();
    treeItemList.clear();
}

bool ImportMediaWidget::addUrl(const QString& imageUrl, const QString& blogname, const QString& postId)
{
    kDebug() << "ImageUrl" << imageUrl << endl
             << "Blogname" << blogname << endl
             << "Post ID" << postId << endl;

    Q_ASSERT( !imageUrl.isEmpty() );
    Q_ASSERT( !blogname.isEmpty() );
    if ( treeItemList.count() ){
        kWarning() << "treeItemList is empty";
    }

    QStringList ignoredUrls;
    ignoredUrls = ItemsManager::self()->getIgnoredMediaUrls(blogname);

    if ( ignoredUrls.contains(imageUrl, Qt::CaseInsensitive) ) {
        blackListedUrls++;
        blacklistLabel->setText(i18n("There are %1 imageUrls blacklisted not showed here.", blackListedUrls));
        return false;
    }

    QString thumbRegExpString;
    BlogServer blogServ= Backend::self()->blogServer( blogname );
    QRegExp thumbnailUrlRx = blogServ.getBlogServerValue("thumbnailRegExp").toRegExp();
    kDebug() << "thumbnailUrlRx pattern:" << thumbnailUrlRx.pattern();
    QTreeWidgetItem *item;
    
    /*NOTE  usually the code that contains a thumbnail looks like this:
     * <a href="myimage.jpg" >
     *   <img src="myimage.thumb.jpg">   <--thumbnailUrlRx will match the string between the '"'.
     *</a>
     * So kblogger will process the thumbnail before the image.
     */
    //FIXME does this only simple condition work reliably?

    if ( !thumbnailUrlRx.pattern().isEmpty() &&
	  imageUrl.contains( thumbnailUrlRx ) 
       ) {
        item = new QTreeWidgetItem( treeItemList.last() ); //Thumbnail
    }else{
        item = new QTreeWidgetItem( mediaTreeWidget ); //No Thumbnail
    }

    KUrl url(imageUrl);
    QString imageName=url.fileName();
    item->setText(fileName_coloumn, imageName );
    item->setIcon(fileName_coloumn, KIcon("document-import") );
    item->setText(url_coloumn, imageUrl );
    item->setText(blogName_coloumn, blogname );
    item->setText(postId_coloumn, postId );
    item->setText(status_coloumn, i18n("Not downloaded yet"));
    mediaTreeWidget->insertTopLevelItem( 0, item );
    treeItemList += item;
    return true;
}

void ImportMediaWidget::accept()
{
    kDebug();
    //Download images
    QList<QTreeWidgetItem*>::ConstIterator it = treeItemList.begin();
    QList<QTreeWidgetItem*>::ConstIterator end = treeItemList.end();
    for( ; it != end; it++ ){
        createMediaFromUrl( *it );
    }
}

void ImportMediaWidget::createMediaFromUrl(QTreeWidgetItem* item)
{
    KUrl url = item->text(url_coloumn);
    kDebug() << url;
    Q_ASSERT( url.isValid() );

    KIO::StoredTransferJob* job = KIO::storedGet( url, KIO::NoReload, KIO::HideProgressInfo );

    jobsMap[ job ] = item;

    connect( job, SIGNAL( result( KJob* ) ),
             this, SLOT( handleFileJobResult( KJob* ) ) );
}

void ImportMediaWidget::handleFileJobResult(KJob* job)
{
    kDebug();
    if( !job ){
        // FIXME this stops the widget from closing at the moment
        // as we cannot remove the job from jobsMap to check if we 
        // are finished
        kError() << "The job died.";
        return;
    }

    QString mediaMimetype = qobject_cast<KIO::StoredTransferJob*>(job)->mimetype();
    QByteArray mediaData = qobject_cast<KIO::StoredTransferJob*>(job)->data();

    QTreeWidgetItem* item = jobsMap[ job ];
    Q_ASSERT( item );

    // TODO check the cases for this event
    if ( mediaData.isEmpty() ) {
        item->setIcon(fileName_coloumn, KIcon("dialog-cancel") );
        item->setText(status_coloumn, i18n("Read-Error. The media is empty.") );
        kError() << "Read-Error: The media is empty.";
        jobsMap.remove( job );
        delete job;
        jobsFinished();
        return;
    }

    if ( job->error() ) {
        item->setIcon(fileName_coloumn, KIcon("dialog-cancel") );
        item->setText(status_coloumn, i18n("Read-Error.") );
        kError() << "Read-Error: " << job->errorString();
        jobsMap.remove( job );
        delete job;
        jobsFinished();
        return;
    }
    jobsMap.remove( job );
    delete job;

    //Settings Vars
    KUrl mediaUrl( item->text(url_coloumn) );
    const QString filename = QFileInfo(mediaUrl.path()).fileName();
    const QString blogname = item->text(blogName_coloumn);

    //Test: media Size < 1Mb TODO make this depending on blog implementation (feature map)
    if ( (mediaData.count() / 1024) > 1024 ) { //Max 1 Mb
        item->setIcon(fileName_coloumn, KIcon("dialog-cancel") );
        item->setText(status_coloumn, i18n("Size > 1Mb: Url Blacklisted"));
        ItemsManager::self()->addToIgnoredMediaUrls( mediaUrl, blogname );
        jobsFinished();
        return;
    }

    if ( mediaMimetype == "text/html" ) {
        item->setIcon(fileName_coloumn, KIcon("dialog-cancel") );
        item->setText(status_coloumn, i18n("Error 404: Url blacklisted"));
        ItemsManager::self()->addToIgnoredMediaUrls( mediaUrl, blogname );
        jobsFinished();
        return;
    }

    Media *media;
    media = new Media( mediaData,
                       filename,
                       mediaMimetype,
                       blogname
                     );

    if ( !mediaUrl.isLocalFile() )
        media->setUrl(mediaUrl);

    if ( media->isCached() ) {
        item->setIcon(fileName_coloumn, KIcon("dialog-ok-apply") );
        item->setText(status_coloumn, QString("%1 Kb").arg(mediaData.count() / 1024));

        //Thumbnail managing
	mediaMap[item]=media;

    } else {
        item->setIcon(fileName_coloumn, KIcon("dialog-cancel") );
        item->setText(status_coloumn, i18n("CacheFile writing Error"));
    }
    
    jobsFinished();
}

void ImportMediaWidget::jobsFinished()
{
    kDebug();
    if( jobsMap.isEmpty() ) {
	processCreatedMedia();
        Q_ASSERT( mParent );
        mParent->close();
    }/*else{
        KMessageBox::error( this, i18n("KBlogger is unable to download all media.") );
    }*/
}

void ImportMediaWidget::processCreatedMedia(){
    kDebug();
    
    for (int i = 0; i < treeItemList.size(); ++i) {
//      for (int i = treeItemList.size()-1; i >0; --i ) {
        QTreeWidgetItem* currentItem=treeItemList[i];
	Q_ASSERT(currentItem);
	
	QTreeWidgetItem* parentItem=currentItem->parent();
	Media* currentMedia=mediaMap[ currentItem ];
	if (!currentMedia){
	    //Media is null when a job exits with error.
	    kWarning() << "Some errors occours: currentMedia is null. Continue..";
	    continue;
	}
	Media* previouslyMedia=0;
	
	if ( i > 0 ){
	    previouslyMedia=mediaMap[ treeItemList[i-1] ];
	}
	//If currentItem has a parent (a QTreeWidgetItem),it's a thumbnail
	if ( parentItem && previouslyMedia ){
	    currentMedia->setFullSizeImage( previouslyMedia );
	    previouslyMedia->setThumbnail( currentMedia );
	}
	//
	ItemsManager::self()->addMedia( currentMedia );
    }
}

} //Namespace

#include "importmediawidget.moc"
