/***************************************************************************

   Copyright (C) 2005-2007 by Christian Weilbach <christian_weilbach@web.de>
   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "profiles/profileconfigdialog.h"
#include "profileconfig.h"

#include <QTimer>

#include <kdebug.h>
#include <kwallet.h>
#include <kconfigdialog.h>
#include <kio/job.h>
#include <kmessagebox.h>
#include <kprogressdialog.h>

#include <kblog/gdata.h>
#include <kblog/blogger1.h>
#include <kblog/metaweblog.h>
#include <kblog/movabletype.h>
#include <kblog/wordpressbuggy.h>

#include "backend/bloglist.h"

const int TIMEOUT = 45000;

namespace KBlogger
{

ProfileConfigDialog::ProfileConfigDialog(KConfigDialog* parent): mWallet(0), mBlog(0),
                               mFetchIdTimer(0),mFetchAPITimer(0),mLastShot(false),
			       mAutoConfProgressDialog(0)
{
    kDebug();

    setupUi(this);

    //Open Wallet
    mWallet = KWallet::Wallet::openWallet( "kdewallet", this->winId() );
    Q_ASSERT( mWallet );
    mWallet->createFolder( "kblogger" );
    mWallet->setFolder( "kblogger" );

    // initialize the combo box for the blog types
    // new code to use the BlogList
    // NOTE: this means reording the BlogList int index breaks old configurations!
    mBlogList = BlogList();
    for( int i=0; i < mBlogList.count(); i++ ){
        // check the url against the url regexp
        kcfg_Type->addItem( mBlogList[i]["sitename"].toString() );
    }

    // manage the automatic ID fetching
    connect(kcfg_User, SIGNAL( textChanged(const QString &) ), this, SLOT( enableFetchBlogId() ) );
    connect(kcfg_Url, SIGNAL( textChanged(const QString &) ), this, SLOT( enableFetchBlogId() ) );
    connect(passwordLineEdit, SIGNAL( textChanged(const QString &) ), this, SLOT( enableFetchBlogId() ) );
    connect(kcfg_Type, SIGNAL( activated( int ) ), this, SLOT( enableFetchBlogId() ) );

    
    connect(parent, SIGNAL(okClicked()),
            this, SLOT(savePasswordToWallet()));
    connect(parent, SIGNAL(applyClicked()),
            this, SLOT(savePasswordToWallet()));

    connect(mFetchIDButton, SIGNAL( clicked() ), this, SLOT( fetchBlogId() ) );
    connect(mAutoconfigureButton, SIGNAL( clicked() ), this, SLOT( autoconfigure() ) );

    enableFetchBlogId();
}

ProfileConfigDialog::~ProfileConfigDialog()
{
    kDebug();
    delete mWallet;
    delete mBlog;
}

void ProfileConfigDialog::readPasswordFromWallet()
{
    kDebug() << kcfg_Url->text();
    if ( kcfg_Url->text().isEmpty() ){
        kError() << "ERROR: The password of a empty url cannot be retrieved from the wallet";
    }

    QString buffer;
    mWallet->readPassword( kcfg_Url->text() + '_' + kcfg_User->text(), buffer );
    kDebug() << "Password from Wallet: " << buffer;
    passwordLineEdit->setText( buffer );
}

void ProfileConfigDialog::savePasswordToWallet()
{
    kDebug();
    if ( !mWallet ) return;
    mWallet->writePassword( kcfg_Url->text() + '_' + kcfg_User->text(), passwordLineEdit->text() );
}

void ProfileConfigDialog::enableFetchBlogId()
{
    kDebug();
    if ( kcfg_User->text().isEmpty() ||
            kcfg_Url->text().isEmpty() ||
            passwordLineEdit->text().isEmpty() ) {
        mFetchIDButton->setEnabled( false );
        mAutoconfigureButton->setEnabled( false );
    }
    else {
        mFetchIDButton->setEnabled( true );
        mAutoconfigureButton->setEnabled( true );
    }
}

void ProfileConfigDialog::autoconfigure()
{
    if( kcfg_User->text().isEmpty() ) {
         kDebug()<< "fill in the username first";
         kcfg_User->setFocus();
         KMessageBox::sorry( this,
              i18n("You need to set the username, password and url of your blog."));
         return;
    }

    if( passwordLineEdit->text().isEmpty() ) {
         kDebug()<< "fill in the password first";
         passwordLineEdit->setFocus();
         KMessageBox::sorry( this,
              i18n("You need to set the username, password and url of your blog."));
         return;
    }

    if( kcfg_Url->text().isEmpty() ) {
         kDebug()<< "fill in the url first";
         kcfg_Url->setFocus();
         KMessageBox::sorry( this,
              i18n("You need to set the username, password and url of your blog."));
         return;
    }

    mAutoconfigureButton->setEnabled( false );
    mFetchIDButton->setEnabled( false );

    // new code to use the BlogList
    for( int i=0; i<mBlogList.count(); i++ ){
        // check the url against the url regexp
        if( !mBlogList[i]["urlRegExp"].toRegExp().isEmpty() && // might be the case for weak html regexp guesses
             mBlogList[i]["urlRegExp"].toRegExp().indexIn( kcfg_Url->text() ) != -1 ){
            kDebug() << "matched :" << mBlogList[i]["sitename"].toString();
            // select the combobox string
            kcfg_Type->setCurrentIndex( kcfg_Type->findText( mBlogList[i]["sitename"].toString() )
                               );
            // adjust the url string 
            kcfg_Url->setText( kcfg_Url->text().replace( mBlogList[i]["urlTransformRegExp"].toRegExp(),
			       mBlogList[i]["urlReplacementString"].toString() )
                            );
	    
            showMessageDialog(i);
	    
            fetchBlogId();
            return;
        }
    }

    showProgressDialog();
    
    // try to get the api from the html otherwise
    httpGetJob = KIO::storedGet( kcfg_Url->text(), KIO::NoReload, KIO::HideProgressInfo );

    connect( httpGetJob, SIGNAL( result( KJob* ) ),
             this, SLOT( gotHtml( KJob* ) ) );
}

void ProfileConfigDialog::fetchBlogId()
{
    kDebug();
    // TODO add fetchBlogId( QString api, QString username, QString password ) or sth. to the backend
    // to avoid parsing the KBlog stuff here
    mFetchIdTimer = new QTimer( this );
    connect( mFetchIdTimer, SIGNAL( timeout() ),
             this, SLOT( handleFetchIDTimeout() ) );
    mFetchIdTimer->start( TIMEOUT );
    delete mBlog;

    // new code to use the BlogList
    for( int i=0; i < mBlogList.count(); i++ ){
        if( mBlogList[i]["sitename"].toString()==kcfg_Type->itemText( kcfg_Type->currentIndex() ) ){
           kDebug() << "matched :" << mBlogList[i]["sitename"].toString();
           kDebug() << "api is:" << mBlogList[i]["api"].toString();

           if( mBlogList[i]["api"].toString()=="Blogger1"
                || mBlogList[i]["api"].toString()=="MetaWeblog"
                || mBlogList[i]["api"].toString()=="MovableType"
                || mBlogList[i]["api"].toString()=="WordpressBuggy" ){  // They all use in fact Blogger1
                mBlog = new KBlog::Blogger1( kcfg_Url->text() );
                qobject_cast<KBlog::Blogger1*>(mBlog)->setUsername( kcfg_User->text() );
                qobject_cast<KBlog::Blogger1*>(mBlog)->setPassword( passwordLineEdit->text() );
                connect( mBlog, SIGNAL(listedBlogs( const QList<QMap<QString, QString> >&)),
                     this, SLOT(fetchedBlogId( const QList<QMap<QString, QString> >&)));
                connect( mBlog, SIGNAL(error( KBlog::Blog::ErrorType, const QString& ) ),
                     this, SLOT( handleFetchBlogIdError( KBlog::Blog::ErrorType, const QString& ) ) );
                qobject_cast<KBlog::Blogger1*>(mBlog)->listBlogs();
            }

            if( mBlogList[i]["api"].toString()=="GData" ){ // GData
                mBlog = new KBlog::GData( kcfg_Url->text() );
                qobject_cast<KBlog::GData*>(mBlog)->setUsername( kcfg_User->text() );
                qobject_cast<KBlog::GData*>(mBlog)->setPassword( passwordLineEdit->text() );
                connect( qobject_cast<KBlog::GData*>(mBlog), SIGNAL(fetchedProfileId( const QString& ) ),
                     this, SLOT( fetchedProfileId( const QString& ) ) );
                qobject_cast<KBlog::GData*>(mBlog)->fetchProfileId();
// TODO fix kblog so we don't need to fetch the profile id first, but use "default" instead
//                 connect( mBlog, SIGNAL(listedBlogs( const QList<QMap<QString, QString> >&)),
//                      this, SLOT(fetchedBlogId( const QList<QMap<QString, QString> >&)));
//                 connect( mBlog, SIGNAL(error( KBlog::Blog::ErrorType, const QString& ) ),
//                      this, SLOT( handleFetchBlogIdError( KBlog::Blog::ErrorType, const QString& ) ) );
//                 qobject_cast<KBlog::GData*>(mBlog)->listBlogs();
            }

            kcfg_BlogId->setEnabled( false );
            mFetchIDButton->setEnabled( false );
            return;
        }
    }

}

void ProfileConfigDialog::fetchedBlogId( const QList<QMap<QString, QString> >& list )
{
    kDebug();
    // TODO readd when we have done multi id handling correctly
//     QList<QMap<QString,QString> >::ConstIterator it = list.begin();
//     QList<QMap<QString,QString> >::ConstIterator end = list.end();
//     for(; it != end; ++it ) {
//         kcfg_BlogId->setText( ( *it ).values().first() );
//     }
    delete mFetchIdTimer;
    mFetchIdTimer = 0;
    kcfg_BlogId->setText( list.first().values().first() );
    kcfg_BlogId->setEnabled( true );
    mFetchIDButton->setEnabled( true );
    mAutoconfigureButton->setEnabled( true );
}

void ProfileConfigDialog::handleFetchBlogIdError( KBlog::Blog::ErrorType, const QString& error )
{
    kDebug();
    
    delete mFetchIdTimer;
    mFetchIdTimer = 0;
    
    KMessageBox::error( this, error );
    
    kcfg_BlogId->setText( QString() );
    kcfg_BlogId->setEnabled( true );
    mFetchIDButton->setEnabled( true );
    mAutoconfigureButton->setEnabled( true );
}

void ProfileConfigDialog::handleFetchIDTimeout()
{
    kDebug();
    
    delete mFetchIdTimer;
    mFetchIdTimer = 0;
    
    KMessageBox::error( this,
                        i18n( "Fetching the blog's id timed out. Are you online?" ));
    kcfg_BlogId->setText( QString() );
    kcfg_BlogId->setEnabled( true );
    mFetchIDButton->setEnabled( true );
    mAutoconfigureButton->setEnabled( true );
}

void ProfileConfigDialog::handleFetchAPITimeout()
{
    kDebug();
    Q_ASSERT( mAutoConfProgressDialog );
    int progressValue= mAutoConfProgressDialog->progressBar()->value();
    if ( progressValue < TIMEOUT ){
      mAutoConfProgressDialog->progressBar()->setValue( progressValue + 10 );
      return;
    }
		    
    hideProgressDialog();
    
    KMessageBox::error( this,
                        i18n( "Autoconfiguration timed out. Are you online?" ));
    
    kcfg_BlogId->setEnabled( true );
    mFetchIDButton->setEnabled( true );
    mAutoconfigureButton->setEnabled( true );
}

void ProfileConfigDialog::fetchedProfileId( const QString &id )
{
    kDebug();
    Q_UNUSED(id);
    mFetchIdTimer->start( TIMEOUT ); // restart the timer for the second id fetching
    connect( qobject_cast<KBlog::GData*>(mBlog), SIGNAL(listedBlogs( const QList<QMap<QString, QString> >&)),
             this, SLOT(fetchedBlogId( const QList<QMap<QString, QString> >&)));
    connect( qobject_cast<KBlog::GData*>(mBlog), SIGNAL(error( KBlog::Blog::ErrorType, const QString& ) ),
             this, SLOT( handleFetchBlogIdError( KBlog::Blog::ErrorType, const QString& ) ) );
    qobject_cast<KBlog::GData*>(mBlog)->listBlogs();
}

void ProfileConfigDialog::gotHtml(KJob *job)
{
    kDebug();
    if(!job) return;
    
    //Delete Uneeded objects
    hideProgressDialog();
    
    //There is a job error?
    if ( job->error() ) {
        KMessageBox::error( this,
                            job->errorString());
	
        mAutoconfigureButton->setEnabled( true );
        return;
    }
    
    QString httpData( qobject_cast<KIO::StoredTransferJob*>(job)->data() );
//     delete job; // TODO readd when autodeletion is off or remove

    // new code to use the BlogList
    for( int i=0; i < mBlogList.count(); i++ ){
        // check the url against the html regexp
        if( mBlogList[i]["htmlRegExp"].toRegExp().indexIn( httpData ) != -1 ){
            kDebug() << "matched :" << mBlogList[i]["sitename"].toString();
            // select the combobox string
            kcfg_Type->setCurrentIndex(
                               kcfg_Type->findText( mBlogList[i]["sitename"].toString() )
                               );
            // adjust the url string 
            kcfg_Url->setText(
                                kcfg_Url->text().replace( mBlogList[i]["urlTransformRegExp"].toRegExp(),
                                                                        mBlogList[i]["urlReplacementString"].toString() )
                            );

	    showMessageDialog(i);
	    
            // HACK if we have recalled ourselves and we are here (have hit the generic xml-rpc) than we need to update the url
            if( mLastShot ){
                kcfg_Url->setText( kcfg_Url->text() + "/xmlrpc.php" );
                mLastShot = false;
            } // we won't match the next one any way so no need to check here

            // try to guess the url
            if( !mBlogList[i]["blogIdRegExp"].toRegExp().isEmpty() 
                 && mBlogList[i]["blogIdRexExp"].toRegExp().indexIn( kcfg_Url->text() )!=-1 ){
                kDebug() << "matched blogId: " << mBlogList[i]["blogIdRegExp"].toRegExp().cap(0);
                kcfg_BlogId->setText( mBlogList[i]["blogIdRegExp"].toRegExp().cap(0) );
                mAutoconfigureButton->setEnabled( true );
            }
            else fetchBlogId();
            return;
        }
    }

    if( !mLastShot ){
        // HACK do a last attempt and recall with +/xmlrpc.php
        showProgressDialog();
        httpGetJob = KIO::storedGet( kcfg_Url->text() + "/xmlrpc.php", KIO::NoReload, KIO::HideProgressInfo );

        mLastShot = true;

        connect( httpGetJob, SIGNAL( result( KJob* ) ),
                 this, SLOT( gotHtml( KJob* ) ) );
    }
    else {
        mAutoconfigureButton->setEnabled( true );
        KMessageBox::error( this,
                            i18n("Impossible to get the API. Check your settings, especially the url."));
    }
}

void ProfileConfigDialog::showProgressDialog(){
    kDebug();
    if ( !mAutoConfProgressDialog ){
      mAutoConfProgressDialog= new KProgressDialog(this, "Weblog Autoconfiguration",
	 					   "Please wait while kblogger is tryng to collect your weblog infos");
    }
    
    mAutoConfProgressDialog->setAllowCancel(false);
    mAutoConfProgressDialog->progressBar()->setMaximum( TIMEOUT );
    mAutoConfProgressDialog->progressBar()->setValue( 0 );
    
    mFetchAPITimer = new QTimer( this );
    mFetchAPITimer->setSingleShot( true );
    connect( mFetchAPITimer, SIGNAL( timeout() ),
                    this, SLOT( handleFetchAPITimeout() ) );
    mFetchAPITimer->start( 10 );
}

void ProfileConfigDialog::hideProgressDialog(){
    kDebug();
    delete mFetchAPITimer;
    mFetchAPITimer = 0;
    delete mAutoConfProgressDialog;
    mAutoConfProgressDialog=0;
}

void ProfileConfigDialog::showMessageDialog(int api){
    kDebug();
    
    if ( api == 5 ){ // 5 is bloggerApi in blogList.cpp
        //Autoconfiguration failed!
	if ( mLastShot ) //Show only the last sorry message
	    KMessageBox::sorry( this,  mBlogList[api]["configInfo"].toString() );
        else{
	    KMessageBox::information( this,  mBlogList[api]["configInfo"].toString() );
        }
    }
}

} //namespace

#include "profileconfigdialog.moc"

