/***************************************************************************
                         kcmpureftpdscript.h - pure-ftpd script generator
                         -------------------
    begin                : Fri Apr 13 2001
    copyright            : (C) 2001-2003 by Claudiu Costin
    email                : claudiuc@kde.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef __KCMPUREFTPDSCRIPT_H__
#define __KCMPUREFTPDSCRIPT_H__

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qmap.h>
#include <qobject.h>

#include <kapp.h>
#include <kcmodule.h>
#include <kconfig.h>
#include <kaboutdata.h>
#include <klistview.h>

#include "script.h"
#include "scripttemplate.h"
#include "kcmpureftpdscriptui.h"


typedef QMap<KListViewItem*,QString> ListMap;


/**
 * It generate startup control scripts for pure-ftpd based on
 * XML templates and user choosed options. For this are used
 * script templates which contain the startup script filled with
 * parsable options like \@PUREFTPD\@, \@OPTIONS\@ etc. There are other
 * attributes needed: line terminator, option seperator, sugested name
 * for output script etc. The first and second enumerated attributes are
 * used to organize generated pure-ftpd command line in a more pleasant
 * and readable form. Script template are organized in a tree-like view
 * where roots are script template type: standalone or superserver.
 *
 * @short Generate startup scripts for pure-ftpd
 * @author Claudiu Costin <claudiuc@kde.org>
 */
class KPureftpdScript : public KCModule {
    Q_OBJECT

public:
    /**
     * Construct KcmPureftpdScript GUI, connect slots and load configuration
     * from configuration file.
     * @param parent is the widget parent
     * @param name of this widget (optional)
     */
    KPureftpdScript(QWidget *parent = 0L, const char *name = 0L);
    /**
      * Destroy KcmPureftpdScript instance
      */
    virtual ~KPureftpdScript();
    /**
      * Load configuration data and set the widgets states and values
      */
    void load();
    /**
      * Save configuration according to the widgets states and values
      */
    void save();
    /**
      * Present user the defaults widgets states and values for the widgets. Basically
      * it execute @ref load()
      */
    void defaults();
    /**
      * Return the button mask such that KControl or kcmshell will know what
      * system buttons to show when the module will be embeded.
      */
    int buttons();
    /**
      * Return rich text help shown in KControl "Help" tab
      */
    QString quickHelp() const;
    /**
      * Return information about KcmPureftpdScript developers, contributors
      * and license terms
      */
    virtual const KAboutData *aboutData() const;

protected slots:
    /**
      * emit the signal @ref changed() only when I not set the GUI elements data
      * from @ref setGui().
      */
    void slotConfigChanged();
    /**
      * Set the authentication editor to the enabled or disabled state
      * according to the radiobuttons state in "Authentication" button group.
      * This way the user can see whether he's using default authentication
      * provided by system or the custom selection made by him.
      *
      * @param index is number of selected radio button.
      */
    void slotAuthType(int index);
    /**
      * Set the "authFile" as disabled when the user choose "unix" or "pam" in
      * methods combobox. It also set the editor buttons (Add, Modify, Delete)
      * such that only relevant buttons will be enabled.
      *
      * @param index is the position of selected item in "authMethod" combobox
      */
    void slotAuthMethodChanged(int index);
    /**
      * Set the "authMethod" and "authFile" when current item in "authList" is
      * changed: user click on item or move highlight with cursor keys.
      * It also set the editor buttons (Add, Modify, Delete)
      * and Up and Down buttons such that only relevant buttons will be enabled.
      *
      * @param item is the selected element in the "authList" list view
      */
    void slotAuthListChanged(QListViewItem *item);
    /** Add new authentication in the authentication list bellow the current
      * element in the list.
      * If the method is "unix" or "pam", the "authFile" content is ignored.
      * It also set the editor buttons (Add, Modify, Delete)
      * and Up and Down buttons such that only relevant buttons will be enabled.
      */
    void slotAuthAdd();
    /** Modify the current selected authentication in the "authList" with the new values
      * from the "authMethod" and "authFile".
      * If the method is "unix" or "pam", the "authFile" content is ignored.
      */
    void slotAuthModify();
    /** Delete the current selected authentication in the "authList".
      * It also set the editor buttons (Add, Modify, Delete)
      * and Up and Down buttons such that only relevant buttons will be enabled.
      */
    void slotAuthDelete();
    /**
      * Move the selected up down in the
      * authentication methods list @var authList
      */
    void slotAuthMoveUp();
    /**
      * Move the selected item down in the
      * authentication methods list @var authList
      */
    void slotAuthMoveDown();
    /**
      * Add a new script based on the script template.
      * The KScriptAdd dialog is opened and then a new KListViewItem is
      * inserted in @var scriptList
      * as a child of coresponding script template. Buttons
      * are set then with @ref setScriptEditButtons() and item
      * is selected.
      */
    void slotScriptAdd();
    /**
      * Edit current selected script in the @var scriptList.
      * The KScriptEdit dialog is opened and then new new settins
      * are saved in the @var script. Item is then marked as
      * changed.
      */
    void slotScriptEdit();
    /**
      * Save current selected script data back to disk. This operation
      * generate or update the XML data file which have ".purescr" extension.
      * The changed icon is removed the point that script item is not
      * changed.
      */
    void slotScriptSave();
    /**
      * Load the old XML data from disk and the coresponding @var script
      * entry is overwrited. Buttons are set then with
      * @ref setScriptEditButtons() and item is marked as unchanged.
      */
    void slotScriptRevert();
    /**
      * Delete the selected item from @var scriptList and
      * asw well the XML data file from disk. Buttons are set then with
      * @ref setScriptEditButtons().
      */
    void slotScriptDelete();
    /**
      * Set the editor buttons (Add, Save, Revert, Delete etc.)
      * such that only relevant buttons will be enabled.
      *
      * @param item is the selected element in the "scriptList" list view
      */
    void slotScriptListChanged(QListViewItem *item);
    /**
      * Execute the @ref slotScriptAdd() or @ref slotScriptEdit() according
      * with the type o double-clicked item.
      *
      * @param item is the selected element in the "scriptList" list view
      */
    void slotScriptListDClicked(QListViewItem *item);
    /**
      * Open the contextual popup menu for @var scriptList KListView. It
      * build and display only the relevant actions for selected KListViewItem.
      *
      * @param item is the selected element in the "scriptList" list view
      * @param pos is mouse position where right-click occured in the KListView
      * @param column is column position in the KListView where mouse event occured
      */
    void slotScriptListRMB(QListViewItem *item, const QPoint &pos, int column);
    /**
      * Save the generated script output to file. This is the text in the
      * @var outputView.
      */
    void slotOutputSave();
    /**
      * Processed right before the tab will be shown. It's main purpose is to save
      * data from widgets (pureftpd configuration) to current script item data
      * structure and mark the item as changed.
      */
    void slotTabChanged(QWidget *widget);
    /**
      * This slot set the @var pref data object for the new script,
      * as transmited from KScriptAdd dialog.
      * It's called right before closing dialog.
      */
    void slotAddPref(ScriptPref data);
    /**
      * This slot set the @var pref data object for edited script, as
      * transmited from KScriptEdit dialog.
      * It's called right before closing dialog.
      */
    void slotEditPref(ScriptPref data);

private:

    void setGui(const GuiData &g);
    void getGui(GuiData &g);
    void setScriptChanged(bool b);
    void setScriptEditButtons();
    void setAuthEditButtons();
    void setAuthMoveButtons();
    void loadTemplates();
    void loadScripts();
    bool canWriteFile(const QString &f);
    QString parseOptions(ScriptItem &s);

    /**
      * This is hold the pointer to instantiated KcmPureftpdScript user interface.
      * Local UI interaction is defined here.
      */
    KPureftpdScriptUI *ui;
    KListViewItem *itemStandalone, *itemSuperserver, *itemOrphans;
    QString saveOutput, parsedScript;
    bool isSettingGui, guiDataChanged;
    int debug;
    ListMap listMap;
    ScriptTemplate *scriptTemplate;
    Script *script;

    /**
      * Our object to access KDE style configuration.
      */
    KConfig   *config;
    /**
      * The icon used in script and templates list to
      * show modified and unsaved scripts.
      */
    QPixmap  modifiedIcon;
    /**
      * This preference object is used for getting out
      * data from dialogs: script edit and script add.
      */
    ScriptPref pref;
    QStringList facilitylist, logfmtlist;
};

#endif

