/***************************************************************************
 *   Copyright (C) 2005 by Felipe Sateler                                  *
 *   fsateler@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#include <qlayout.h>

#include <klocale.h>
#include <kglobal.h>
#include <kaboutdata.h>
#include <kparts/genericfactory.h>
#include <klistview.h>
#include <kpushbutton.h>
#include <qgroupbox.h>
#include <qvaluelist.h>
#include <kdesktopfile.h>
#include <kdirlister.h>
#include <kstandarddirs.h>

#include <iostream>

#include "autostart.h"
#include "autostartwidget.h"

typedef KGenericFactory<Autostart, QWidget> AutostartFactory;
K_EXPORT_COMPONENT_FACTORY( kcm_Autostart, AutostartFactory("kcmautostart"))

extern "C" {
	KCModule* create_autostart(QWidget* parent, const char* name)
	{
		return new Autostart(parent,name);
	}
}

Autostart::Autostart(QWidget *parent, const char *name, const QStringList&)
	: KCModule(parent, name), myAboutData(0), widget(0)
{
	myAboutData = new KAboutData("kcmautostart", I18N_NOOP("Autostart"), "0.1", 
								 "Module for configuring apps that autostart",
								 KAboutData::License_GPL_V2, NULL, NULL,
								 "fsateler@gmail.com");
	myAboutData->addAuthor("Felipe Sateler", "", "fsateler@gmail.com", "");
	
	QBoxLayout *l = new QVBoxLayout( this );
	l->setAutoAdd( TRUE );
	
	widget = new AutostartWidget(this, "AutostartWidget");
	connect( widget, SIGNAL(changed()), this, SLOT(widgetChanged()) );
	
	load();
};


Autostart::~Autostart()
{}

void Autostart::load()
{
	lister = new KDirLister();
	
	connect( lister, SIGNAL(completed()), this, SLOT(completed()) );
	connect( lister, SIGNAL(newItems(const KFileItemList& )), this, SLOT(newItems(const KFileItemList& )) );
	
	lister->openURL( KGlobal::dirs()->localkdedir()+"/Autostart" );
}

/*
void Autostart::defaults()
{
	// insert your default settings code here...
	emit changed(true);
}
*/

void Autostart::save()
{
	// First remove all files in the directory
	lister = new KDirLister();
	
	connect( lister, SIGNAL(completed()), this, SLOT(removeCompleted()) );
	connect( lister, SIGNAL(newItems(const KFileItemList& )), this, SLOT(removeItems(const KFileItemList& )) );
	
	lister->openURL(KGlobal::dirs()->localkdedir()+"/Autostart");
	
	info = widget->save();
	QValueListConstIterator<AppInfo> it = info.constBegin();
	QValueListConstIterator<AppInfo> end = info.constEnd();
	while( it!=end )
	{
		KDesktopFile file(KGlobal::dirs()->localkdedir()+"/Autostart/"+(*it).name+".desktop");
		file.writeEntry("Name", (*it).name);
		file.writeEntry("Comment", (*it).comment);
		file.writeEntry("Exec",(*it).command);
		++it;
	}

	emit changed(true);
}


int Autostart::buttons()
{
	return /*KCModule::Default|*/KCModule::Apply|KCModule::Help;
}


void Autostart::configChanged()
{
	// insert your saving code here...
	emit changed(true);
}


QString Autostart::quickHelp() const
{
	return i18n("Configures applications that are automatically"
			" started during KDE startup");
}

void Autostart::completed()
{
	// Ensure no more signals are called
	lister->deleteLater();
	widget->load(info);
	info.clear();
}

void Autostart::removeCompleted()
{
	lister->deleteLater();
}

void Autostart::newItems(const KFileItemList& items)
{
	QPtrListIterator<KFileItem> it( items );
	KFileItem *curr;
	while( ( curr = it.current() ) != NULL ){
		++it;
		AppInfo newInfo;
		if ( !curr->isFile() )
			continue;
		if( KDesktopFile::isDesktopFile( curr->url().path() ) )
		{
			newInfo = parseDesktopFile( curr->url() );
		}
		else
		{
			newInfo.name = curr->url().fileName();
			newInfo.command = QString::null;
			newInfo.comment = QString::null;
		}
		info.append(newInfo);
	}
}

void Autostart::removeItems(const KFileItemList& items)
{
	QPtrListIterator<KFileItem> it( items );
	KFileItem *curr;
	while( ( curr = it.current() ) != NULL ){
		++it;
		if ( !curr->isFile() )
			continue;
		unlink( curr->localPath() );
	}
}

AppInfo Autostart::parseDesktopFile(const KURL& url)
{
	KDesktopFile file( url.path() );
	AppInfo app;
	
	app.name = file.readName();
	app.comment = file.readComment();
	app.command = file.readEntry("Exec");
	
	return app;
}

void Autostart::widgetChanged()
{
	emit changed(true);
}

#include "autostart.moc"
