/*
 * Copyright (C) 2001 Johnathan Burchill,,, <jkerrb@users.sourceforge.net>
 */

#ifndef _KDARVIEW_H_
#define _KDARVIEW_H_

#include "kdardlg.h"

#include <kio/jobclasses.h>
#include <kparts/part.h>
#include <kdariface.h>

#include "kdarListRecursive.h"

#include <qdatetime.h>

class controller;
class KDarConfig;
class kdar;

class kdarCreateDialog;
class kdarCreateArchiveWizard;
class kdarEventHandler;
class kdarExtractDialog;
class kdarProgressDialog;

class KConfig;
class KDialogBase;
//class KIO::KDarListJob;
class KListView;
class KParts::ReadOnlyPart;
class KPrinter;
class KProgress;
class KURL;
class KURLPixmapProvider;

class QPainter;
class QPaintDeviceMetrics;
class QPoint;
class QTimer;

/**
 * This is the main view class for kdar.  Most of the non-menu,
 * non-toolbar, and non-statusbar (e.g., non frame) GUI code should go
 * here.
 *
 * @short Main view
 * @author Johnathan Burchill, <jkerrb@users.sourceforge.net>
 * @version VERSION
 */
class kdarView : public KDARDlg, public kdarIface
{
    Q_OBJECT
public:
    /**
     * Constructor
     */
    kdarView( controller *aController, QWidget *parent);
    /**
     * Destructor
     */
    virtual ~kdarView();

    /**
     * Returns the global KConfig *.
     */
    KConfig * globalConfig() const;
    /**
     * Print this view to any medium -- paper or not
     */
    void print(QPainter *, KPrinter *, QPaintDeviceMetrics *);
    /**
     * This function is called when this app is restored.  The KConfig
     * object points to the session management config file that was saved
     * with @ref saveProperties
     */
    void readProperties();
    /**
     * This function is called when it is time for the app to save its
     * properties for session management purposes.
     */
    void saveProperties();
    /**
     * Toggles the visibility of the KDar log window
     */
    void showLogWindow( bool setVisible = true );
    /**
     * Handles dropping a disk-archive file onto the main view.
     */
    void dropEvent(QDropEvent *event);

    /**
     * Points to the advanced creation dialog.
     */
    kdarCreateDialog *createDlg;
    /**
     * Points to the creation wizard dialog.
     */
    kdarCreateArchiveWizard *createDlgWizard;
    /**
     * Points to the creation configure object
     */
    KDarConfig *m_createConfig;
    /**
     * Points to the restoration dialog object
     */
    kdarExtractDialog *extractDlg;
    /**
     * Append a message to the message window
     */
    void appendMessage( const QString & );
    /**
     * Stop checking the slice size
     */
    void stopFileSizeChecks();
    /**
     * Start checking the slice size
     */
    void startFileSizeChecks();
    /**
     * Suspend checking the slice size
     */
    bool suspendFileSizeChecks();
    /**
     * Resume checking the slice size
     */
    void resumeFileSizeChecks();
    /**
     * Display a message on the main statusbar
     */
    void displayStatusBarMessage( const QString &, const int );
    /**
     * Update the progress dialog's file info
     */
    void updateProgressDialogFileInfo( QString & );
    /**
     * Update the progress dialog's extra file info, like whether
     * EA could be saved, or file could not be saved or read, etc.
     */
    void updateProgressDialogFileExtraInfo( QString & );
    /**
     * Display a new top-level archive list item. text is
     * the archive's name.
     */
    void displayNewArchive( QString &, bool );
    /**
     * Enable/disable the use of the KDE wallet
     */
    void enableWallet( bool enable = true );
    /**
     * Use the KDE wallet to store/retrieve passwords?
     */
    bool useWallet() const;

protected:
    /**
     * This function traps QCustomEvents
     */
    virtual void customEvent( QCustomEvent * );

signals:

public slots:
    /**
     * Cancel a libdar operation.
     * @return returns whether the operation was successfully cancelled.
     */
    bool cancelOperation();
    /**
     * Allows user to configure all KDar options.
     */
    void optionsPreferences();
    /**
     * Open an archive whose name is "text". If "text" is null, the user must
     * select an archive using a KFileDialog.
     */
    void slotArchiveChoose( const QString& text = QString::null );
    /**
     * Overloaded function. Behaves like the above function, but takes a KURL
     * as input.
     */
    void slotArchiveChoose( const KURL & );
    /**
     * A context menu for the archiveBrowser KListView
     */
    void slotArchiveContextMenu( KListView *, QListViewItem *, const QPoint & );
    /**
     * This function creates an archive from the filesystem.
     */
    void slotArchiveCreate();
    /**
     * Calls libdar::get_children_of() to populate the expandItem
     */
    void slotArchiveDescend( QListViewItem * expandItem );
    /**
     * This function compares an archive's contents with the contents of
     * the filesystem.
     */
    void slotArchiveDiff();
    /**
     * This function restores an archive onto the filesystem.
     */
    void slotArchiveExtract();
    /**
     * This function isolates a catalog from an archive.
     */
    void slotArchiveIsolate();
    /**
     * List the top-level contents of the archive.
     */
    void slotArchiveList( const bool loadArchive = true );
    /**
     * This function opens listViewItem if is closed, and closes it if
     * it open. If the item has no children, and it has no parent, then
     * slotArchiveList() is called.
     */
    void slotArchiveOpenFromDoubleClick(QListViewItem * item);
    /**
     * This function tests an archive for data integrity. Uses a CRC.
     */
    void slotArchiveTest();
    /**
     * Clears the message window.
     */
    void slotClearMessages();
    /**
     * Restores a creation profile from a file called "profile".
     */
    void slotRestoreCreationProfile( const QString & profile = QString::null );
    /**
     * Configure the creation settings for a new archive.
     */
    void slotConfigureCreationSettings();
    /**
     * Configure the restoration settings for an existing archive.
     */
    void slotConfigureRestoreSettings();
    /**
     * Save current creation settings to a profile file.
     */
    void slotSaveCreationProfile();
    /**
     * Save current restoration settings to a profile file.
     */
    void slotSaveRestoreProfile();
    /**
     * Restores a restoration profile from a file called "profile".
     */
    void slotRestoreRestoreProfile();
    /**
     * Export an equivalent dar archive creation command line.
     */
    void slotExportDarCreateCommand();
    /**
     * Export an equivalent dar archive restoration command line.
     */
    void slotExportDarRestoreCommand();
//    /**
//     * Sets the KDarConfig configuration object pointer.
//     */
//    void    setKDarConfig( KDarConfig * );

private slots:
    /**
     * Checks the size of the archive slice that is currently being created.
     */
    void slotCreationCheckFileSize();
    /**
     * Updates the creation progress bar with the number of files added,
     * compared to the total number of files that will be added to the archive.
     */
    void slotUpdateProgressBar();
    /**
     * Increases the estimated number of files to be added to the archive
     * as estimated by the KDarListRecursive job.
     */
    void slotIncrementTotalFiles( KIO::Job * job, const unsigned long );
    /**
     * Sets the total number of files that have been estimated to be added to
     * an archive. This is a guess, and is not exact, but suffices for most
     * purposes. This is called when the KIO::listRecursive() ListJob has
     * finished.
     */
    void slotSetTotalFiles( KIO::Job * );
    /**
     * Updates the progress dialog with the elapsed time.
     */
    void slotElapsedTime();
    /**
     * Calls the contoller's changePassword function
     */
    void slotChangePassword();

private:
    /**
     * Update the main window when a different font has been configured.
     */
    void updateGUI();
    /**
     * Returns a QString of the time elapsed since the libdar operation began.
     */
    QString elapsedTime();
    /**
     * Disable the parts of KDar that depend on missing libdar features.
     *
     * param[in] config is the KDarConfig object to be disabled
     */
    void disableFeatures( KDarConfig * config );

    controller *m_controller;
    kdarEventHandler *m_eventHandler;
    KURLPixmapProvider *kdarPixmapProvider;

    KDarConfig *m_kdarconfig;
    KDarConfig *m_restoreConfig;

    QTimer *fileSizeTimer;
    QTimer *progressTimer;
    QTimer *m_elapsedTimer;

    KIO::KDarListJob * m_rootDirListJob;
    KProgress *progressBar;
    kdarProgressDialog *m_progressDialog;
    long m_filesProcessedAdded;
    long m_filesProcessedSkipped;
    long m_filesProcessedTotal;
    long m_totalFiles;
    long m_tmpTotalFiles;

    KDialogBase *logDialog;
    KParts::ReadOnlyPart* logPart;

    QDateTime m_startTime;
};

#endif // _KDARVIEW_H_
