/* Bespin widget style for Qt4
   Copyright (C) 2007 Thomas Luebking <thomas.luebking@web.de>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 */

#include <QAbstractItemView>
// #include <QAbstractScrollArea>
// #include <QAbstractSlider>
#include <QApplication>
#include <QLayout>
#include <QMenu>
#include <QMenuBar>
#include <QPainter>
#include <QPen>
#include <QToolBar>
#include <QToolTip>
#include <QTreeView>
#include <QtDBus/QDBusInterface>

#include "colors.h"

#ifdef Q_WS_X11
#include "xproperty.h"
#endif

#include "visualframe.h"
#include "bespin.h"
#include "hacks.h"

#include "macmenu.h"

#include "animator/hover.h"
#include "animator/aprogress.h"
#include "animator/tab.h"

#include "makros.h"
#undef CCOLOR
#undef FCOLOR
#define CCOLOR(_TYPE_, _FG_) PAL.color(QPalette::Active, config._TYPE_##_role[_FG_])
#define FCOLOR(_TYPE_) PAL.color(QPalette::Active, QPalette::_TYPE_)

using namespace Bespin;

extern Config config;
extern Dpi dpi;

static inline void
setBoldFont(QWidget *w, bool bold = true)
{
    QFont fnt = w->font();
    fnt.setBold(bold);
    w->setFont(fnt);
}

void BespinStyle::polish ( QApplication * app )
{
    QPalette pal = app->palette();
    polish(pal);
    app->setPalette(pal);
}

#define _SHIFTCOLOR_(clr) clr = QColor(CLAMP(clr.red()-10,0,255),CLAMP(clr.green()-10,0,255),CLAMP(clr.blue()-10,0,255))

static QColor
mid(const QColor &c1, const QColor &c2, int w1 = 1, int w2 = 1)
{
    int sum = (w1+w2);
    return QColor(((w1*c1.red() + w2*c2.red())/sum) & 0xff,
                 ((w1*c1.green() + w2*c2.green())/sum) & 0xff,
                 ((w1*c1.blue() + w2*c2.blue())/sum) & 0xff,
                 ((w1*c1.alpha() + w2*c2.alpha())/sum) & 0xff);
}

static QDBusInterface bespinDeco( "org.kde.kwin", "/BespinDeco", "org.kde.BespinDeco");

#undef PAL
#define PAL pal

void BespinStyle::polish( QPalette &pal )
{
    QColor c = pal.color(QPalette::Active, QPalette::Background);
    if (config.bg.mode > Plain)
    {
        int h,s,v;
        c.getHsv(&h,&s,&v);
        if (v < 80) // very dark colors won't make nice backgrounds ;)
            c.setHsv(h,s,80);
        pal.setColor( QPalette::Window, c );
    }

    // AlternateBase
    pal.setColor(QPalette::AlternateBase, mid(pal.color(QPalette::Active, QPalette::Base),
                                              pal.color(QPalette::Active, QPalette::Text),15,1));
    // highlight colors
    const int highlightGray = qGray(pal.color(QPalette::Active, QPalette::Highlight).rgb());
    const QColor grey(highlightGray,highlightGray,highlightGray);
    pal.setColor(QPalette::Disabled, QPalette::Highlight, grey);

    // dark, light & etc are tinted... no good:
    pal.setColor(QPalette::Dark, QColor(70,70,70));
    pal.setColor(QPalette::Mid, QColor(100,100,100));
    pal.setColor(QPalette::Midlight, QColor(220,220,220));
    pal.setColor(QPalette::Light, QColor(240,240,240));

    // Link colors can not be set through qtconfig - and the colors suck
    pal.setColor(QPalette::Link, mid(pal.color(QPalette::Active, QPalette::Text),
                                     pal.color(QPalette::Active, QPalette::Highlight), 1, 8));
    pal.setColor(QPalette::LinkVisited, mid(pal.color(QPalette::Active, QPalette::Text),
                                            pal.color(QPalette::Active, QPalette::Highlight), 1, 4));

#if QT_VERSION >= 0x040400
    // tooltip (NOTICE not configurable by qtconfig, kde can, let's see what we're gonna do on this...)
    pal.setColor(QPalette::ToolTipBase, pal.color(QPalette::Active, config.bg.tooltip_role[Bg]));
    pal.setColor(QPalette::ToolTipText, pal.color(QPalette::Active, config.bg.tooltip_role[Fg]));
#endif

    // inactive palette
    if (config.fadeInactive)
    { // fade out inactive foreground and highlight colors...
        pal.setColor(QPalette::Inactive, QPalette::Highlight,
                    mid(pal.color(QPalette::Active, QPalette::Highlight), grey, 2,1));
        pal.setColor(QPalette::Inactive, QPalette::WindowText,
                    mid(pal.color(QPalette::Active, QPalette::Window), pal.color(QPalette::Active, QPalette::WindowText), 1,4));
        pal.setColor(QPalette::Inactive, QPalette::ButtonText,
                    mid(pal.color(QPalette::Active, QPalette::Button), pal.color(QPalette::Active, QPalette::ButtonText), 1,4));
        pal.setColor(QPalette::Inactive, QPalette::Text,
                    mid(pal.color(QPalette::Active, QPalette::Base), pal.color(QPalette::Active, QPalette::Text), 1,4));
    }

    // fade disabled palette
    pal.setColor(QPalette::Disabled, QPalette::WindowText,
                 mid(pal.color(QPalette::Active, QPalette::Window), pal.color(QPalette::Active, QPalette::WindowText),2,1));
    pal.setColor(QPalette::Disabled, QPalette::Base,
                 mid(pal.color(QPalette::Active, QPalette::Window), pal.color(QPalette::Active, QPalette::Base),1,2));
    pal.setColor(QPalette::Disabled, QPalette::Text,
                 mid(pal.color(QPalette::Active, QPalette::Base), pal.color(QPalette::Active, QPalette::Text)));
    pal.setColor(QPalette::Disabled, QPalette::AlternateBase,
                 mid(pal.color(QPalette::Disabled, QPalette::Base), pal.color(QPalette::Disabled, QPalette::Text),15,1));

    // more on tooltips... (we force some colors...)
    QPalette toolPal = QToolTip::palette();
    const QColor bg = pal.color(config.bg.tooltip_role[Bg]);
    const QColor fg = pal.color(config.bg.tooltip_role[Fg]);
    toolPal.setColor(QPalette::Window, bg);
    toolPal.setColor(QPalette::WindowText, fg);
    toolPal.setColor(QPalette::Base, bg);
    toolPal.setColor(QPalette::Text, fg);
    toolPal.setColor(QPalette::Button, bg);
    toolPal.setColor(QPalette::ButtonText, fg);
    toolPal.setColor(QPalette::Highlight, fg); // sic!
    toolPal.setColor(QPalette::HighlightedText, bg); // sic!
    toolPal.setColor(QPalette::ToolTipBase, bg);
    toolPal.setColor(QPalette::ToolTipText, fg);
    QToolTip::setPalette(toolPal);


#ifdef Q_WS_X11
    if (appType == GTK)
    {
        bespinDeco.call(QDBus::NoBlock, "styleByPid", QCoreApplication::applicationPid(),
                        XProperty::encode(FCOLOR(Window), FCOLOR(WindowText), config.bg.mode),
                        XProperty::encode(CCOLOR(kwin.active, Bg), CCOLOR(kwin.active, Fg), GRAD(kwin)[1]),
                        XProperty::encode(CCOLOR(kwin.inactive, Bg), fg, GRAD(kwin)[0]));
    }
#endif
}

#if 0
static QMenuBar *
bar4popup(QMenu *menu)
{
    if (!menu->menuAction())
        return 0;
    if (menu->menuAction()->associatedWidgets().isEmpty())
        return 0;
    foreach (QWidget *w, menu->menuAction()->associatedWidgets())
        if (qobject_cast<QMenuBar*>(w))
            return static_cast<QMenuBar *>(w);
    return 0;
}
#endif


inline static void
polishGTK(QWidget * widget)
{
    enum MyRole{Bg = BespinStyle::Bg, Fg = BespinStyle::Fg};
    if (widget->objectName() == "QPushButton" ||
        widget->objectName() == "QComboBox" ||
        widget->objectName() == "QCheckBox" ||
        widget->objectName() == "QRadioButton" )
    {
        QPalette pal = widget->palette();
        pal.setColor(QPalette::Disabled, QPalette::Button,
                    Colors::mid(Qt::black, FCOLOR(Window),5,100));
        pal.setColor(QPalette::Inactive, QPalette::Button, CCOLOR(btn.std, Bg));
        pal.setColor(QPalette::Active, QPalette::Button, CCOLOR(btn.active, Bg));

        pal.setColor(QPalette::Disabled, QPalette::ButtonText,
                    Colors::mid(FCOLOR(Window), FCOLOR(WindowText),3,1));
        pal.setColor(QPalette::Inactive, QPalette::ButtonText, CCOLOR(btn.std, Fg));
        pal.setColor(QPalette::Active, QPalette::ButtonText, CCOLOR(btn.active, Fg));
        widget->setPalette(pal);
    }
    if (widget->objectName() == "QTabWidget" ||
        widget->objectName() == "QTabBar")
    {
        QPalette pal = widget->palette();
        pal.setColor(QPalette::Inactive, QPalette::Window, CCOLOR(tab.std, Bg));
        pal.setColor(QPalette::Active, QPalette::Window, CCOLOR(tab.active, Bg));

        pal.setColor(QPalette::Disabled, QPalette::WindowText,
                    Colors::mid(CCOLOR(tab.std, Bg), CCOLOR(tab.std, Fg),3,1));
        pal.setColor(QPalette::Inactive, QPalette::WindowText, CCOLOR(tab.std, Fg));
        pal.setColor(QPalette::Active, QPalette::WindowText, CCOLOR(tab.active, Fg));
        widget->setPalette(pal);
    }
   
    if (widget->objectName() == "QMenuBar" )
    {
        QPalette pal = widget->palette();
        pal.setColor(QPalette::Inactive, QPalette::Window,
                    Colors::mid(FCOLOR(Window), CCOLOR(menu.bar, Bg)));
        pal.setColor(QPalette::Active, QPalette::Window, CCOLOR(menu.active, Bg));

        pal.setColor(QPalette::Inactive, QPalette::WindowText, CCOLOR(menu.bar, Fg));
        pal.setColor(QPalette::Active, QPalette::WindowText, CCOLOR(menu.active, Fg));
        widget->setPalette(pal);
    }

    if (widget->objectName() == "QMenu" )
    {
        QPalette pal = widget->palette();
        pal.setColor(QPalette::Inactive, QPalette::Window, CCOLOR(menu.std, Bg));
        pal.setColor(QPalette::Active, QPalette::Window, CCOLOR(menu.active, Bg));

        pal.setColor(QPalette::Inactive, QPalette::WindowText, CCOLOR(menu.std, Fg));
        pal.setColor(QPalette::Active, QPalette::WindowText, CCOLOR(menu.active, Fg));
        widget->setPalette(pal);
    }
}

void
BespinStyle::polish( QWidget * widget )
{
    // GTK-Qt gets a special handling - see above
    if (appType == GTK)
    {
        polishGTK(widget);
        return;
    }

    // !!! protect against polishing /QObject/ attempts! (this REALLY happens from time to time...)
    if (!widget)
        return;

#ifdef MOUSEDEBUG
   widget->installEventFilter(this);
#endif

    // NONONONONO!!!!! ;)
    if (qobject_cast<VisualFramePart*>(widget))
        return;

    // apply any user selected hacks
    Hacks::add(widget);

    //BEGIN Window handling                                                                        -
    if (widget->isWindow() && !widget->inherits("QTipLabel"))
    {
        QPalette pal = widget->palette();

        if (config.bg.mode > Plain)
            widget->setAttribute(Qt::WA_StyledBackground);

        //BEGIN Popup menu handling                                                                -
        if (QMenu *menu = qobject_cast<QMenu *>(widget))
        {
            if (config.menu.glassy)
            {   // glass mode popups
                menu->setAttribute(Qt::WA_MacBrushedMetal);
                menu->setAttribute(Qt::WA_StyledBackground);
            }
            // opacity
            menu->setWindowOpacity( config.menu.opacity/100.0 );
            // color swapping
            menu->setAutoFillBackground(true);
            menu->setBackgroundRole ( config.menu.std_role[Bg] );
            menu->setForegroundRole ( config.menu.std_role[Fg] );
            if (config.menu.boldText)
                setBoldFont(menu);
            
            // eventfiltering to reposition MDI windows and correct distance to menubars
            menu->installEventFilter(this);
#if 0
            /// NOTE this was intended to be for some menu mock from nuno where the menu
            // reaches kinda ribbon-like into the bar
            // i'll keep it to remind myself and in case i get it to work one day ;-)
            if (bar4popup(menu))
            {
                QAction *action = new QAction( menu->menuAction()->iconText(), menu );
                connect (action, SIGNAL(triggered(bool)), menu, SLOT(hide()));
                menu->insertAction(menu->actions().at(0), action);
            }
#endif
        }
        //END Popup menu handling                                                                  -

        /// WORKAROUND Qt color bug, uses daddies palette and FGrole, but TooltipBase as background
        else if (widget->inherits("QWhatsThat"))
            widget->setPalette(QToolTip::palette()); // so this is Qt bug WORKAROUND
    
        else
        {   /// modal dialogs
            if (config.bg.modal.invert || config.bg.modal.glassy || config.bg.modal.opacity < 100)
            // the modality isn't necessarily set yet, so we catch it on QEvent::Show
                widget->installEventFilter(this);

            // talk to kwin about colors, gradients, etc.
            Qt::WindowFlags ignore =    Qt::Sheet | Qt::Drawer | Qt::Popup | Qt::ToolTip |
                                        Qt::SplashScreen | Qt::Desktop |
                                        Qt::X11BypassWindowManagerHint;// | Qt::FramelessWindowHint; <- could easily change mind...?!
            ignore &= ~Qt::Dialog; // erase dialog, it's in drawer et al. - takes away window as well
            if (!(widget->windowFlags() & ignore))
                setupDecoFor(widget); // this can be expensive, so avoid for popups, combodrops etc.
        }

    }
    //END Window handling                                                                          -

    //BEGIN Frames                                                                                 -
    else if (QFrame *frame = qobject_cast<QFrame *>(widget)) // sic!!! no window frames!
    {
        // sunken looks soo much nicer ;)
        if (frame->parentWidget() && frame->parentWidget()->inherits("KTitleWidget"))
            frame->setFrameShadow(QFrame::Sunken);

        // Kill ugly winblows frames... (qShadeBlablabla stuff)
        else if (   frame->frameShape() == QFrame::Box ||
                    frame->frameShape() == QFrame::Panel ||
                    frame->frameShape() == QFrame::WinPanel)
            frame->setFrameShape(QFrame::StyledPanel);

        // Kill ugly line look (we paint our styled v and h lines instead ;)
        else if (   frame->frameShape() == QFrame::HLine ||
                    frame->frameShape() == QFrame::VLine)
            widget->installEventFilter(this);

        // scrollarea hovering
        if (qobject_cast<QAbstractScrollArea*>(frame)
#ifdef QT3_SUPPORT
            || frame->inherits("Q3ScrollView")
#endif
            )
        {
            Animator::Hover::manage(frame);
            if (QAbstractItemView *itemView = qobject_cast<QAbstractItemView*>(widget) )
            {
                if (QTreeView* tv = qobject_cast<QTreeView*>(frame))
                {   // allow all treeviews to be animated!
                    if (config.hack.treeViews && tv->viewport() &&
                        tv->viewport()->autoFillBackground())
                        tv->setAnimated ( true );
                }
                else // treeview hovering sucks, as the "tree" doesn't get an update
                { // Enable hover effects in listview, all itemviews like in kde is pretty annoying
                    itemView->viewport()->setAttribute(Qt::WA_Hover);
                    if (widget->inherits("QHeaderView"))
                        widget->setAttribute(Qt::WA_Hover);
                }
            }
        }

        /// Tab Transition animation,
        if (widget->inherits("QStackedWidget"))
            // NOTICE do NOT(!) apply this on tabs explicitly, as they contain a stack!
            Animator::Tab::manage(widget);

        /// QToolBox handling - a shame they look that crap by default!
        if (widget->inherits("QToolBox"))
        {   // get rid of QPalette::Button
            widget->setBackgroundRole(QPalette::Window);
            widget->setForegroundRole(QPalette::WindowText);
            if (widget->layout())
            {   // get rid of nasty indention
                widget->layout()->setMargin ( 0 );
                widget->layout()->setSpacing ( 0 );
            }
        }

        /// "Frame above Content" look, but ...
        else if (isSpecialFrame(widget))
        {   // ...QTextEdit etc. can be handled more efficiently
            if (frame->lineWidth() == 1)
                frame->setLineWidth(dpi.f4); // but must have enough indention
        }
        else
            VisualFrame::manage(frame);
    }
    //END FRAMES                                                                                   -

    //BEGIN PUSHBUTTONS - hovering/animation                                                       -
    else if (widget->inherits("QAbstractButton"))
    {
        if (widget->inherits("QToolBoxButton") || IS_HTML_WIDGET )
            widget->setAttribute(Qt::WA_Hover); // KHtml
        else
        {
            if (widget->inherits("QToolButton"))
            {
                widget->setBackgroundRole(QPalette::Window);
                widget->setForegroundRole(QPalette::WindowText);
            }
            Animator::Hover::manage(widget);
        }

        // NOTICE WORKAROUND - this widget uses the style to paint the bg, but hardcodes the fg...
        // TODO: inform Joseph Wenninger <jowenn@kde.org> and really fix this
        // (fails all styles w/ Windowcolored ToolBtn and QPalette::ButtonText != QPalette::WindowText settings)
        if (widget->inherits("KMultiTabBarTab"))
        {
            QPalette pal = widget->palette();
            pal.setColor(QPalette::Active, QPalette::Button, pal.color(QPalette::Active, QPalette::Window));
            pal.setColor(QPalette::Inactive, QPalette::Button, pal.color(QPalette::Inactive, QPalette::Window));
            pal.setColor(QPalette::Disabled, QPalette::Button, pal.color(QPalette::Disabled, QPalette::Window));

            pal.setColor(QPalette::Active, QPalette::ButtonText, pal.color(QPalette::Active, QPalette::WindowText));
            pal.setColor(QPalette::Inactive, QPalette::ButtonText, pal.color(QPalette::Inactive, QPalette::WindowText));
            pal.setColor(QPalette::Disabled, QPalette::ButtonText, pal.color(QPalette::Disabled, QPalette::WindowText));
            widget->setPalette(pal);
        }
        // NOTICE WORKAROUND - this widget paints no bg, uses foregroundcolor() to paint the text...
        // and has - of course - foregroundRole() == QPalette::ButtonText
        // TODO: inform Peter Penz <peter.penz@gmx.at> or *cough* Aaron J. Seigo <aseigo@kde.org> and really fix this
        if (widget->inherits("KUrlButton"))
        {
            widget->setBackgroundRole(QPalette::Window);
            widget->setForegroundRole(QPalette::WindowText);
        }
    }
    
    //BEGIN COMBOBOXES - hovering/animation                                                        -
    else if (widget->inherits("QComboBox"))
    {
        if (IS_HTML_WIDGET)
            widget->setAttribute(Qt::WA_Hover);
        else
            Animator::Hover::manage(widget);
    }
    //BEGIN SLIDERS / SCROLLBARS / SCROLLAREAS - hovering/animation                                -
    else if (qobject_cast<QAbstractSlider*>(widget))
    {
        widget->setAttribute(Qt::WA_Hover);
        // NOTICE
        // QAbstractSlider::setAttribute(Qt::WA_OpaquePaintEvent) saves surprisinlgy little CPU
        // so that'd just gonna add more complexity for literally nothing...
        if (widget->inherits("QScrollBar"))
        {   // ...as the slider is usually not bound to e.g. a "scrollarea"
            QWidget *dad = widget;
            if (!widget->parentWidget())
            {   // this catches e.g. plasma used QGraphicsProxyWidgets...
                qWarning("Bespin, transparent scrollbar!");
                widget->setAttribute(Qt::WA_OpaquePaintEvent, false);
            }
            while ((dad = dad->parentWidget()))
            {   // digg for a potential KHTMLView ancestor, making this a html input scroller
                if (dad->inherits("KHTMLView"))
                {   // NOTICE this slows down things as it triggers a repaint of the frame
                    widget->setAttribute(Qt::WA_OpaquePaintEvent, false);
                    // ...but this re-enbales speed and currently does the job
                    // TODO how's css/khtml policy on applying colors?
                    widget->setAutoFillBackground ( true );
                    widget->setBackgroundRole ( QPalette::Base ); // QPalette::Window looks wrong
                    break;
                }
            }
            
            /// Scrollarea hovering - yes, this is /NOT/ redundant to the one above!
            if (QWidget *area = widget->parentWidget())
            {
                if ((area = area->parentWidget())) // sic!
                {
                    if (qobject_cast<QAbstractScrollArea*>(area))
                        area = 0; // this is handled for QAbstractScrollArea, but...
                    else // Konsole, Kate, etc. need a special handling!
                        area = widget->parentWidget();
                }
                if (area)
                    Animator::Hover::manage(area, true);
            }
        }
    }

    //BEGIN PROGRESSBARS - hover/animation and bold font                                           -
    else if (widget->inherits("QProgressBar"))
    {
        widget->setAttribute(Qt::WA_Hover);
        setBoldFont(widget);
        Animator::Progress::manage(widget);
    }

    //BEGIN Tab animation, painting override                                                       -
    else if (qobject_cast<QTabBar *>(widget))
    {
        widget->setAttribute(Qt::WA_Hover);
        // the eventfilter overtakes the widget painting to allow tabs ABOVE the tabbar
        widget->installEventFilter(this);
    }

    /// Menubars and toolbar default to QPalette::Button - looks crap and leads to flicker...?!
    QMenuBar *mbar = qobject_cast<QMenuBar *>(widget);
    if (mbar && !((appType == QtDesigner) && mbar->inherits("QDesignerMenuBar")))
        MacMenu::manage(mbar);

    bool isTopContainer = (mbar || qobject_cast<QToolBar *>(widget));
#ifdef QT3_SUPPORT
    isTopContainer = isTopContainer || widget->inherits("Q3ToolBar");
#endif
    if (isTopContainer || qobject_cast<QToolBar *>(widget->parent()))
    {
        widget->setBackgroundRole(QPalette::Window);
        widget->setForegroundRole(QPalette::WindowText);
        if (!isTopContainer && widget->inherits("QToolBarHandle"))
            widget->setAttribute(Qt::WA_Hover);
    }

    /// hover some leftover widgets
    if (widget->inherits("QAbstractSpinBox") || widget->inherits("QSplitterHandle") ||
        widget->inherits("QWorkspaceTitleBar") || widget->inherits("Q3DockWindowResizeHandle"))
        widget->setAttribute(Qt::WA_Hover);

    /// this is for QToolBox kids - they're autofilled by default - what looks crap
    if (widget->autoFillBackground() && widget->parentWidget() &&
        ( widget->parentWidget()->objectName() == "qt_scrollarea_viewport" ) &&
        widget->parentWidget()->parentWidget() && //grampa
        qobject_cast<QAbstractScrollArea*>(widget->parentWidget()->parentWidget()) &&
        widget->parentWidget()->parentWidget()->parentWidget() && // grangrampa
        widget->parentWidget()->parentWidget()->parentWidget()->inherits("QToolBox") )
        {
            widget->parentWidget()->setAutoFillBackground(false);
            widget->setAutoFillBackground(false);
        }

    /// KHtml css colors can easily get messed up, either because i'm unsure about what colors
    /// are set or KHtml does wrong OR (mainly) by html "designers"
    if (IS_HTML_WIDGET)
    {   // the eventfilter watches palette changes and ensures contrasted foregrounds...
        widget->installEventFilter(this);
        QEvent ev(QEvent::PaletteChange);
        eventFilter(widget, &ev);
    }
}
#undef PAL

void
BespinStyle::unPolish( QApplication *app )
{
    app->setPalette(QPalette());
}

void
BespinStyle::unPolish( QWidget *widget )
{
    if (QFrame *frame = qobject_cast<QFrame *>(widget))
        VisualFrame::release(frame);
    if (QMenuBar *mbar = qobject_cast<QMenuBar *>(widget))
        MacMenu::release(mbar);

    Animator::Hover::release(widget);
    Animator::Progress::release(widget);
    Animator::Tab::release(widget);
    Hacks::remove(widget);

    widget->removeEventFilter(this);
}
#undef CCOLOR
#undef FCOLOR
