/*
 * KAsteroids - Copyright Martin R. Jones 1997 <mjones@kde.org>
 *
 * Part of the KDE project
 */

#ifndef __SPRITES_H__
#define __SPRITES_H__

#include <QGraphicsPixmapItem>

#define ID_ROCK_LARGE           1024
#define ID_ROCK_MEDIUM          1025
#define ID_ROCK_SMALL           1026

#define ID_MISSILE              1030

#define ID_BIT                  1040
#define ID_EXHAUST              1041

#define ID_ENERGY_POWERUP       1310
#define ID_TELEPORT_POWERUP     1311
#define ID_BRAKE_POWERUP        1312
#define ID_SHIELD_POWERUP       1313
#define ID_SHOOT_POWERUP        1314

#define ID_SHIP                 1350
#define ID_SHIELD               1351

#define MAX_SHIELD_AGE          350
#define MAX_POWERUP_AGE         500
#define MAX_MISSILE_AGE         40

class KAstSprite : public QGraphicsPixmapItem
{
public:
    KAstSprite( QList<QPixmap>* frames, QGraphicsScene* scene )
        : QGraphicsPixmapItem( 0, scene ), mFrames(frames), mCurrentFrame(-1),
        mXVelocity(0), mYVelocity(0)
    {
        setFrame(0);
    }
    void setFrame(int frame)
    {
        if( frame != mCurrentFrame && !mFrames->isEmpty() )
        {
            mCurrentFrame = frame % mFrames->count();
            setPixmap( mFrames->at( mCurrentFrame ) );
        }
    }
    int frame() const { return mCurrentFrame; }
    int frameCount() const { return mFrames->count(); }
    void setVelocity(qreal xv, qreal yv) { mXVelocity = xv; mYVelocity = yv; }
    qreal xVelocity() const { return mXVelocity; }
    qreal yVelocity() const { return mYVelocity; }
    virtual void advance(int phase) 
    { if(phase==0) return; moveBy( mXVelocity, mYVelocity ); }
private:
    QList<QPixmap>* mFrames;
    int mCurrentFrame;
    qreal mXVelocity;
    qreal mYVelocity;
};

class KMissile : public KAstSprite
{
public:
    KMissile( QList<QPixmap> *frames, QGraphicsScene *scene ) 
        : KAstSprite( frames, scene ), myAge(0)
        { }

    virtual int type() const { return ID_MISSILE; }

    void growOlder() { myAge++; }
    bool expired() { return myAge > MAX_MISSILE_AGE; }

private:
    int myAge;
};

class KBit : public KAstSprite
{
public:
    KBit( QList<QPixmap> *frames, QGraphicsScene *scene ) 
        : KAstSprite( frames, scene ), death(7)
	{  }

    virtual int type() const {  return ID_BIT; }

    void setDeath( int d ) { death = d; }
    void growOlder() { death--; }
    bool expired() const { return death <= 0; }

private:
    int death;
};

class KExhaust : public KAstSprite
{
public:
    KExhaust( QList<QPixmap> *frames, QGraphicsScene *scene ) 
        : KAstSprite( frames, scene ), death(1)
	{ }

    virtual int type() const {  return ID_EXHAUST; }

    void setDeath( int d ) { death = d; }
    void growOlder() { death--; }
    bool expired() const { return death <= 0; }

private:
    int death;
};

class KPowerup : public KAstSprite
{
public:
  KPowerup( QList<QPixmap> *frames, QGraphicsScene *scene, int t ) 
        : KAstSprite( frames, scene ), myAge(0), myType( t )
  { }

  virtual int type() const { return myType; }

  void growOlder() { myAge++; }
  bool expired() const { return myAge > MAX_POWERUP_AGE; }

protected:
  int myAge;
  int myType;
};

class KRock : public KAstSprite
{
public:
    KRock (QList<QPixmap> *frames, QGraphicsScene *scene, int t, int sk, int st) 
        : KAstSprite( frames, scene ), myType(t), skip(sk), cskip(sk), step(st)
        { }

    void nextFrame()
	{
	    if (cskip-- <= 0) {
		setFrame( (frame()+step+frameCount())%frameCount() );
		cskip = qAbs(skip);
	    }
	}

    virtual int type() const { return myType; }

private:
    int myType;
    int skip;
    int cskip;
    int step;
};

class KShield : public KAstSprite
{
public:
  KShield( QList<QPixmap> *frames, QGraphicsScene *scene )
      : KAstSprite( frames, scene ) {}

  virtual int type() const { return ID_SHIELD; }
};

#endif
