/* Yo Emacs, this -*- C++ -*-

  Copyright (C) 1999-2001 Jens Hoefkens
  jens@hoefkens.com

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  $Id: kplayerlist.cpp 695059 2007-08-01 09:44:53Z dimsuz $

*/

#include "kplayerlist.h"
#include "kplayerlist.moc"

#include <QLayout>
#include <kicon.h>
#include <QGroupBox>
#include <QMenu>
#include <QCheckBox>
//#include <q3listview.h>
#include <Q3ListView>

#include <QDateTime>
#include <QLabel>
#include <QTabWidget>
#include <QHideEvent>
#include <QShowEvent>
#include <QGridLayout>
#include <QApplication>

#include <kconfig.h>
#include <kaction.h>
#include <kstandardaction.h>
#include <kiconloader.h>
#include <kdebug.h>
#include <kactioncollection.h>

#include <string.h>
#include <stdio.h>
#include <ktoolinvocation.h>
#include <kglobal.h>

#include "kbgfibs.h"
#include "version.h"


/*
 * Simple container for information on columns of the list view.
 *
 * index : the internal index in the list
 * width : width of the column in pixel
 * show  : whether the column is visible
 * cb    : check box for the setup dialog
 */
class KFibsPlayerListCI {

public:

    int index, width;
    bool show;
    QCheckBox *cb;
    QString key, name;

};

/*
 * Extension of the QListViewItem class that has a custom key function
 * that can deal with the different items of the player list.
 */
class KFibsPlayerListLVI : public K3ListViewItem {

public:

    /*
     * Constructor
     */
    KFibsPlayerListLVI(KFibsPlayerList *parent) : K3ListViewItem(parent) { _plist = parent; }

    /*
     * Destructor
     */
    virtual ~KFibsPlayerListLVI() {}

    /*
     * Overloaded key function for sorting
     */
    virtual QString key(int col, bool) const
    {
        int real_col = _plist->cIndex(col);

        QString s = text(col);

        switch (real_col) {
        case KFibsPlayerList::Player:
        case KFibsPlayerList::Opponent:
        case KFibsPlayerList::Watches:
        case KFibsPlayerList::Client:
        case KFibsPlayerList::Email:
        case KFibsPlayerList::Status:
        case KFibsPlayerList::Host:
            s = s.toLower();
            break;
        case KFibsPlayerList::Idle:
        case KFibsPlayerList::Experience:
            s.sprintf("%08d", s.toInt());
            break;
        case KFibsPlayerList::Rating:
            s.sprintf("%08d", (int)(1000*s.toDouble()));
            break;
        case KFibsPlayerList::Time:
            s = s.toLower();
            break;
        default:
            kDebug(10500) << "KFibsPlayerListLVI::key(): illegal column";
            break;
        }
        return s;
    }

private:

    KFibsPlayerList *_plist;

};

/*
 * Private data of the player list
 */
class KFibsPlayerListPrivate {

public:

    /*
     * Named constants for the popup menu actions
     */
    enum MenuID {Info, Talk, Mail, InviteD, Invite1, Invite2, Invite3, Invite4,
                 Invite5, Invite6, Invite7, InviteR, InviteU,
                 Look, Watch, Unwatch, BlindAct, Update, Reload, Close, ActionEnd};

    /*
     * Various actions for the context menu
     */
    QAction *mAct[ActionEnd];

    /*
     * All relevant information on the columns
     */
    KFibsPlayerListCI *mCol[KFibsPlayerList::LVEnd];

    /*
     * Context menus for player related commands
     */
    QMenu *mPm[2];

    /*
     * ID of the invite menu in the context menu
     */
    QAction *mInAct;

    /*
     * Are we watching?
     */
    bool mWatch;

    /*
     * count similar clients - KFibs & kbackgammon
     */
    int mCount[2];

    /*
     * Short abbreviations for Blind, Ready, and Away.
     */
    QString mAbrv[KFibsPlayerList::MaxStatus];

    /*
     * Name of the last selected player - for internal purposes
     */
    QString mUser;

    /*
     * Our own name
     */
    QString mName;

    /*
     * Email address of the last selected player - for internal purposes
     */
    QString mMail;

};


// == constructor, destructor and setup ========================================

/*
 * Construct the playerlist and do some initial setup
 */
KFibsPlayerList::KFibsPlayerList(QWidget *parent)
    : K3ListView(parent)
{
    d = new KFibsPlayerListPrivate();
    KActionCollection* actions = new KActionCollection(this);

    /*
     * Allocate the column information
     */
    for (int i = 0; i < LVEnd; i++)
        d->mCol[i] = new KFibsPlayerListCI();

    /*
     * Initialize variables
     */
    d->mCol[Player]->name     = i18n("Player");
    d->mCol[Opponent]->name   = i18n("Opponent");
    d->mCol[Watches]->name    = i18n("Watches");
    d->mCol[Status]->name     = i18n("Status");
    d->mCol[Rating]->name     = i18n("Rating");
    d->mCol[Experience]->name = i18n("Exp.");
    d->mCol[Idle]->name       = i18n("Idle");
    d->mCol[Time]->name       = i18n("Time");
    d->mCol[Host]->name       = i18n("Host name");
    d->mCol[Client]->name     = i18n("Client");
    d->mCol[Email]->name      = i18n("Email");

    // These strings shouldn't be translated!!
    d->mCol[Player]->key     = "player";
    d->mCol[Opponent]->key   = "opponent";
    d->mCol[Watches]->key    = "watches";
    d->mCol[Status]->key     = "status";
    d->mCol[Rating]->key     = "rating";
    d->mCol[Experience]->key = "experience";
    d->mCol[Idle]->key       = "idle";
    d->mCol[Time]->key       = "time";
    d->mCol[Host]->key       = "hostname";
    d->mCol[Client]->key     = "client";
    d->mCol[Email]->key      = "email";

    d->mCount[0] = d->mCount[1] = 0;

    d->mAbrv[Blind] = i18nc("abbreviate blind", "B");
    d->mAbrv[Away ] = i18nc("abbreviate away", "A");
    d->mAbrv[Ready] = i18nc("abbreviate ready", "R");

    d->mName = QString();

    d->mWatch = false;

    /*
     * Get a sane caption, initialize some eye candy and read the
     * configuration - needed for the column information.
     */
    updateCaption();
    setWindowIcon(qApp->windowIcon().pixmap(16,16));
    this->setWhatsThis( i18n("This window contains the player list. It shows "
                               "all players that are currently logged into FIBS."
                               "Use the right mouse button to get a context "
                               "menu with helpful information and commands."));

    readColumns();

    /*
     * Put the columns into the list view
     */
    for (int i = 0; i < LVEnd; i++) {
        if (d->mCol[i]->show) {
            d->mCol[i]->index = addColumn(d->mCol[i]->name, d->mCol[i]->width);
            if (i == Experience || i == Rating || i == Time || i == Idle)
                setColumnAlignment(d->mCol[i]->index, Qt::AlignRight);
        } else {
            d->mCol[i]->index = -1;
        }
    }
    setAllColumnsShowFocus(true);

    /*
     * Create context menus
     */
    d->mPm[0] = new QMenu();
    d->mPm[1] = new QMenu(i18n("Invite"));

    /*
     * Create the whole set of actions
     */
    d->mAct[KFibsPlayerListPrivate::Info] = actions->addAction("fibs_info");
    d->mAct[KFibsPlayerListPrivate::Info]->setIcon(KIcon("help-contents"));
    d->mAct[KFibsPlayerListPrivate::Info]->setText(i18n("Info"));
    connect(d->mAct[KFibsPlayerListPrivate::Info],  SIGNAL(triggered(bool) ), SLOT(slotInfo()));
    d->mAct[KFibsPlayerListPrivate::Talk] = actions->addAction("fibs_talk");
    d->mAct[KFibsPlayerListPrivate::Talk]->setIcon(KIcon(PROG_NAME "-chat.png"));
    d->mAct[KFibsPlayerListPrivate::Talk]->setText(i18n("Talk"));
    connect(d->mAct[KFibsPlayerListPrivate::Talk], SIGNAL(triggered(bool)), SLOT(slotTalk()));

    d->mAct[KFibsPlayerListPrivate::Look] = actions->addAction("fibs_lock");
    d->mAct[KFibsPlayerListPrivate::Look]->setText(i18n("Look"));
    connect(d->mAct[KFibsPlayerListPrivate::Look], SIGNAL(triggered(bool) ), SLOT(slotLook()));
    d->mAct[KFibsPlayerListPrivate::Watch] = actions->addAction("fibs_watch");
    d->mAct[KFibsPlayerListPrivate::Watch]->setText(i18n("Watch"));
    connect(d->mAct[KFibsPlayerListPrivate::Watch], SIGNAL(triggered(bool) ), SLOT(slotWatch()));
    d->mAct[KFibsPlayerListPrivate::Unwatch] = actions->addAction("fibs_unwatch");
    d->mAct[KFibsPlayerListPrivate::Unwatch]->setText(i18n("Unwatch"));
    connect(d->mAct[KFibsPlayerListPrivate::Unwatch], SIGNAL(triggered(bool) ), SLOT(slotUnwatch()));
    d->mAct[KFibsPlayerListPrivate::BlindAct] = actions->addAction("fibs_blind");
    d->mAct[KFibsPlayerListPrivate::BlindAct]->setText(i18n("Blind"));
    connect(d->mAct[KFibsPlayerListPrivate::BlindAct], SIGNAL(triggered(bool) ), SLOT(slotBlind()));
    d->mAct[KFibsPlayerListPrivate::Update] = actions->addAction("fibs_update");
    d->mAct[KFibsPlayerListPrivate::Update]->setText(i18n("Update"));
    connect(d->mAct[KFibsPlayerListPrivate::Update], SIGNAL(triggered(bool) ), SLOT(slotUpdate()));

    d->mAct[KFibsPlayerListPrivate::Reload] = KStandardAction::redisplay(this, SLOT(slotReload()), this);
    actions->addAction(d->mAct[KFibsPlayerListPrivate::Reload]->objectName(), d->mAct[KFibsPlayerListPrivate::Reload]);
    d->mAct[KFibsPlayerListPrivate::Mail]   = KStandardAction::mail(this, SLOT(slotMail()), this);
    actions->addAction(d->mAct[KFibsPlayerListPrivate::Mail]->objectName(), d->mAct[KFibsPlayerListPrivate::Mail]);
    d->mAct[KFibsPlayerListPrivate::Close]  = KStandardAction::close(this, SLOT(hide()), this);
    actions->addAction(d->mAct[KFibsPlayerListPrivate::Close]->objectName(), d->mAct[KFibsPlayerListPrivate::Close]);

    d->mAct[KFibsPlayerListPrivate::InviteD] = actions->addAction("fibs_ud");
    d->mAct[KFibsPlayerListPrivate::InviteD]->setText(i18n("Use Dialog"));
    connect(d->mAct[KFibsPlayerListPrivate::InviteD], SIGNAL(triggered(bool) ), SLOT(slotInviteD()));
    d->mAct[KFibsPlayerListPrivate::Invite1] = actions->addAction("fibs_pm1");
    d->mAct[KFibsPlayerListPrivate::Invite1]->setText(i18n("1 Point Match"));
    connect(d->mAct[KFibsPlayerListPrivate::Invite1], SIGNAL(triggered(bool) ), SLOT(slotInvite1()));
    d->mAct[KFibsPlayerListPrivate::Invite2] = actions->addAction("fibs_pm2");
    d->mAct[KFibsPlayerListPrivate::Invite2]->setText(i18n("2 Point Match"));
    connect(d->mAct[KFibsPlayerListPrivate::Invite2], SIGNAL(triggered(bool) ), SLOT(slotInvite2()));
    d->mAct[KFibsPlayerListPrivate::Invite3] = actions->addAction("fibs_pm3");
    d->mAct[KFibsPlayerListPrivate::Invite3]->setText(i18n("3 Point Match"));
    connect(d->mAct[KFibsPlayerListPrivate::Invite3], SIGNAL(triggered(bool) ), SLOT(slotInvite3()));
    d->mAct[KFibsPlayerListPrivate::Invite4] = actions->addAction("fibs_pm4");
    d->mAct[KFibsPlayerListPrivate::Invite4]->setText(i18n("4 Point Match"));
    connect(d->mAct[KFibsPlayerListPrivate::Invite4], SIGNAL(triggered(bool) ), SLOT(slotInvite4()));
    d->mAct[KFibsPlayerListPrivate::Invite5] = actions->addAction("fibs_pm5");
    d->mAct[KFibsPlayerListPrivate::Invite5]->setText(i18n("5 Point Match"));
    connect(d->mAct[KFibsPlayerListPrivate::Invite5], SIGNAL(triggered(bool) ), SLOT(slotInvite5()));
    d->mAct[KFibsPlayerListPrivate::Invite6] = actions->addAction("fibs_pm6");
    d->mAct[KFibsPlayerListPrivate::Invite6]->setText(i18n("6 Point Match"));
    connect(d->mAct[KFibsPlayerListPrivate::Invite6], SIGNAL(triggered(bool) ), SLOT(slotInvite6()));
    d->mAct[KFibsPlayerListPrivate::Invite7] = actions->addAction("fibs_pm7");
    d->mAct[KFibsPlayerListPrivate::Invite7]->setText(i18n("7 Point Match"));
    connect(d->mAct[KFibsPlayerListPrivate::Invite7], SIGNAL(triggered(bool) ), SLOT(slotInvite7()));
    d->mAct[KFibsPlayerListPrivate::InviteU] = actions->addAction("fibs_unlimited");
    d->mAct[KFibsPlayerListPrivate::InviteU]->setText(i18n("Unlimited"));
    connect(d->mAct[KFibsPlayerListPrivate::InviteU], SIGNAL(triggered(bool) ), SLOT(slotInviteU()));
    d->mAct[KFibsPlayerListPrivate::InviteR] = actions->addAction("fibs_resume");
    d->mAct[KFibsPlayerListPrivate::InviteR]->setText(i18n("Resume"));
    connect(d->mAct[KFibsPlayerListPrivate::InviteR], SIGNAL(triggered(bool) ), SLOT(slotInviteR()));

    /*
     * Fill normal context menu
     */
    d->mPm[0]->addAction( d->mAct[KFibsPlayerListPrivate::Info] );
    d->mPm[0]->addAction( d->mAct[KFibsPlayerListPrivate::Talk] );
    d->mPm[0]->addAction( d->mAct[KFibsPlayerListPrivate::Mail] );
    d->mPm[0]->addSeparator();
    d->mInAct = d->mPm[0]->addMenu(d->mPm[1]); // save Act for later
    d->mPm[0]->addAction( d->mAct[KFibsPlayerListPrivate::Look    ]);
    d->mPm[0]->addAction( d->mAct[KFibsPlayerListPrivate::Watch    ]);
    d->mPm[0]->addAction( d->mAct[KFibsPlayerListPrivate::Unwatch   ]);
    d->mPm[0]->addAction( d->mAct[KFibsPlayerListPrivate::BlindAct] );
    d->mPm[0]->addSeparator();
    d->mPm[0]->addAction( d->mAct[KFibsPlayerListPrivate::Update] );
    d->mPm[0]->addAction( d->mAct[KFibsPlayerListPrivate::Reload] );
    d->mPm[0]->addSeparator();
    d->mPm[0]->addAction( d->mAct[KFibsPlayerListPrivate::Close] );

    /*
     * Fill the invitation menu
     */
    d->mPm[1]->addAction( d->mAct[KFibsPlayerListPrivate::InviteD] );
    d->mPm[1]->addSeparator();
    d->mPm[1]->addAction( d->mAct[KFibsPlayerListPrivate::Invite1] );
    d->mPm[1]->addAction( d->mAct[KFibsPlayerListPrivate::Invite2] );
    d->mPm[1]->addAction( d->mAct[KFibsPlayerListPrivate::Invite3] );
    d->mPm[1]->addAction( d->mAct[KFibsPlayerListPrivate::Invite4] );
    d->mPm[1]->addAction( d->mAct[KFibsPlayerListPrivate::Invite5] );
    d->mPm[1]->addAction( d->mAct[KFibsPlayerListPrivate::Invite6] );
    d->mPm[1]->addAction( d->mAct[KFibsPlayerListPrivate::Invite7] );
    d->mPm[1]->addSeparator();
    d->mPm[1]->addAction( d->mAct[KFibsPlayerListPrivate::InviteU] );
    d->mPm[1]->addAction( d->mAct[KFibsPlayerListPrivate::InviteR] );

    /*
     * Right mouse button gets context menu, double click gets player info
     */
    connect(this, SIGNAL(contextMenu(K3ListView *, Q3ListViewItem *, const QPoint &)),
            this, SLOT(showContextMenu(K3ListView *, Q3ListViewItem *, const QPoint &)));
    connect(this, SIGNAL(doubleClicked(Q3ListViewItem *, const QPoint &, int)),
            this, SLOT(getPlayerInfo(Q3ListViewItem *, const QPoint &, int)));
}

/*
 * Destructor deletes members
 */
KFibsPlayerList::~KFibsPlayerList()
{
    for (int i = 0; i < LVEnd; i++)
        delete d->mCol[i];
    delete d->mPm[0];
    delete d->mPm[1];
    delete d;
}


// == settings and config ======================================================

/*
 * Called when the setup dialog is positively closed
 */
void KFibsPlayerList::setupOk()
{
    int i;
    bool change = false;

    for (i = 1; i < LVEnd; i++)
        change |= (d->mCol[i]->cb->isChecked() != d->mCol[i]->show);

    /*
     * Only juggle with the columns if something changed
     */
    if (change) {

        /*
         * It's important to remove the columns in reverse order
         */
        for (i = LVEnd-1; i > 0; i--)
            if (d->mCol[i]->show)
                removeColumn(d->mCol[i]->index);

        /*
         * Now add all columns that are selected
         */
        for (i = 1; i < LVEnd; i++) {
            if ((d->mCol[i]->show = d->mCol[i]->cb->isChecked())) {
                d->mCol[i]->index = addColumn(d->mCol[i]->name, d->mCol[i]->width);
                if (i == Experience || i == Rating || i == Time || i == Idle)
                    setColumnAlignment(d->mCol[i]->index, Qt::AlignRight);
            } else {
                d->mCol[i]->index = -1;
            }
        }

        /*
         * Reload the list
         */
        slotReload();
    }

    /*
     * store the new settings
     */
    saveConfig();

}

/*
 * Setup dialog page of the player list - allow the user to select the
 * columns to show
 */
void KFibsPlayerList::getSetupPages(QTabWidget *nb, int space)
{
    int i;

    /*
     * Main Widget
     */
    QWidget *w = new QWidget(nb);
    QGridLayout *gl = new QGridLayout(w);
    gl->setMargin(space);

    /*
     * Label
     */
    QGroupBox *gbl = new QGroupBox(w);
    gbl->setTitle(i18n("Column Selection"));

    gl->addWidget(gbl, 0, 0);

    /*
     * Note that the first column (Player == 0) is always there
     */
    QLabel *lb = new QLabel(i18n("Select all the columns that you would\n"
                                 "like to be shown in the player list."), gbl);

    for (i = 1; i < LVEnd; i++) {
        d->mCol[i]->cb = new QCheckBox(d->mCol[i]->name, gbl);
        d->mCol[i]->cb->setChecked(d->mCol[i]->show);
    }

    gl = new QGridLayout(gbl);
    gl->setMargin(20);
    gl->addWidget(lb, 0, 0);

    // two column layout....
    for (i = 1; i < LVEnd/2; i++) {
        gl->addWidget(d->mCol[2*i-1]->cb, i, 0);
        gl->addWidget(d->mCol[2*i  ]->cb, i, 1);
    }
    gl->addWidget(d->mCol[2*i-1]->cb, i, 0);
    if (2*i < LVEnd)
        gl->addWidget(d->mCol[2*i]->cb, i, 1);

    /*
     * put in the page and connect
     */
    nb->addTab(w, i18n("&Playerlist"));
}

/*
 * Nothing to cancel/undo
 */
void KFibsPlayerList::setupCancel()
{
    // do nothing
}

/*
 * By default all entries are checked
 */
void KFibsPlayerList::setupDefault()
{
    for (int i = 0; i < LVEnd; i++)
        d->mCol[i]->cb->setChecked(true);
}

/*
 * Restore the columns
 */
void KFibsPlayerList::readColumns()
{
    KSharedConfig::Ptr config = KGlobal::config();
    KConfigGroup cg(config, objectName());

    for (int i = 0; i < LVEnd; i++) {
        d->mCol[i]->show  = cg.readEntry("col-" + d->mCol[i]->key, true);
        d->mCol[i]->width = cg.readEntry("col-w-" + d->mCol[i]->key, -1);
    }
}

/*
 * Restore the saved settings
 */
void KFibsPlayerList::readConfig()
{
    KSharedConfig::Ptr config = KGlobal::config();
    KConfigGroup cg(config, objectName());

    QPoint pos, defpos(10, 10);
    pos = cg.readEntry("ori", defpos);
    setGeometry(pos.x(), pos.y(), cg.readEntry("wdt",460),
                cg.readEntry("hgt",190));

    (cg.readEntry("vis", false)) ? show() : hide();

    readColumns();
}

/*
 * Save current settings
 */
void KFibsPlayerList::saveConfig()
{
    KSharedConfig::Ptr config = KGlobal::config();
    KConfigGroup cg(config, objectName());

    cg.writeEntry("ori", pos());
    cg.writeEntry("hgt", height());
    cg.writeEntry("wdt", width());

    cg.writeEntry("vis", isVisible());

    for (int i = 0; i < LVEnd; i++) {
        cg.writeEntry("col-" + d->mCol[i]->key, d->mCol[i]->show);
        cg.writeEntry("col-w-" + d->mCol[i]->key,
                           (d->mCol[i]->show) ? columnWidth(d->mCol[i]->index) : -1);
    }
}


// == popup menu slots and functions ===========================================

/*
 * Save selected player, update the menu entries and show the popup menu
 */
void KFibsPlayerList::showContextMenu(K3ListView *, Q3ListViewItem *i, const QPoint &p)
{
    /*
     * Get the name of the selected player
     */
    d->mUser = (i ? i->text(Player) : QString());

    d->mAct[KFibsPlayerListPrivate::Info  ]->setText(i18n("Info on %1" , d->mUser));
    d->mAct[KFibsPlayerListPrivate::Talk  ]->setText(i18n("Talk to %1" , d->mUser));
    d->mAct[KFibsPlayerListPrivate::Mail  ]->setText(i18n("Email to %1", d->mUser));
    d->mAct[KFibsPlayerListPrivate::Look  ]->setText(i18n("Look at %1" , d->mUser));
    d->mAct[KFibsPlayerListPrivate::Watch ]->setText(i18n("Watch %1"   , d->mUser));
    d->mAct[KFibsPlayerListPrivate::Update]->setText(i18n("Update %1"  , d->mUser));

    d->mAct[KFibsPlayerListPrivate::Info    ]->setEnabled(i);
    d->mAct[KFibsPlayerListPrivate::Talk    ]->setEnabled(i);
    d->mAct[KFibsPlayerListPrivate::Mail    ]->setEnabled(i);
    d->mAct[KFibsPlayerListPrivate::Look    ]->setEnabled(i);
    d->mAct[KFibsPlayerListPrivate::Watch   ]->setEnabled(i);
    d->mAct[KFibsPlayerListPrivate::Update  ]->setEnabled(i);
    d->mAct[KFibsPlayerListPrivate::BlindAct]->setEnabled(i);

    d->mAct[KFibsPlayerListPrivate::Unwatch]->setEnabled(d->mWatch);

    d->mInAct->setEnabled(i && d->mName != d->mUser);
    d->mInAct->setText(i18n("Invite %1", d->mUser));

    d->mMail = (i && d->mCol[Email]->show ? i->text(d->mCol[Email]->index) : QString());
    d->mAct[KFibsPlayerListPrivate::Mail]->setEnabled(!d->mMail.isEmpty());

    if (i && d->mCol[Status]->show)
        d->mAct[KFibsPlayerListPrivate::BlindAct]->setText
            ((i->text(d->mCol[Status]->index).contains(d->mAbrv[Blind])) ?
             i18n("Unblind %1", d->mUser) : i18n("Blind %1", d->mUser));
    else
        d->mAct[KFibsPlayerListPrivate::BlindAct]->setText(i18n("Blind"));

    // show the menu
    d->mPm[0]->popup(p);
}

/*
 * Reload the entire list
 */
void KFibsPlayerList::slotReload()
{
    emit fibsCommand("rawwho");
    clear();
}

/*
 * Stop watching
 */
void KFibsPlayerList::slotUnwatch()
{
    emit fibsCommand("unwatch");
}

/*
 * Blind/Unblind user
 */
void KFibsPlayerList::slotBlind()
{
    emit fibsCommand("blind " + d->mUser);
}

/*
 * Start talking to user
 */
void KFibsPlayerList::slotTalk()
{
    emit fibsTalk(d->mUser);
}

/*
 * Request information on user
 */
void KFibsPlayerList::slotInfo()
{
    emit fibsCommand("whois " + d->mUser);
}

/*
 * Look at user
 */
void KFibsPlayerList::slotLook()
{
    emit fibsCommand("look " + d->mUser);
}

/*
 * Send an email to player user at address email
 */
void KFibsPlayerList::slotMail()
{
    KToolInvocation::invokeMailer(d->mMail, QString());
}

/*
 * Request a new entry for user
 */
void KFibsPlayerList::slotUpdate()
{
    emit fibsCommand("rawwho " + d->mUser);
}

/*
 * Watch user and get an updated board
 */
void KFibsPlayerList::slotWatch()
{
    emit fibsCommand("watch " + d->mUser);
    emit fibsCommand("board");
}

/*
 * Request information about the selected user
 */
void KFibsPlayerList::getPlayerInfo(Q3ListViewItem *i, const QPoint &, int col)
{
    int num = cIndex(col);
    if (col < 0 || num < 0 || num > 2 || i->text(num).isEmpty())
        num = 0;
    emit fibsCommand("whois " + i->text(num));
}

/*
 * Invite the selected user.
 */
void KFibsPlayerList::slotInviteD()
{
    emit fibsInvite(d->mUser);
}
void KFibsPlayerList::slotInvite1() { emit fibsCommand("invite " + d->mUser + " 1"); }
void KFibsPlayerList::slotInvite2() { emit fibsCommand("invite " + d->mUser + " 2"); }
void KFibsPlayerList::slotInvite3() { emit fibsCommand("invite " + d->mUser + " 3"); }
void KFibsPlayerList::slotInvite4() { emit fibsCommand("invite " + d->mUser + " 4"); }
void KFibsPlayerList::slotInvite5() { emit fibsCommand("invite " + d->mUser + " 5"); }
void KFibsPlayerList::slotInvite6() { emit fibsCommand("invite " + d->mUser + " 6"); }
void KFibsPlayerList::slotInvite7() { emit fibsCommand("invite " + d->mUser + " 7"); }

void KFibsPlayerList::slotInviteU() { emit fibsCommand("invite " + d->mUser + " unlimited"); }
void KFibsPlayerList::slotInviteR() { emit fibsCommand("invite " + d->mUser); }


// == inserting and updating the list ==========================================

/*
 * Add or change the entry of player with the corresponding string
 * from the server - rawwho
 */
void KFibsPlayerList::changePlayer(const QString &line)
{
    char entry[LVEnd][100];
    char ready[2], away[2];
    Q3ListViewItem *i;
    QDateTime fromEpoch;
    QString str_entry[LVEnd], tmp;

    entry[Status][0] = '\0';

    // the line comes from FIBS and is 7 bit ASCII
    sscanf(line.toLatin1(), "%99s %99s %99s %1s %1s %99s %99s %99s %99s %99s %99s %99s", entry[Player], entry[Opponent],
           entry[Watches], ready, away, entry[Rating], entry[Experience], entry[Idle], entry[Time],
           entry[Host], entry[Client], entry[Email]);

    // convert time
    tmp = entry[Time];
    fromEpoch.setTime_t(tmp.toUInt());
    strcpy(entry[Time], fromEpoch.toString().toLatin1());

    // clear empty strings and copy
    for (int j = 0; j < LVEnd; j++) {
        if ((str_entry[j] = entry[j]) == "-")
            str_entry[j] = "";
    }
    str_entry[Status].replace(Ready, 1, ready[0] == '0' ? "-" : d->mAbrv[Ready]);
    str_entry[Status].replace(Away,  1, away [0] == '0' ? "-" : d->mAbrv[Away ]);
    str_entry[Status].replace(Blind, 1, "-");

    // disable drawing until the end of update
    setUpdatesEnabled(false);

    // try to find the item in the list
    Q3ListViewItemIterator it(this);
    for ( ; it.current(); ++it) {
        if (it.current()->text(0) == str_entry[Player]) {
            i = it.current();
            goto found;
        }
    }

    // getting here means we have to create a new entry
    i = new KFibsPlayerListLVI(this);

    // count the KFibs and KBackgammon clients
    if (str_entry[Client].contains("KFibs"))
        d->mCount[0]++;
    else if (str_entry[Client].contains(PROG_NAME))
        d->mCount[1]++;

    // new entry requires an update to the player count
    updateCaption();

    goto update;

 found:

    // getting here means the player is in the list - update private status
    str_entry[Status].replace(Blind,1,i->text(Status).contains
                              (d->mAbrv[Blind]) ? d->mAbrv[Blind] : "-");

 update:

    for (int j = 0; j < LVEnd; j++) {
        if (d->mCol[j]->show)
            i->setText(d->mCol[j]->index, str_entry[j]);
    }

    // find out if we are watching somebody
    if (d->mName == str_entry[Player])
        d->mWatch = !str_entry[Watches].isEmpty();
}

/*
 * Remove player from the list
 */
void KFibsPlayerList::deletePlayer(const QString &player)
{
    Q3ListViewItemIterator it(this);
    for ( ; it.current(); ++it) {
        if (it.current()->text(0) == player) {
            if (it.current()->text(Client).contains(PROG_NAME))
                --d->mCount[1];
            else if (it.current()->text(Client).contains("KFibs"))
                --d->mCount[0];
            delete it.current();
            updateCaption();
            return;
        }
    }
}

/*
 * Set/Unset the status stat in the corresponding column of the list
 */
void KFibsPlayerList::changePlayerStatus(const QString &player, int stat, bool flag)
{
    Q3ListViewItem *i = 0;

    /*
     * Find the correct line
     */
    Q3ListViewItemIterator it(this);
    for ( ; it.current(); ++it) {
        if (it.current()->text(Player) == player) {
            i = it.current();
            break;
        }
    }
    if (!i)	return;

    /*
     * Update the status flag
     */
    i->setText(Status, i->text(Status).replace(stat, 1, (flag) ? d->mAbrv[stat] : "-"));
}


// == various slots and functions ==============================================

/*
 * Reverse column to index mapping. Return negative on error.
 */
int KFibsPlayerList::cIndex(int col)
{
    for (int i = 0; i < LVEnd; i++)
        if (d->mCol[i]->index == col)
            return i;
    return -1;
}

/*
 * Catch hide events, so the engine's menu can be update.
 */
void KFibsPlayerList::showEvent(QShowEvent *e)
{
    K3ListView::showEvent(e);
    emit windowVisible(true);
}

/*
 * Catch hide events, so the engine's menu can be update.
 */
void KFibsPlayerList::hideEvent(QHideEvent *e)
{
    emit windowVisible(false);
    K3ListView::hideEvent(e);
}

/*
 * Called at the end of updates to re-enable the UI
 */
void KFibsPlayerList::stopUpdate()
{
    setUpdatesEnabled(true);
    triggerUpdate();
}

/*
 * Knowing our own name allows us to disable certain menu entries for
 * ourselves.
 */
void KFibsPlayerList::setName(const QString &name)
{
    d->mName = name;
}

/*
 * Update the caption of the list by including the current client
 * count
 */
void KFibsPlayerList::updateCaption()
{
    setWindowTitle(i18n("Player List - %1 - %2/%3", childCount(), d->mCount[0], d->mCount[1]));
}

/*
 * Clear the list and reset the client counters
 */
void KFibsPlayerList::clear()
{
    d->mCount[0] = 0;
    d->mCount[1] = 0;
    Q3ListView::clear();
}

// EOF
