/***************************************************************************
 *   Copyright (C) 2005--2006 by Stefan Kebekus                            *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   Copyright (C) 2005 by Wilfried Huss                                   *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <kaction.h>
#include <kactioncollection.h>
#include <kdialog.h>
#include <kfiledialog.h>
#include <kparts/genericfactory.h>
#include <kprinter.h>
#include <kmessagebox.h>
#include <kselectaction.h>

#include "kvsprefs.h"
#include <qapplication.h>
#include <q3paintdevicemetrics.h>
#include <QPrinter>
//Added by qt3to4:
#include <QTemporaryFile>
#include <QToolTip>

#include "ByteStream.h"
#include "DjVuToPS.h"
#include "kprintDialogPage_DJVUpageoptions.h"
#include "kprintDialogPage_DJVUconversionoptions.h"
#include "kvs_debug.h"
#include "djvumultipage.h"
#include "pageInsertionWidget.h"
#include "pageRangeWidget.h"
#include "prefs.h"


K_EXPORT_COMPONENT_FACTORY( ligaturePlugin_DJVU, KGenericFactory<DjVuMultiPage>( "ligaturePlugin_DJVU" ) )


DjVuMultiPage::DjVuMultiPage(QObject *parent, const QStringList&)
  : ligaturePluginGUI(parent),
    djvuRenderer(this)
{
  // Render modes
  QStringList renderModes;
  renderModes.append(i18n("Color"));
  renderModes.append(i18n("Black and White"));
  renderModes.append(i18n("Show foreground only"));
  renderModes.append(i18n("Show background only"));
  renderModeAction = actionCollection()->add<KSelectAction>("render_mode");
  renderModeAction->setText(i18n("Render Mode"));
  renderModeAction->setItems(renderModes);

  renderModeAction->setCurrentItem(Prefs::renderMode());

  insertPagesAction = actionCollection()->addAction("insert_pages");
  insertPagesAction->setText(i18n("Insert Pages..."));
  connect(insertPagesAction, SIGNAL(triggered(bool)), this, SLOT(slotInsertPages()));
  deletePagesAction = actionCollection()->addAction("delete_pages");
  deletePagesAction->setText(i18n("Delete Pages..."));
  connect(deletePagesAction, SIGNAL(triggered(bool)), this, SLOT(slotDeletePages()));

  // change the rendermode
  connect(renderModeAction, SIGNAL(activated(int)), this, SLOT(setRenderMode(int)));

  /* It is very important that this method is called in the
     constructor. Otherwise ligaturePluginGUI does not know how to render
     files, and crashes may result. */
  setRenderer(&djvuRenderer);

  setXMLFile("djvumultipage.rc");

  enableActions(false);
}


DjVuMultiPage::~DjVuMultiPage()
{
  ;
}


KAboutData* DjVuMultiPage::createAboutData()
{
  /* You obviously want to change this to match your setup */
  KAboutData* about = new KAboutData("djvumultipage", I18N_NOOP("KDjView"), "0.1",
                          I18N_NOOP("Ligature DjVu Plugin."),
                          KAboutData::License_GPL,
                          "Wilfried Huss",
                          I18N_NOOP("This program displays DjVu files."));

  about->addAuthor ("Stefan Kebekus",
                    I18N_NOOP("Ligature plugin"),
                    "kebekus@kde.org",
                    "http://www.mi.uni-koeln.de/~kebekus");

  about->addAuthor ("Wilfried Huss",
                    I18N_NOOP("DjVu file loading"),
                    "Wilfried.Huss@gmx.at");

  return about;
}

void DjVuMultiPage::enableActions(bool b)
{
  ligaturePluginGUI::enableActions(b);

  insertPagesAction->setEnabled(b);
  deletePagesAction->setEnabled(b);
}

void DjVuMultiPage::setFile(bool r)
{
  enableActions(r);
}

void DjVuMultiPage::setRenderMode(int mode)
{
  // Save renderMode for future uses
  switch (mode)
  {
    case Prefs::EnumRenderMode::BlackAndWhite:
      Prefs::setRenderMode(Prefs::EnumRenderMode::BlackAndWhite);

      break;
    case Prefs::EnumRenderMode::Foreground:
      Prefs::setRenderMode(Prefs::EnumRenderMode::Foreground);

      break;
    case Prefs::EnumRenderMode::Background:
      Prefs::setRenderMode(Prefs::EnumRenderMode::Background);

      break;
    default:  //Prefs::EnumRenderMode::Color
      Prefs::setRenderMode(Prefs::EnumRenderMode::Color);
  }
  Prefs::writeConfig();
  emit renderModeChanged();
}


void DjVuMultiPage::slotDeletePages()
{
  if (dataModel->numberOfPages() == 0)
    return;

  KDialog dialog( parentWdg);
  dialog.setCaption( i18n("Delete Pages") );
  dialog.setButtons( KDialog::Ok|KDialog::Cancel );
  dialog.setDefaultButton( KDialog::Ok );

  PageRangeWidget range( 1, dataModel->numberOfPages(), dataModel->currentPageNumber(), &dialog);
  range.setToolTip( i18n( "Select the pages you wish to delete." ) );
  dialog.setButtonText(KDialog::Ok, i18n("Delete Pages"));
  dialog.setMainWidget(&range);
  if (dialog.exec() != QDialog::Accepted)
    return;

  djvuRenderer.deletePages(range.getFrom(), range.getTo());
  dataModel->deletePages(range.getFrom(), range.getTo());
  emit documentHasBeenModified();
}


void DjVuMultiPage::slotInsertPages()
{
  // Inserting into an empty document doesn't make much sense
  if (dataModel->numberOfPages() == 0)
    return;

  // Obtain file name for insertion
  QString filter = i18n("*.djvu|DjVu file (*.djvu)");
  QString fname = KFileDialog::getOpenFileName(KUrl("kfiledialog:///djvu?global"), filter, parentWdg, i18n("Insert DJVU file..."));
  if (fname.isEmpty())
    return;

  // Show dialog, so user can choose where to insert the file
  KDialog dialog( parentWdg);
  dialog.setCaption( i18n("Insert Pages...") );
  dialog.setModal( true );
  dialog.setButtons( KDialog::Ok|KDialog::Cancel );
  dialog.setDefaultButton( KDialog::Ok );
  PageInsertionWidget pgins( dataModel->currentPageNumber(), dataModel->numberOfPages(), fname, filter, &dialog);
  pgins.setToolTip( i18n("Select the file you wish to insert, and the plase where the file should be inserted."));
  dialog.setButtonText(KDialog::Ok, i18n("Insert Pages"));
  dialog.setMainWidget(&pgins);

  // Enable OK-button only if the selected file exists
  dialog.enableButtonOk(QFile::exists(pgins.getFile()));
  connect(&pgins, SIGNAL(have_valid_file(bool)), &dialog, SLOT(enableButtonOk(bool)));

  // Show dialog
  if (dialog.exec() != QDialog::Accepted)
    return;

  PageNumber oldNrPages = djvuRenderer.totalPages();
  djvuRenderer.insertPages(pgins.getFile(), pgins.getPage());

  int numInserted = djvuRenderer.totalPages() - oldNrPages;
  if (numInserted > 0)
    dataModel->insertPages(pgins.getPage(), numInserted);
  else {
    kError() << "Inserting pages did not increase the number of pages!" << endl;
    dataModel->setNumberOfPages(djvuRenderer.totalPages());
  }
  emit documentHasBeenModified();
}


void DjVuMultiPage::print()
{
  // Paranoid safety checks
  if (djvuRenderer.isEmpty())
    return;

  // Allocate the printer structure
  KPrinter *printer = getPrinter(false);
  if (printer == 0)
    return;

  KPrintDialogPage_DJVUPageOptions *pageOptions = new KPrintDialogPage_DJVUPageOptions();
  if (pageOptions == 0) {
    kError(kvs::djvu) << "DjVuMultiPage::print(): Cannot allocate new KPrintDialogPage_PageOptions structure" << endl;
    delete printer;
    return;
  }
  printer->addDialogPage( pageOptions );

  KPrintDialogPage_DJVUConversionOptions *conversionOptions = new KPrintDialogPage_DJVUConversionOptions();
  if (pageOptions == 0) {
    kError(kvs::djvu) << "DjVuMultiPage::print(): Cannot allocate new KPrintDialogPage_ConversionOptions structure" << endl;
    delete printer;
    return;
  }
  printer->addDialogPage( conversionOptions );

  // initialize the printer using the print dialog
  if ( printer->setup(parentWdg, i18n("Print %1", m_file.section('/', -1))) ) {
    // Now do the printing.
    QList<int> pageList = printer->pageList();
    if (pageList.isEmpty())
      printer->abort();
    else {
      // Printing usually takes a while. This is to keep the GUI
      // updated.
      qApp->processEvents();

      // Printing...
      DjVuToPS converter;
      DjVuToPS::Options &options = converter.options;

      // Set PostScript Language Level, taking 3 as the default
      options.set_format(DjVuToPS::Options::PS);
      QString op = printer->option( "kde-kdjvu-pslevel" );
      if (op == "1")
	options.set_level(1);
      else
	if (op == "3")
	  options.set_level(3);
	else
	  options.set_level(2);

      // Set page size orientation
      if (printer->option( "kde-ligature-rotatepage" ) == "true")
	options.set_orientation (DjVuToPS::Options::AUTO);
      else
	if ( printer->orientation() == KPrinter::Landscape )
	  options.set_orientation (DjVuToPS::Options::LANDSCAPE);
	else
	  options.set_orientation (DjVuToPS::Options::PORTRAIT);

      // Set render mode, taking "color" as default
      op = printer->option("kde-kdjvu-rendermode");
      if (op == "black-and-white")
	options.set_mode(DjVuToPS::Options::BW);
      else
	if (op == "foreground")
	  options.set_mode(DjVuToPS::Options::FORE);
	else
	  if (op == "background")
	    options.set_mode(DjVuToPS::Options::BACK);
	  else
	    options.set_mode(DjVuToPS::Options::COLOR);

      // Set Color or Grayscale mode
      if (printer->colorMode() == KPrinter::Color)
	options.set_color(true);
      else
	options.set_color(false);

      // Set Zoom
      if (printer->option( "kde-kdjvu-fitpage" ) == "true")
	options.set_zoom(0);
      else
	options.set_zoom(100);

      QTemporaryFile tmpPSFile;
      // Must open the QTemporaryFile to access the name.
      tmpPSFile.open();
      const QString tmpPSFileName = tmpPSFile.fileName();
      tmpPSFile.close();

      if (djvuRenderer.convertToPSFile(converter, tmpPSFileName, pageList ) == true)
	printer->printFiles( QStringList(tmpPSFileName), true );
      else
	printer->abort();
    }
    delete printer;
  }
}


bool DjVuMultiPage::isReadWrite() const
{
  return true;
}


bool DjVuMultiPage::isModified() const
{
  return djvuRenderer.isModified();
}


bool DjVuMultiPage::slotSave(const QString &fileName)
{
  // Paranoid safety checks
  if (djvuRenderer.isEmpty())
    return false;
  if (fileName.isEmpty())
    return false;

  bool r = djvuRenderer.save(fileName);
  if (r == false)
    KMessageBox::error(parentWdg, i18n("<qt>Error saving the document to the file <strong>%1</strong>. The document is <strong>not</strong> saved.</qt>", fileName),
		       i18n("Error saving document"));
  return r;
}



#include "djvumultipage.moc"
