/***************************************************************************
 *   Copyright (C) 2005--2006 by Stefan Kebekus                            *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <config.h>

#include <kmessagebox.h>
#include <klocale.h>
#include <qfileinfo.h>
#include <qpainter.h>

#include "documentWidget.h"
#include "faxrenderer.h"
#include "faxmultipage.h"
#include "kvs_debug.h"

//#define KF_DEBUG

FaxRenderer::FaxRenderer(ligaturePluginGUI* _multiPage)
  : DocumentRenderer(_multiPage), fax(0)
{
#ifdef KF_DEBUG
  kError(kvs::fax) << "FaxRenderer( parent=" << par << " )" << endl;
#endif
}



FaxRenderer::~FaxRenderer()
{
#ifdef KF_DEBUG
  kDebug(kvs::fax) << "~FaxRenderer" << endl;
#endif

  // Wait for all access to this documentRenderer to finish
  QMutexLocker locker(&mutex);
}


RenderedDocumentPagePixmap* FaxRenderer::drawPage(const JobId& id)
{
#ifdef KF_DEBUG
  kDebug(kvs::fax) << "FaxRenderer::drawPage(JobId) called, page number " << id.pageNumber << endl;
#endif

  // Paranoid safety checks
  if (!id.pageNumber.isValid()) {
    kDebug(kvs::fax) << "FaxRenderer::drawPage(JobId) called with a invalid page number" << endl;
    return 0;
  }

  // Wait for all access to this documentRenderer to finish
  QMutexLocker locker(&mutex);

  // more paranoid safety checks
  if (id.pageNumber > numPages) {
    kError(kvs::fax) << "FaxRenderer::drawPage(JobId) called for page number " << id.pageNumber
	      << " but the current fax file has only " << numPages << " pages." << endl;
    return 0;
  }

  RenderedDocumentPagePixmap* page = multiPage->createDocumentPagePixmap(id);

  double resolution = id.resolution;

  QImage img = getRawImage(id.pageNumber);
  if (img.isNull())
    return 0;

  SimplePageSize psize = pageSizes[id.pageNumber - 1];
  if (psize.isValid())
  {
    // Compute an image for the page.

    // WARNING: It may be tempting to compute the image size in
    // pixel, using page->height() and page->width(). DON'T DO
    // THAT. Ligature uses transformations e.g. to rotate the
    // page, and sets the argument 'resolution' accordingly. Similar
    // problems occur if Ligature required a shrunken version of
    // the page, e.g. to print multiple pages on one sheet of paper.

    int width_in_pixel = qRound(resolution * psize.width().getLength_in_inch());
    int height_in_pixel = qRound(resolution * psize.height().getLength_in_inch());

    img = img.scaled(width_in_pixel, height_in_pixel, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
    page->setImage(img);
    page->resize(width_in_pixel, height_in_pixel);
  }
  else
  {
    kError(kvs::fax) << "FaxRenderer::drawPage() called, but page size for page " << id.pageNumber << " is invalid." << endl;
    delete page;
    return 0;
  }

  // To indicate that the page was drawn, we set the appropriate flas in the page structure
  page->isEmpty = false;
  return page;
}


bool FaxRenderer::setFile(const QString &fname, const KUrl &)
{
#ifdef KF_DEBUG
  kDebug(kvs::fax) << "FaxRenderer::setFile(" << fname << ") called" << endl;
#endif

  // Wait for all access to this documentRenderer to finish
  QMutexLocker locker(&mutex);

  // If fname is the empty string, then this means: "close".
  if (fname.isEmpty())
  {
    if (fax)
    {
      TIFFClose(fax);
      fax = 0;
    }
    kDebug(kvs::fax) << "FaxRenderer::setFile( ... ) called with empty filename. Closing the file." << endl;
    return true;
  }

  // Paranoid saftey checks: make sure the file actually exists, and
  // that it is a file, not a directory. Otherwise, show an error
  // message and exit..
  QFileInfo fi(fname);
  QString   filename = fi.absoluteFilePath();
  if (!fi.exists() || fi.isDir()) {
    KMessageBox::error( parentWidget,
			i18n("<qt><strong>File error.</strong> The specified file '%1' does not exist.</qt>", filename),
			i18n("File Error"));
    // the return value 'false' indicates that this operation was not successful.
    return false;
  }

  // Now we assume that the file is fine and load the file into the
  // fax member. We abort on error and give an error message.
  fax = TIFFOpen(QFile::encodeName( filename ), "r");
  if (!fax)
  {
    KMessageBox::error(parentWidget,
      i18n("<qt><strong>File error.</strong> The specified file '%1' could not be loaded.</qt>", filename),
      i18n("File Error"));

    clear();
    return false;
  }

  // Set the number of pages page sizes
  tdir_t dirs = TIFFNumberOfDirectories(fax);
  numPages = dirs;

  // Set the page size for the first page in the pageSizes array.
  // The rest of the page sizes will be calculated on demand by the drawPage function.
  pageSizes.resize(numPages);
  Length w,h;

  if (numPages != 0)
  {
    for(tdir_t pg = 0; pg < dirs; pg++)
    {
      if (!TIFFSetDirectory(fax, pg))
        continue;

      QPoint dpi = getDPI(pg + 1);

      quint32 width = 0;
      quint32 height = 0;

      if (TIFFGetField(fax, TIFFTAG_IMAGEWIDTH, &width) != 1 ||
          TIFFGetField(fax, TIFFTAG_IMAGELENGTH, &height) != 1 )
        continue;

      w.setLength_in_inch(width / (double)dpi.x());
      h.setLength_in_inch(height / (double)dpi.y());
      pageSizes[pg].setPageSize(w, h);
    }
  }

  // the return value 'true' indicates that this operation was not successful.
  return true;
}


QImage FaxRenderer::getRawImage(PageNumber page)
{
  if (!TIFFSetDirectory(fax, page - 1))
    return QImage();

  quint32 width = 0;
  quint32 height = 0;

  if (TIFFGetField(fax, TIFFTAG_IMAGEWIDTH, &width) != 1 ||
      TIFFGetField(fax, TIFFTAG_IMAGELENGTH, &height) != 1 )
    return QImage();

  QImage img(width, height, 32);
  quint32* data = (quint32*)img.bits();

  if (TIFFReadRGBAImageOriented(fax, width, height, (uint32*)data, ORIENTATION_TOPLEFT) != 0)
  {
    quint32 size = width * height;
    for (quint32 i = 0; i < size; ++i)
    {
      Q_UINT32 red = (data[i] & 0x00FF0000) >> 16;
      Q_UINT32 blue = (data[i] & 0x000000FF) << 16;
      data[i] = (data[i] & 0xFF00FF00) + red + blue;
    }
  }
  else
  {
    return QImage();
  }

  return img;
}


QPoint FaxRenderer::getDPI(PageNumber page)
{
  if (!TIFFSetDirectory(fax, page-1))
    return QPoint(0, 0);

  float dpix = 0.0;
  float dpiy = 0.0;
  if (TIFFGetField(fax, TIFFTAG_XRESOLUTION, &dpix) != 1 ||
      TIFFGetField(fax, TIFFTAG_YRESOLUTION, &dpiy) != 1)
    return QPoint(0, 0);

  kDebug(kvs::fax) << "resolutions, dpi x = " << dpix << ", dpi y = "  << dpiy << "." << endl;

  if (dpix <= 1 || dpiy <= 1) {
    kError(kvs::fax) << "File invalid resolutions, dpi x = " << dpix << ", dpi y = "  << dpiy << ". This information will be ignored and 75 DPI assumed." << endl;
    dpix = dpiy = 75;
  }

  return QPoint((int)dpix, (int)dpiy);
}

#include "faxrenderer.moc"
