// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
//
// Class: documentPage
//
// Widget for displaying TeX DVI files.
// Part of KDVI- A previewer for TeX DVI files.
//
// (C) 2004-2006 Stefan Kebekus. 
// (C) 2005-2006 Wilfried Huss
//
// Distributed under the GPL.

#ifndef _rendereddocumentpage_h_
#define _rendereddocumentpage_h_

#include "ligature_export.h"

#include "pageNumber.h"

#include <QVector>
#include <QObject>
#include <QSize>
#include <QString>

class Hyperlink;
class QPainter;
class QPoint;
class QRect;
class QRegion;
class TextBox;
class TextSelection;

class KPDFPageTransition;


/** @short Represents a page in a document, contains the page's graphical
    representation and resolution-dependent information.

    This abstract class represents a page in a document. Its
    implementations contain graphical representations of document pages,
    and resolution-dependent information. The most important members are

    - the page number

    - a vector that contains hyperlinks

    - a vector for source hyperlinks; this vector is currently used by
      KDVI only, and is therefore not documented much here.

    - a vector that contains the document text together with
    information about the position of the text on the page. This
    information is used in ligature e.g. for selecting text with the
    mouse or for the "export as text..." functionality

    - the implementations of the documentPage will also contain a
    QPaintDevice, onto which the graphical representation of the page
    is rendered. This could for instance be a QPixmap, if the page is
    to be displayed on a computer screen, or a QPrinter for printing.

    The ligature program uses the documentPage class in the
    following manner: first, it is constructed, the page number is
    invalid. The page number is then set to a reasonable value using
    the setPageNumber() method. After this, the
    documentRenderer.drawPage() method of an implementation of the
    documentRenderer is used to render a graphical representation into
    the QPaintDevice at a given resolution, and to fill the
    (source-)hyperlink and text vectors, which are also
    resolution-dependent. After the data has been used, the
    documentPage is clear()ed, perhaps a new page number set, and
    documentRenderer.drawPage() is used again.

    @author Stefan Kebekus <kebekus@kde.org>
*/

class LIGATURECORE_EXPORT RenderedDocumentPage : public QObject
{
  Q_OBJECT

 public:
  /** \brief Default constructor

  The default constructor constructs a RenderedDocumentPage whose pageNumber
  is invalid. It sets the member 'isEmpty' to 'true'.
  */
  RenderedDocumentPage();

  virtual ~RenderedDocumentPage();

  /** \brief Set page number

  This method sets the number of the page that this instance
  represents. A value of '0' means that this class does not represent
  any page ('invalid page'). This method also calls clear().
  */
  virtual void setPageNumber(const PageNumber& pagenr);

  /** \brief Get page number

  @returns The number of the page that this instance represents.
  */
  PageNumber getPageNumber() const {return pageNr;}

  /** \brief Returns the size of the page in pixel. */
  virtual QSize size() = 0;

  /** \brief Returns the width of the page in pixel.

  @warning If you are implementing DocumentRenderer::drawPage(), it
  may be tempting to compute the image size in pixel, using
  page->height() and page->width(). DON'T DO THAT. Ligature uses
  transformations e.g. to rotate the page, and sets the argument
  'resolution' accordingly; these changes are not reflected in
  page->height() and page->width(). Similar problems occur if
  Ligature required a shrunken version of the page, e.g. to print
  multiple pages on one sheet of paper. See the documentation of
  DocumentRenderer::drawPage() to learn how to compute the sizes
  properly.
  */
  virtual int width() {return size().width(); }

  /** \brief Returns the height of the page in pixel.

  @warning If you are implementing DocumentRenderer::drawPage(), it
  may be tempting to compute the image size in pixel, using
  page->height() and page->width(). DON'T DO THAT. Ligature uses
  transformations e.g. to rotate the page, and sets the argument
  'resolution' accordingly; these changes are not reflected in
  page->height() and page->width(). Similar problems occur if
  Ligature required a shrunken version of the page, e.g. to print
  multiple pages on one sheet of paper. See the documentation of
  DocumentRenderer::drawPage() to learn how to compute the sizes
  properly.
  */
  virtual int height() {return size().height(); }

  /** \brief Text contained in the document page

  This vector contains the document text together with information
  about the position of the text on the page. This information is
  generated by the documentRenderer.drawPage() method. It is used in
  ligature e.g. for selecting text with the mouse or for the "export
  as text..." functionality.
  */
  QVector<TextBox> textBoxList;

  /** \brief Calculates the text selected by the given rectangle.
      The parameter @param normal selects which mode that is used
      for calculating the selection. See selection.svg for an explanation
      of the two different modes. */
  TextSelection select(const QRect&, bool normal = true);

  /** \brief Selects the character which lies at the given point. */
  TextSelection select(const QPoint&);

  /** \brief Selects the text inside the given rectangle.
      This functions differs from @ref select(), because it does not
      try to compute the actual layout of the text.
      @returns a string of all characters inside the given rectangle,
      even if it is not continuous text. */
  QString rectangleSelect(const QRect&) const;

  QRegion selectedRegion(const TextSelection& selection);

  /** Finds the first occurence of str starting by the current index.
  If the text is found a corresponding TextSelection is returned.
  If the text is not found a empty selection is returned. */
  TextSelection find(const QString& str, int index = 0, bool caseSensitive = true);

  /** Finds the first occurence of str starting by the current index searching backwards.
  If the text is found a corresponding TextSelection is returned.
  If the text is not found a empty selection is returned.
  If index < 0 start the search at the end of the page. */
  TextSelection findRev(const QString& str, int index = 0, bool caseSensitive = true);

  /** \brief Hyperlinks on the document page

  This vector contains the hyperlinks that appear on the page. This
  information is generated by the documentRenderer.drawPage()
  method. It is used in ligature so users can use the mouse to
  navigate in the document through hyperlinks.
  */
  QVector<Hyperlink> hyperLinkList;

  // set to 'false' in the constructor, set to 'true' by the renderer,
  // if something really has been rendered
  bool isEmpty;

  /** \brief Clears the data structures

  Clears the contents of the class, but leaves pageNumber intact. For
  performance reasons, it does not free the memory for the
  QValueVectors so that lengthy re-allocations won't be necessary
  later.
  */
  virtual void clear();

  /** \brief Transition effect of the page

  Returns the transition effect of the page. 0 if no transition is
  specified for this page.
  */
  KPDFPageTransition* transition();

  void setTransition(KPDFPageTransition* pageTransition);

  /** Indicates if hyperlinks are drawn
      
  Some document formats, e.g. PDF, specify how hyperlinks appear on
  screen, and the hyperlink is drawn by the documentRenderer. In that
  case, the documentRenderer sets this member to 'true'. Ligature
  will then not attempt to mark the hyperlink itself.
  
  This member is set to 'false' in the constructor.
  */
  bool hyperlinks_are_marked;

 private:
  /** \brief Page number

  Number of the page that this instance represents a value of '0'
  means that this class does not represent any page.
  */
  PageNumber  pageNr;

  KPDFPageTransition* _transition;

  QString pageText;
};


#endif
