/* This file is part of the KDE libraries
    Copyright (C) 1997 Martin Jones (mjones@kde.org)

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/
//-----------------------------------------------------------------------------
// Selector widgets for KDE Color Selector, but probably useful for other
// stuff also.

#ifndef KSELECTOR_H
#define KSELECTOR_H

#include <QAbstractSlider>
#include <QWidget>

#include <kdelibs_export.h>

/**
 * KSelector is the base class for other widgets which
 * provides the ability to choose from a one-dimensional
 * range of values. An example is the KGradientSelector
 * which allows to choose from a range of colors.
 *
 * A custom drawing routine for the widget surface has
 * to be provided by the subclass.
 */
class KDEUI_EXPORT KSelector : public QAbstractSlider
{
  Q_OBJECT
  Q_PROPERTY( int value READ value WRITE setValue )
  Q_PROPERTY( int minValue READ minimum WRITE setMinimum )
  Q_PROPERTY( int maxValue READ maximum WRITE setMaximum )
public:

  /**
   * Constructs a horizontal one-dimensional selection widget.
   */
  explicit KSelector( QWidget *parent=0 );
  /**
   * Constructs a one-dimensional selection widget with
   * a given orientation.
   */
  explicit KSelector( Qt::Orientation o, QWidget *parent = 0 );
  /*
   * Destructs the widget.
   */
  ~KSelector();

  /**
   * @return the rectangle on which subclasses should draw.
   */
  QRect contentsRect() const;

  /**
   * Sets the indent option of the widget to i.
   * This determines whether a shaded frame is drawn.
   */
  void setIndent( bool i );

  /**
   * @return whether the indent option is set.
   */
  bool indent() const;

protected:
  /**
   * Override this function to draw the contents of the control.
   * The default implementation does nothing.
   *
   * Draw only within contentsRect().
   */
  virtual void drawContents( QPainter * );
  /**
   * Override this function to draw the cursor which
   * indicates the current value.
   */
  virtual void drawArrow( QPainter *painter, const QPoint &pos );

  virtual void paintEvent( QPaintEvent * );
  virtual void mousePressEvent( QMouseEvent *e );
  virtual void mouseMoveEvent( QMouseEvent *e );
  virtual void mouseReleaseEvent( QMouseEvent *e );
  virtual void wheelEvent( QWheelEvent * );

private:
  QPoint calcArrowPos( int val );
  void moveArrow( const QPoint &pos );

private:
  class Private;
  Private * const d;
};


/**
 * The KGradientSelector widget allows the user to choose
 * from a one-dimensional range of colors which is given as a
 * gradient between two colors provided by the programmer.
 *
 * \image html kgradientselector.png "KDE Gradient Selector Widget"
 *
 **/
class KDEUI_EXPORT KGradientSelector : public KSelector
{
  Q_OBJECT

  Q_PROPERTY( QColor firstColor READ firstColor WRITE setFirstColor )
  Q_PROPERTY( QColor secondColor READ secondColor WRITE setSecondColor )
  Q_PROPERTY( QString firstText READ firstText WRITE setFirstText )
  Q_PROPERTY( QString secondText READ secondText WRITE setSecondText )

public:
  /**
   * Constructs a horizontal color selector which
   * contains a gradient between white and black.
   */
  explicit KGradientSelector( QWidget *parent=0 );
  /**
   * Constructs a colors selector with orientation o which
   * contains a gradient between white and black.
   */
  KGradientSelector( Qt::Orientation o, QWidget *parent=0 );
  /**
   * Destructs the widget.
   */
  ~KGradientSelector();
  /**
   * Sets the two colors which span the gradient.
   */
  void setColors( const QColor &col1, const QColor &col2 )
  {	color1 = col1; color2 = col2; update();}
  void setText( const QString &t1, const QString &t2 )
  {	text1 = t1; text2 = t2; update(); }

  /**
   * Set each color on its own.
   */
  void setFirstColor( const QColor &col )
  { color1 = col; update(); }
  void setSecondColor( const QColor &col )
  { color2 = col; update(); }

  /**
   * Set each description on its own
   */
  void setFirstText( const QString &t )
  { text1 = t; update(); }
  void setSecondText( const QString &t )
  { text2 = t; update(); }

  const QColor firstColor() const
  { return color1; }
  const QColor secondColor() const
  { return color2; }

  const QString firstText() const
  { return text1; }
  const QString secondText() const
  { return text2; }

protected:

  virtual void drawContents( QPainter * );
  virtual QSize minimumSize() const
  { return sizeHint(); }

private:
  void init();
  QColor color1;
  QColor color2;
  QString text1;
  QString text2;

private:
  class KGradientSelectorPrivate;
  KGradientSelectorPrivate *d;
};

#endif		// KSELECTOR_H
