// -*- c-basic-offset: 2 -*-

/* This file is part of the KDE libraries
   Copyright (C) 2000 Kurt Granroth <granroth@kde.org>
   Copyright (C) 2006 Hamish Rodda <rodda@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <kanimatedbutton.h>

#include <QAction>
#include <QPixmap>
#include <QTimer>
#include <QImage>
#include <QToolBar>
#include <QPainter>

#include <kdebug.h>
#include <kiconloader.h>

class KAnimatedButtonPrivate
{
public:
  int                    frames;
  int                    current_frame;
  QPixmap                pixmap;
  QTimer                 timer;
  QString                icon_name;
};

KAnimatedButton::KAnimatedButton( QWidget *parent )
  : QToolButton( parent ),
    d( new KAnimatedButtonPrivate )
{
  connect( &d->timer, SIGNAL(timeout()), this, SLOT(slotTimerUpdate()));
}

KAnimatedButton::~KAnimatedButton()
{
  d->timer.stop();

  delete d;
}

void KAnimatedButton::start()
{
  d->current_frame = 0;
  d->timer.start( 50 );
}

void KAnimatedButton::stop()
{
  d->current_frame = 0;
  d->timer.stop();
  updateCurrentIcon();
}

void KAnimatedButton::setIcons( const QString& icons )
{
  if ( d->icon_name == icons )
    return;

  d->icon_name = icons;
  updateIcons();
}

QString KAnimatedButton::icons( ) const
{
   return d->icon_name;
}

void KAnimatedButton::slotTimerUpdate()
{
  if(!isVisible())
    return;

  d->current_frame++;
  if (d->current_frame == d->frames)
     d->current_frame = 0;

  updateCurrentIcon();
}

void KAnimatedButton::updateCurrentIcon( )
{
  if ( d->pixmap.isNull() )
    return;

  int w = d->pixmap.width();
  int h = w;

  QPixmap pix(w, h);

  {
    QPainter p(&pix);
    p.drawPixmap(QPoint(0,0), d->pixmap, QRect(0, d->current_frame*h, w, h));
  }

  setIcon(QIcon(pix));
}

void KAnimatedButton::updateIcons()
{
  QString path = KIconLoader::global()->iconPath(d->icon_name, -iconDimensions());
  QImage img(path);

  if (img.isNull())
     return;

  d->current_frame = 0;
  d->frames = img.height() / img.width();
  if (d->pixmap.width() != iconDimensions())
  {
     img = img.scaled(iconDimensions(), iconDimensions()*d->frames, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
  }
  d->pixmap = QPixmap::fromImage(img);

  updateCurrentIcon();
}

int KAnimatedButton::iconDimensions() const
{
  return qMin(iconSize().width(), iconSize().height());
}

#include "kanimatedbutton.moc"
