/* This file is part of the KDE libraries
    Copyright (C) 1998 Mark Donohoe <donohoe@kde.org>
    Copyright (C) 2001 Ellis Whitehead <ellis@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "kkeybutton.h"
#include "kshortcutdialog.h"

#include <qcursor.h>
#include <qdrawutil.h>
#include <qpainter.h>
#include <QPolygon>
#include <QStyle>

#include <kdebug.h>
#include <kglobalaccel.h>
#include <klocale.h>

#ifdef Q_WS_X11
#define XK_XKB_KEYS
#define XK_MISCELLANY
#include <X11/Xlib.h>	// For x11Event()
#include <X11/keysymdef.h> // For XK_...

#ifdef KeyPress
const int XFocusOut = FocusOut;
const int XFocusIn = FocusIn;
const int XKeyPress = KeyPress;
const int XKeyRelease = KeyRelease;
#undef KeyRelease
#undef KeyPress
#undef FocusOut
#undef FocusIn
#endif // KeyPress
#endif // Q_WS_X11

//static const char* psTemp[] = {
//  I18N_NOOP("Primary"), I18N_NOOP("Alternate"), I18N_NOOP("Multi-Key")
//};

/***********************************************************************/
/* KKeyButton                                                          */
/*                                                                     */
/* Initially added by Mark Donohoe <donohoe@kde.org>                   */
/*                                                                     */
/***********************************************************************/

class KKeyButton::KKeyButtonPrivate
{
public:
	KKeyButtonPrivate(KKeyButton *q): q(q) {}
  
	KKeyButton *q;
	KShortcut m_cut;
	bool m_bEditing;
};

KKeyButton::KKeyButton(QWidget *parent)
:	QPushButton( parent ), d(new KKeyButtonPrivate(this))
{
	setFocusPolicy( Qt::StrongFocus );
	d->m_bEditing = false;
	connect( this, SIGNAL(clicked()), this, SLOT(captureShortcut()) );
	setShortcut( KShortcut() );
}

KKeyButton::~KKeyButton ()
{
	delete d;
}

const KShortcut& KKeyButton::shortcut() const
{
	return d->m_cut;
}

void KKeyButton::setShortcut( const KShortcut& cut )
{
	d->m_cut = cut;
	QString keyStr = d->m_cut.toString();
	keyStr.replace(QLatin1Char('&'), QLatin1String("&&"));
	setText( keyStr.isEmpty() ? i18n("None") : keyStr );
}

void KKeyButton::setText( const QString& text )
{
	QPushButton::setText( text );
	setFixedSize( sizeHint().width()+12, sizeHint().height()+8 );
}

void KKeyButton::captureShortcut()
{
	KShortcut cut;

	d->m_bEditing = true;
	repaint();

	{
		KShortcutDialog dlg( d->m_cut, this );
		if( dlg.exec() == KDialog::Accepted )
			cut = dlg.shortcut();
	} // emit the signal after the dialog is destroyed, otherwise it still has grab

	if( !cut.isEmpty() )
		emit capturedShortcut( cut );

	d->m_bEditing = false;
	repaint();
}

void KKeyButton::paintEvent( QPaintEvent* )
{
  QPainter painter(this);

  QPolygon a( 4 );
  a.setPoint( 0, 0, 0) ;
  a.setPoint( 1, width(), 0 );
  a.setPoint( 2, 0, height() );
  a.setPoint( 3, 0, 0 );

  QRegion r1( a );
  painter.setClipRegion( r1 );
  painter.setBrush( palette().color( QPalette::Background ).light() );
  painter.drawRoundRect( 0, 0, width(), height(), 20, 20);

  a.setPoint( 0, width(), height() );
  a.setPoint( 1, width(), 0 );
  a.setPoint( 2, 0, height() );
  a.setPoint( 3, width(), height() );

  QRegion r2( a );
  painter.setClipRegion( r2 );
  painter.setBrush( palette().color( QPalette::Background ).dark() );
  painter.drawRoundRect( 0, 0, width(), height(), 20, 20 );

  painter.setClipping( false );
  if( width() > 12 && height() > 8 )
    qDrawShadePanel( &painter, 6, 4, width() - 12, height() - 8,
                     palette(), true, 1, 0L );
  if ( d->m_bEditing )
  {
    painter.setPen( palette().color( QPalette::Base ) );
    painter.setBrush( palette().color( QPalette::Base ) );
  }
  else
  {
    painter.setPen( palette().color( QPalette::Background ) );
    painter.setBrush( palette().color( QPalette::Background ) );
  }
  if( width() > 14 && height() > 10 )
    painter.drawRect( 7, 5, width() - 14, height() - 10 );

  painter.setPen( palette().color( QPalette::Text ) );

  style()->drawItemText( &painter, rect(), Qt::AlignCenter | Qt::TextShowMnemonic,
                         palette(), isEnabled(), text() );

  painter.setBrush( Qt::NoBrush );
  if( hasFocus() || d->m_bEditing )
  {
    if( width() > 16 && height() > 12 )
      painter.drawRect( 8, 6, width() - 16, height() - 12 );
  }

}


#include "kkeybutton.moc"
