/*  This file is part of the KDE project
    Copyright (C) 2006 Kevin Ottens <ervin@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.

*/

#ifndef SOLID_OPTICALDISC_H
#define SOLID_OPTICALDISC_H

#include <kdelibs_export.h>

#include <solid/volume.h>

namespace Solid
{
    /**
     * This capability is available on optical discs.
     *
     * An optical disc is a volume that can be inserted in a cdrom drive.
     */
    class SOLID_EXPORT OpticalDisc : public Volume
    {
        Q_OBJECT
        Q_ENUMS( ContentType DiscType )
        Q_FLAGS( ContentTypes )
        Q_PROPERTY( ContentTypes availableContent READ availableContent )
        Q_PROPERTY( DiscType discType READ discType )
        Q_PROPERTY( bool appendable READ isAppendable )
        Q_PROPERTY( bool blank READ isBlank )
        Q_PROPERTY( bool rewritable READ isRewritable )
        Q_PROPERTY( qulonglong capacity READ capacity )

    public:
        /**
         * This enum type defines the type of content available in an optical disc.
         *
         * - Audio : A disc containing audio
         * - Data : A disc containing data
         * - VideoCd : A Video Compact Disc (VCD)
         * - SuperVideoCd : A Super Video Compact Disc (SVCD)
         * - VideoDvd : A Video Digital Versatile Disc (DVD-Video)
         */
        enum ContentType { Audio = 0x01, Data = 0x02, VideoCd = 0x04, SuperVideoCd = 0x08, VideoDvd = 0x10 };

        /**
         * This type stores an OR combination of ContentType values.
         */
        Q_DECLARE_FLAGS( ContentTypes, ContentType )

        /**
         * This enum type defines the type of optical disc it can be.
         *
         * - CdRom : A Compact Disc Read-Only Memory (CD-ROM)
         * - CdRecordable : A Compact Disc Recordable (CD-R)
         * - CdRewritable : A Compact Disc ReWritable (CD-RW)
         * - DvdRom : A Digital Versatile Disc Read-Only Memory (DVD-ROM)
         * - DvdRam : A Digital Versatile Disc Random Access Memory (DVD-RAM)
         * - DvdRecordable : A Digital Versatile Disc Recordable (DVD-R)
         * - DvdRewritable : A Digital Versatile Disc ReWritable (DVD-RW)
         * - DvdPlusRecordable : A Digital Versatile Disc Recordable (DVD+R)
         * - DvdPlusRewritable : A Digital Versatile Disc ReWritable (DVD+RW)
         * - UnknownDiscType : An undetermined disc type
         */
        enum DiscType { CdRom, CdRecordable, CdRewritable, DvdRom, DvdRam,
                        DvdRecordable, DvdRewritable,
                        DvdPlusRecordable, DvdPlusRewritable, UnknownDiscType };


        /**
         * Creates a new OpticalDisc object.
         * You generally won't need this. It's created when necessary using
         * Device::as().
         *
         * @param backendObject the capability object provided by the backend
         * @see Solid::Device::as()
         */
        explicit OpticalDisc( QObject *backendObject );

        /**
         * Destroys an OpticalDisc object.
         */
        virtual ~OpticalDisc();


        /**
         * Get the Solid::Capability::Type of the OpticalDisc capability.
         *
         * @return the OpticalDisc capability type
         * @see Solid::Ifaces::Enums::Capability::Type
         */
        static Type capabilityType() { return Capability::OpticalDisc; }


        /**
         * Retrieves the content types this disc contains (audio, video,
         * data...).
         *
         * @return the flag set indicating the available contents
         */
        ContentTypes availableContent() const;

        /**
         * Retrieves the disc type (cdr, cdrw...).
         *
         * @return the disc type
         */
        DiscType discType() const;

        /**
         * Indicates if it's possible to write additional data to the disc.
         *
         * @return true if the disc is appendable, false otherwise
         */
        bool isAppendable() const;

        /**
         * Indicates if the disc is blank.
         *
         * @return true if the disc is blank, false otherwise
         */
        bool isBlank() const;

        /**
         * Indicates if the disc is rewritable.
         *
         * A disc is rewritable if you can write on it several times.
         *
         * @return true if the disc is rewritable, false otherwise
         */
        bool isRewritable() const;

        /**
         * Retrieves the disc capacity (that is the maximum size of a
         * volume could have on this disc).
         *
         * @return the capacity of the disc in bytes
         */
        qulonglong capacity() const;
    };
}

Q_DECLARE_OPERATORS_FOR_FLAGS( Solid::OpticalDisc::ContentTypes )

#endif
