/* This file is part of the KDE libraries
    Copyright (c) 2006 David Faure <faure@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "qtest_kde.h"
#include "kmainwindow_unittest.h"
#include "kmainwindow_unittest.moc"
#include <kmainwindow.h>
#include <kglobal.h>
#include <QResizeEvent>
#include <ktoolbar.h>

QTEST_KDEMAIN( KMainWindow_UnitTest, GUI )

void KMainWindow_UnitTest::testDefaultName()
{
    KMainWindow mw;
    mw.show();
    mw.ensurePolished();
    QCOMPARE( mw.objectName(), QString::fromLatin1( "qttest/MainWindow_1" ) );
    KMainWindow mw2;
    mw2.show();
    mw2.ensurePolished();
    QCOMPARE( mw2.objectName(), QString::fromLatin1( "qttest/MainWindow_2" ) );
}

void KMainWindow_UnitTest::testFixedName()
{
    KMainWindow mw;
    mw.setObjectName( "mymainwindow" );
    mw.show();
    mw.ensurePolished();
    QCOMPARE( mw.objectName(), QString::fromLatin1( "qttest/mymainwindow" ) );
    KMainWindow mw2;
    mw2.setObjectName( "mymainwindow" );
    mw2.show();
    mw2.ensurePolished();
    QCOMPARE( mw2.objectName(), QString::fromLatin1( "qttest/mymainwindow2" ) );
}

void KMainWindow_UnitTest::testNameWithHash()
{
    KMainWindow mw;
    mw.setObjectName( "composer_#" );
    mw.show();
    mw.ensurePolished();
    QCOMPARE( mw.objectName(), QString::fromLatin1( "qttest/composer_1" ) );
    KMainWindow mw2;
    mw2.setObjectName( "composer_#" );
    mw2.show();
    mw2.ensurePolished();
    QCOMPARE( mw2.objectName(), QString::fromLatin1( "qttest/composer_2" ) );
}

void KMainWindow_UnitTest::testNameWithSpecialChars()
{
    KMainWindow mw;
    mw.setObjectName( "a#@_test/" );
    mw.show();
    mw.ensurePolished();
    QCOMPARE( mw.objectName(), QString::fromLatin1( "qttest/a___test_" ) );
    KMainWindow mw2;
    mw2.setObjectName( "a#@_test/" );
    mw2.show();
    mw2.ensurePolished();
    QCOMPARE( mw2.objectName(), QString::fromLatin1( "qttest/a___test_2" ) );
}

static bool s_mainWindowDeleted;
class MyMainWindow : public KMainWindow
{
public:
    MyMainWindow() : KMainWindow(),
                     m_queryClosedCalled( false ),
                     m_queryExitCalled( false )
    {
    }
    /*reimp*/ bool queryClose() {
        m_queryClosedCalled = true;
        return true;
    }
    /*reimp*/ bool queryExit() {
        m_queryExitCalled = true;
        return true;
    }
    ~MyMainWindow() {
        s_mainWindowDeleted = true;
    }
    bool m_queryClosedCalled;
    bool m_queryExitCalled;
};

// Here we test
// - that queryClose is called
// - that queryExit is called
// - that autodeletion happens
void KMainWindow_UnitTest::testDeleteOnClose()
{
    KGlobal::ref(); // don't let the deref in KMainWindow quit the app.
    s_mainWindowDeleted = false;
    MyMainWindow* mw = new MyMainWindow;
    QVERIFY( mw->testAttribute( Qt::WA_DeleteOnClose ) );
    mw->close();
    QVERIFY( mw->m_queryClosedCalled );
    QVERIFY( mw->m_queryExitCalled );
    qApp->sendPostedEvents( mw, QEvent::DeferredDelete );
    QVERIFY( s_mainWindowDeleted );
}

void KMainWindow_UnitTest::testSaveWindowSize()
{
    KMainWindow mw;
    KToolBar* tb = new KToolBar(&mw); // we need a toolbar to trigger an old bug in saveMainWindowSettings
    tb->setObjectName("testtb");
    mw.resize(800, 600);

    // Send the pending resize event (resize() only sets Qt::WA_PendingResizeEvent)
    QResizeEvent e(mw.size(), QSize());
    QApplication::sendEvent(&mw, &e);

    KConfigGroup cfg(KGlobal::config(), "TestWindowSize");
    mw.saveMainWindowSettings(cfg);
    mw.close();

    KMainWindow mw2;
    tb = new KToolBar(&mw2);
    tb->setObjectName("testtb");
    mw2.resize(50, 50);
    mw2.applyMainWindowSettings(cfg);
    QCOMPARE(mw2.size(), QSize(800, 600));
}

void KMainWindow_UnitTest::testAutoSaveSettings()
{
    KMainWindow mw;
    KToolBar* tb = new KToolBar(&mw); // we need a toolbar to trigger an old bug in saveMainWindowSettings
    tb->setObjectName("testtb");
    const QString group("AutoSaveTestGroup");
    mw.setAutoSaveSettings(group);
    mw.resize(800, 600);

    // Send the pending resize event (resize() only sets Qt::WA_PendingResizeEvent)
    QResizeEvent e(mw.size(), QSize());
    QApplication::sendEvent(&mw, &e);

    mw.close();

    KMainWindow mw2;
    tb = new KToolBar(&mw2);
    tb->setObjectName("testtb");
    mw2.setAutoSaveSettings(group);
    QCOMPARE(mw2.size(), QSize(800, 600));
}
