/*
 *  This file is part of the KDE libraries
 *  Copyright (c) 2001 Michael Goffioul <kdeprint@swing.be>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License version 2 as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 **/

#include "editentrydialog.h"

#include <QtGui/QLineEdit>
#include <QtGui/QCheckBox>
#include <QtGui/QSpinBox>
#include <QtGui/QComboBox>
#include <QtGui/QLabel>
#include <Qt3Support/Q3Header>
#include <k3listview.h>
#include <QtGui/QLayout>
#include <QtGui/QStackedWidget>
#include <klocale.h>
#include <kiconloader.h>

EditEntryDialog::EditEntryDialog(PrintcapEntry *entry, QWidget *parent, const char *name)
        : KDialog(parent)
{
    setObjectName(name);
    setModal(true);
    setButtons(Ok | Cancel);

    QWidget *w = new QWidget(this);
    setMainWidget(w);

    QLabel *lab0 = new QLabel(i18n("Aliases:"), w);
    m_aliases = new QLineEdit(w);
    m_view = new K3ListView(w);
    m_view->addColumn("");
    m_view->header()->hide();
    m_type = new QComboBox(w);
    m_type->addItem(i18n("String"));
    m_type->addItem(i18n("Number"));
    m_type->addItem(i18n("Boolean"));
    m_stack = new QStackedWidget(w);
    m_boolean = new QCheckBox(i18n("Enabled"), m_stack);
    m_string = new QLineEdit(m_stack);
    m_number = new QSpinBox(m_stack);
    m_number->setRange(0, 9999);
    m_number->setSingleStep(1);
    m_stack->insertWidget(0, m_string);
    m_stack->insertWidget(2, m_boolean);
    m_stack->insertWidget(1, m_number);
    m_name = new QLineEdit(w);

    QVBoxLayout *l0 = new QVBoxLayout(w);
    l0->setMargin(0);
    l0->setSpacing(10);
    QHBoxLayout *l1 = new QHBoxLayout();
    l1->setMargin(0);
    l1->setSpacing(10);
    QHBoxLayout *l2 = new QHBoxLayout();
    l2->setMargin(0);
    l2->setSpacing(5);
    l0->addLayout(l1);
    l1->addWidget(lab0);
    l1->addWidget(m_aliases);
    l0->addWidget(m_view);
    l0->addLayout(l2);
    l2->addWidget(m_name, 0);
    l2->addWidget(m_type, 0);
    l2->addWidget(m_stack, 1);

    if (entry) {
        setCaption(i18n("Printcap Entry: %1", entry->name));
        m_fields = entry->fields;
        m_aliases->setText(entry->aliases.join("|"));
        Q3ListViewItem *root = new Q3ListViewItem(m_view, entry->name), *item = 0;
        root->setSelectable(false);
        root->setOpen(true);
        root->setPixmap(0, SmallIcon("document-print"));
        for (QMap<QString, Field>::ConstIterator it = m_fields.begin(); it != m_fields.end(); ++it)
            item = new Q3ListViewItem(root, item, (*it).toString(), it.key());
    }

    m_block = true;
    enableButton(Ok, false);
    slotItemSelected(NULL);
    slotTypeChanged(0);
    m_block = false;

    connect(m_view, SIGNAL(selectionChanged(Q3ListViewItem*)), SLOT(slotItemSelected(Q3ListViewItem*)));
    connect(m_string, SIGNAL(textChanged(const QString&)), SLOT(slotChanged()));
    connect(m_boolean, SIGNAL(toggled(bool)), SLOT(slotChanged()));
    connect(m_number, SIGNAL(valueChanged(int)), SLOT(slotChanged()));
    connect(m_type, SIGNAL(activated(int)), SLOT(slotTypeChanged(int)));
    connect(m_name, SIGNAL(textChanged(const QString&)), SLOT(slotChanged()));

    resize(500, 400);
}

Field EditEntryDialog::createField()
{
    Field f;
    f.name = m_name->text();
    f.type = (Field::Type)(m_type->currentIndex());
    switch (f.type) {
    case Field::String: f.value = m_string->text(); break;
    case Field::Integer: f.value = m_number->cleanText(); break;
    case Field::Boolean: f.value = (m_boolean->isChecked() ? "1" : "0"); break;
    }
    return f;
}

void EditEntryDialog::slotChanged()
{
    if (!m_block && m_view->currentItem()) {
        Field f = createField();
        if (f.name != m_current)
            m_fields.remove(m_current);
        m_fields[f.name] = f;
        m_view->currentItem()->setText(0, f.toString());
    }
}

void EditEntryDialog::slotItemSelected(Q3ListViewItem *item)
{
    m_stack->setEnabled(item);
    m_name->setEnabled(item);
    m_type->setEnabled(item);
    if (item) {
        m_block = true;
        m_current = item->text(1);
        Field f = m_fields[m_current];
        m_name->setText(f.name);
        m_type->setCurrentIndex(f.type);
        slotTypeChanged(f.type);
        m_string->setText(f.value);
        m_number->setValue(f.value.toInt());
        m_boolean->setChecked(f.value.toInt() == 1);
        m_block = false;
    }
}

void EditEntryDialog::fillEntry(PrintcapEntry *entry)
{
    entry->aliases = m_aliases->text().split('|', QString::SkipEmptyParts);
    entry->fields = m_fields;
}

void EditEntryDialog::slotTypeChanged(int ID)
{
    m_stack->setCurrentIndex(ID);
    slotChanged();
}

#include "editentrydialog.moc"
