/* This file is part of Noatun

  Copyright 2000-2006 by Charles Samuels <charles@kde.org>
  Copyright 2003-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#ifndef NOATUNPLAYLISTSAVER_H
#define NOATUNPLAYLISTSAVER_H

#include <kurl.h>
#include <qmap.h>
#include <qtextstream.h>

#include <noatun/playlist.h>
#include <noatun_export.h>

class QFile;

class NoatunXMLStructure;
class MSASXStructure;

namespace Noatun
{

/**
 * @short A generic playlist file loader/saver
 *
 * It supports the following formats:
 * <ul>
 * <li>Noatun XML playlist format, @see http://noatun.kde.org/doc/playlistformat.php</li>
 * <li>Normal and extended M3U as written by XMMS/WinAmp</li>
 * <li>PLS, common playlist format for listing shoutcast streams, not used much in other places</li>
 * <li>ASX, XMLish playlist format written by Windows Media Player</li>
 * </ul>
 *
 * @see PlaylistType
 * @todo Support XSPF playlists (http://www.xspf.org/)
 * @todo Support any useful/sane format found on http://gonze.com/playlists/playlist-format-survey.html
 * @todo Clean up and use mimetypes for determining what format to load/save
 **/
class NOATUN_EXPORT PlaylistSaver
{
public:
	/**
	 * Known playlist formats
	 **/
	enum PlaylistType
	{
		Default=0, /// Use a sane default for loading/saving
		XMLPlaylist=1, /// Save/Load as Noatun XML playlist
		M3U=2, /// Load/Save as simple M3U
		PLS=4, /// Load as PLS, very common for shoutcast-streams, saving as PLS is not supported
		EXTM3U=8, /// Load/Save as extended M3U, also contains title and length information
		ASX=16 /// Load as Windows Media Player playlist, saving as ASX is not supported (yet)
	};

	/**
	 *  Provides a list of supported mimetypes
	 **/
	static QStringList mimetypes();

	/**
	 * Provides a list of supported file extensions,
	 * useful if you cannot determine the mimetype of a file
	 **/
	static QStringList extensions();

	PlaylistSaver();
	virtual ~PlaylistSaver();

	/**
	 * Saves a playlist file to @p url
	 * if @p opt is set to Default, save() will save a Noatun XML playlist.
	 **/
	bool save(const KUrl &url, const PlaylistType opt = Default);

	/**
	 * Loads a playlist file from @p url
	 * if @p opt is set to Default, load() will try to detect the playlist format.
	 **/
	bool load(const KUrl &url, const PlaylistType opt = Default);

	/**
	 * read the item, and add it to your list
	 * Given is a list of properties which coincide
	 * with the standard noatun ones
	 **/
	virtual void readItem(const Noatun::PropertyMap &properties) = 0;

	/**
	 * add this item to the XML file
	 * or a null item if we're at the end
	 **/
	virtual PlaylistItem writeItem() = 0;

	/**
	 * this is called when you should either
	 * clear your list, or start writing from the start of the list
	 *
	 * You usually don't need to implement this, since it'l always
	 * be called immediately after load() or save()
	 **/
	virtual void reset() {}

private:
	bool saveLocal(QFile &file, const PlaylistType opt);

	/**
	 * guess the list's content between M3U or
	 * PLS, and give it to you
	 *
	 * this is also the way to make icecast work.
	 * you can also pass true icecast urls
	 * here and it'l do its magic.
	 *
	 * This calls readItem
	 **/
	bool metalist(const QString &file, const KUrl &originalUrl);

	bool loadXML(const QString &file, const PlaylistType x);
	bool loadASX(const QString &file, const KUrl &originalUrl, const PlaylistType opt = Default);
	bool loadM3U(const QString &file, const KUrl &originalUrl, const PlaylistType opt = Default);
	bool loadPLS(const QString &file, const PlaylistType x = Default);

	bool saveXML(QTextStream &stream);
	bool saveM3U(QTextStream &stream, PlaylistType opt = Default);

private:
	class Private;
	PlaylistSaver::Private *d; // unused

}; // END class PlaylistSaver

} // END namespace Noatun

#endif
