/* This file is part of Noatun

  Copyright 2000-2006 by Charles Samuels <charles@kde.org>
  Copyright 2000 by Nikolas Zimmermann <wildfox@kde.org>
  Copyright 2003-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#ifndef N_PLUGIN_H
#define N_PLUGIN_H

#include <kgenericfactory.h>
#include <kxmlguiclient.h>
#include <noatun_export.h>

class KPluginInfo;

namespace Noatun
{

class Interface;
class Global;
class PreferencesPage;
class PluginPrivate;

/**
 * @short Base class for all plugins
 **/
class NOATUN_EXPORT Plugin : public QObject, public KXMLGUIClient
{
Q_OBJECT
public:
	Plugin(const KComponentData &instance, Global *global, const char *name);
	virtual ~Plugin();

	/**
	 * called directly after the plugin was loaded, just in case
	 * you want Noatun to be "ready" with your class
	 **/
	virtual void init();

	QString pluginName() const;

	KPluginInfo *pluginInfo() const;
	const QStringList pluginInterfaces() const;

	virtual Interface *getInterface(const QString &name) = 0;

	/**
	 * Defines where actions ( @ref KAction ) added to this Plugin should show up
	 **/
	enum PluginPlace
	{
		/// Either this Plugin has no actions or it doesn't want them plugged into
		/// any other part
		PlugNowhere = 0,
		/// Plugin actions are plugged into mainwindow menubar/toolbar/popups
		PlugInFrontend,
		/// Plugin actions are plugged into playlist window menubar/toolbar/popups
		/// in case there exists a window
		PlugInPlaylist
	};

	Plugin::PluginPlace plugInto() const;
	void setPlugInto(PluginPlace where);

	Global *global();
	const Global *global() const;

	template <typename T>
	T *interface();

public Q_SLOTS:
	/**
	 * Called by Noatun when your plugin is unloaded.
	 * Use this for (async) cleanup actions and emit readyForUnload(this)
	 * after your plugin is safe to be unloaded.
	 * The default implementation immediately emits readyForUnload().
	 **/
	virtual void requestUnload();

Q_SIGNALS:
	void readyForUnload(Plugin *);

private Q_SLOTS:
	void slotNewFrontend();
	void slotNewPlaylist();

private:
	PluginPrivate * const d;
};


/**
 * @internal
 **/
class PluginFactoryBase : public KLibFactory
{
	Global     *mNoatunInstance;
	bool        m_catalogueInitialized;
	const char *m_instanceName;

public:
	explicit PluginFactoryBase(const char *componentName)
		: m_instanceName(componentName)
	{
		m_catalogueInitialized = false;
	}

	void setNoatunInstance(Global *nInstance)
	{
		mNoatunInstance = nInstance;
	}

	Global *global() { return mNoatunInstance; }

	virtual const KComponentData &componentData()=0;

	const char *componentName() const { return m_instanceName; }

protected:
	void initializeMessageCatalogue()
	{
		if (!m_catalogueInitialized)
		{
			m_catalogueInitialized = true;
			setupTranslations();
		}
	}

	virtual void setupTranslations()
	{
		if (componentData().isValid())
		{
			KGlobal::locale()->insertCatalog(
				QString::fromAscii(componentData().componentName())
			);
		}
	}
};


template<typename T>
class PluginFactory : public PluginFactoryBase
{
public:
	explicit PluginFactory(const char *componentName) : PluginFactoryBase(componentName)
	{
		//kDebug(66666) << k_funcinfo << "componentName = " << componentName << endl;
	}

	virtual QObject* createObject(
			QObject*,
			const char* name,
			const QStringList &
		)
	{
		this->initializeMessageCatalogue();
		return new T(componentData(), global(), name);
	}

protected:
	virtual const KComponentData &componentData()
	{
		if (!componentName())
		{
			kWarning(66666) << "Noatun::PluginFactory: instance requested but no "
				"instance name or about data passed to the constructor!" << endl;
			s_instance = new KComponentData();
		}
		if (!s_instance)
		{
			s_instance = new KComponentData(componentName());
		}
		return *s_instance;
	}

private:
	static KComponentData *s_instance;
};

template <typename T>
KComponentData *PluginFactory<T>::s_instance = 0;

} // namespace Noatun

#endif // N_PLUGIN_H
