/* This file is part of Noatun

  This file was created on 2005-05-23

  Copyright 2005-2007 by Stefan Gehn <mETz81@web.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/
#include "noatun_akode.h"

#include <noatun/player.h>
#include <noatun/global.h>

#include <akode/decoder.h>

#include <qcoreapplication.h>
#include <qevent.h>
#include <qfile.h>

using namespace Noatun;
using namespace std;


// ---------------------------------------------------------------------------

class AkodePluginEvent : public QEvent
{
public:
	enum Purpose
	{
		PurposeStateChange = 0,
		PurposeEOF         = 1,
		PurposeError       = 2
	};

	AkodePluginEvent(Purpose p, aKode::Player::State s = aKode::Player::Closed)
		: QEvent(QEvent::User), purpose(p), state(s)
	{
	}

	Purpose              purpose;
	aKode::Player::State state;
};


// ---------------------------------------------------------------------------


K_EXPORT_COMPONENT_FACTORY(noatun_akode, PluginFactory<AKodePlugin>("noatun_akode"))

AKodePlugin* AKodePlugin::s_self; // needed to access class from static functions


// ---------------------------------------------------------------------------


AKodePlugin::AKodePlugin(const KComponentData &instance, Global *parent, const char* name)
	: Plugin(instance, parent, name), EngineInterface(parent), mPlayer(0)
{
	s_self = this;
	kDebug(66666) << k_funcinfo << endl;
}

AKodePlugin::~AKodePlugin()
{
	kDebug(66666) << k_funcinfo << endl;
	delete mPlayer;
}

Interface *AKodePlugin::getInterface(const QString &interface)
{
	if (interface == "EngineInterface")
		return this;
	return 0;
}

void AKodePlugin::init()
{
	kDebug(66666) << k_funcinfo << endl;
	mPlayer = new aKode::Player();
	if (mPlayer->open("auto"))
	{
		kDebug(66666) << "Successfully initialized aKode" << endl;
		mPlayer->setManager(this);
	}
	else
	{
		kError(66666) << "Could not initialize aKode!" << endl;
		delete mPlayer;
		mPlayer = 0;
	}
}

void AKodePlugin::requestUnload()
{
	kDebug(66666) << k_funcinfo << endl;
	stop();
	emit readyForUnload(this);
}


// ----------------------------------------------------------------------------
// EngineInterface implementation:


Noatun::Player::State AKodePlugin::convertState(aKode::Player::State s)
{
	if (s == aKode::Player::Playing)
		return Noatun::Player::PlayingState;
	else if (s == aKode::Player::Paused)
		return Noatun::Player::PausedState;
	return Noatun::Player::StoppedState;
}


Player::State AKodePlugin::state() const
{
	if (mPlayer)
		return convertState(mPlayer->state());
	return Noatun::Player::StoppedState;
}


int AKodePlugin::position() const
{
	if (mPlayer)
	{
		aKode::Decoder *dec = mPlayer->decoder();
		if (dec)
			return dec->position();
	}
	return -1;
}


int AKodePlugin::length() const
{
	if (mPlayer)
	{
		aKode::Decoder *dec = mPlayer->decoder();
		if (dec)
			return dec->length();
	}
	return -1;
}


QStringList AKodePlugin::mimeTypes() const
{
	QSet<QString> types;
	types << "audio/x-wav"; // built into akode

	// aKode lacks a way to determine supported mimetypes :(
	const list<string> decoderList = aKode::DecoderPluginHandler::listDecoderPlugins();
	for (list<string>::const_iterator i = decoderList.begin(); i != decoderList.end(); i++)
	{
		if (*i == "mpeg")
		{
			types << "audio/x-mp2";
			types << "audio/x-mp3";
		}
		else if (*i == "mpc")
		{
			types << "audio/x-musepack";
		}
		else if (*i == "xiph")
		{
			//FIXME: we cannot know which formats the xiph decoder supports :(
			types << "application/ogg";
			types << "audio/x-flac";
			types << "audio/x-speex";
		}
		else if (*i == "ffmpeg")
		{
			types << "audio/vnd.rn-realaudio";
			types << "audio/x-pn-realaudio";
			types << "audio/x-flac";
			types << "audio/x-oggflac";
			types << "audio/x-speex";
			types << "audio/mp4";
			types << "audio/ac3";
			types << "audio/aac";
			types << "audio/vorbis";
			types << "audio/x-mp3";
			types << "audio/x-mp1";
			types << "audio/x-mp2";
			types << "audio/mpeg";
			types << "audio/x-ms-wma";
		}
	}

	return types.toList();
}


unsigned int AKodePlugin::softwareVolume() const
{
	if (mPlayer)
		return (unsigned int)(100.00 * mPlayer->volume() + 0.5);
	return 0u;
}


void AKodePlugin::setSoftwareVolume(unsigned int percent)
{
	if (mPlayer)
		mPlayer->setVolume(percent * 0.01);
}


bool AKodePlugin::play(const KUrl &url)
{
	kDebug(66666) << k_funcinfo << "asked to play " << url << endl;
	if (mPlayer)
	{
		if (url.isEmpty() && state() == Noatun::Player::PausedState)
		{
			kDebug(66666) << k_funcinfo << "Unpausing..." << endl;
			mPlayer->resume();
			return true;
		}

		if (!url.isLocalFile())
		{
			kDebug(66666) << k_funcinfo << "Cannot handle remote files yet" << endl;
			return false;
		}

		mPlayer->stop();
		if (mPlayer->load(QFile::encodeName(url.path())))
		{
			mPlayer->play();
			//playbackStarted();
			return true;
		}
	}
	kWarning(66666) << "Cannot play " << url << endl;
	return false;
}


void AKodePlugin::pause()
{
	kDebug(66666) << k_funcinfo << endl;
	if (mPlayer)
		mPlayer->pause();
}


void AKodePlugin::stop()
{
	kDebug(66666) << k_funcinfo << endl;
	if (mPlayer)
	{
		mPlayer->stop();
		mPlayer->unload(); // free resources
	}
}


void AKodePlugin::setPosition(int msec)
{
	if (mPlayer && msec >= 0)
	{
		kDebug(66666) << k_funcinfo << "msec = " << msec << endl;
		mPlayer->decoder()->seek(msec);
	}
}


void AKodePlugin::customEvent(QEvent *_e)
{
	AkodePluginEvent *e = dynamic_cast<AkodePluginEvent*>(_e);
	if (!e)
		return;
	if (e->purpose == AkodePluginEvent::PurposeEOF)
	{
		kDebug(66666) << k_funcinfo << "end of file" << endl;
		stop();
		EngineInterface::playbackFinished();
	}
	else if (e->purpose == AkodePluginEvent::PurposeError)
	{
		kDebug(66666) << k_funcinfo << "error" << endl;
		stop();
		//playbackFailed();
		EngineInterface::errorOccurred(Player::NormalError, QString::null);
	}
	else if (e->purpose == AkodePluginEvent::PurposeStateChange)
	{
		EngineInterface::stateChanged(convertState(e->state));
	}
}


// ---------------------------------------------------------------------------


void AKodePlugin::stateChangeEvent(aKode::Player::State newState)
{
	QCoreApplication::postEvent(self(), new AkodePluginEvent(AkodePluginEvent::PurposeStateChange, newState));
}


void AKodePlugin::eofEvent()
{
	QCoreApplication::postEvent(self(), new AkodePluginEvent(AkodePluginEvent::PurposeEOF));
}


void AKodePlugin::errorEvent()
{
	QCoreApplication::postEvent(self(), new AkodePluginEvent(AkodePluginEvent::PurposeError));
}


#include "noatun_akode.moc"
