/*  This file is part of the KDE project
    Copyright (C) 2006 Matthias Kretz <kretz@kde.org>
    Copyright (C) 2006-2007 Tim Beaulen <tbscope@gmail.com>

    Based on code by Fluendo:
    Copyright (C) 2006 Wim Taymans <wim@fluendo.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "controlitem.h"

#include <QString>
#include <QByteArray>
#include <QGraphicsScene>
#include <QRectF>
#include <QFontMetrics>
#include <QFont>
#include <QPixmap>
#include <QPixmapCache>
#include <QImage>
#include <QPainterPath>

#include <kdebug.h>

namespace Phonon
{
namespace GStreamer
{

QPixmap ControlItem::cached(const QString &img)
{
    if (QPixmap *p = QPixmapCache::find(img))
        return *p;

    QPixmap pm;
    pm = QPixmap::fromImage(QImage(img), Qt::OrderedDither | Qt::OrderedAlphaDither);
    if (pm.isNull())
        return QPixmap();

    QPixmapCache::insert(img, pm);
    return pm;
}

ControlItem::ControlItem(QGraphicsView *control)
{
    m_paused = false;
}

QRectF ControlItem::boundingRect() const
{
    QFont font("times", 14);
    QFontMetrics fm(font);

    QRectF f;

    f.setWidth(fm.boundingRect("This is a test to control play and pause!").width() + 50);
    f.setHeight(fm.boundingRect("This is a test to control play and pause!").height() + 50);
 
    f.adjust(-3,-3,3,3);

    return f;
}

void ControlItem::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    QRect r = boundingRect().toRect();

    painter->save();

    painter->setRenderHint(QPainter::Antialiasing);

    int radius = 14;
    int radius2 = radius*2;
    QPainterPath clipPath;
    clipPath.moveTo(radius, 0);
    clipPath.arcTo(r.right() - radius2, 0, radius2, radius2, 90, -90);
    clipPath.arcTo(r.right() - radius2, r.bottom() - radius2, radius2, radius2, 0, -90);
    clipPath.arcTo(r.left(), r.bottom() - radius2, radius2, radius2, 270, -90);
    clipPath.arcTo(r.left(), r.top(), radius2, radius2, 180, -90);
    painter->setClipPath(clipPath);
    QPixmap titleStretch = cached(":/images/title_stretch.png");
    QPixmap topLeft = cached(":/images/groupframe_topleft.png");
    QPixmap topRight = cached(":/images/groupframe_topright.png");
    QPixmap bottomLeft = cached(":/images/groupframe_bottom_left.png");
    QPixmap bottomRight = cached(":/images/groupframe_bottom_right.png");
    QPixmap leftStretch = cached(":/images/groupframe_left_stretch.png");
    QPixmap topStretch = cached(":/images/groupframe_top_stretch.png");
    QPixmap rightStretch = cached(":/images/groupframe_right_stretch.png");
    QPixmap bottomStretch = cached(":/images/groupframe_bottom_stretch.png");
    QPixmap play = cached(":/images/player_play.png");
    QPixmap pause = cached(":/images/player_pause.png");
    QPixmap save = cached(":/images/filesaveas.png");
    QLinearGradient lg(0, 0, 0, r.height());
    lg.setColorAt(0, QColor(224,224,224));
    lg.setColorAt(1, QColor(255,255,255));
    painter->setPen(Qt::NoPen);
    painter->setBrush(lg);
    painter->drawRect(r.adjusted(0, titleStretch.height()/2, 0, 0));
    painter->setClipping(false);

    int topFrameOffset = titleStretch.height()/2 - 2;
    painter->drawPixmap(r.topLeft() + QPoint(0, topFrameOffset), topLeft);
    painter->drawPixmap(r.topRight() - QPoint(topRight.width()-1, 0)
                        + QPoint(0, topFrameOffset), topRight);
    painter->drawPixmap(r.bottomLeft() - QPoint(0, bottomLeft.height()-1), bottomLeft);
    painter->drawPixmap(r.bottomRight() - QPoint(bottomRight.width()-1,
                        bottomRight.height()-1), bottomRight);

    QRect left = r;
    left.setY(r.y() + topLeft.height() + topFrameOffset);
    left.setWidth(leftStretch.width());
    left.setHeight(r.height() - topLeft.height() - bottomLeft.height() - topFrameOffset);
    painter->drawTiledPixmap(left, leftStretch);

    QRect top = r;
    top.setX(r.x() + topLeft.width());
    top.setY(r.y() + topFrameOffset);
    top.setWidth(r.width() - topLeft.width() - topRight.width());
    top.setHeight(topLeft.height());
    painter->drawTiledPixmap(top, topStretch);

    QRect right = r;
    right.setX(r.right() - rightStretch.width()+1);
    right.setY(r.y() + topRight.height() + topFrameOffset);
    right.setWidth(rightStretch.width());
    right.setHeight(r.height() - topRight.height()
                    - bottomRight.height() - topFrameOffset);
    painter->drawTiledPixmap(right, rightStretch);

    QRect bottom = r;
    bottom.setX(r.x() + bottomLeft.width());
    bottom.setY(r.bottom() - bottomStretch.height()+1);
    bottom.setWidth(r.width() - bottomLeft.width() - bottomRight.width());
    bottom.setHeight(bottomLeft.height());
    painter->drawTiledPixmap(bottom, bottomStretch);

    QPoint p;
    p.setX(r.x() + topLeft.width());
    p.setY(r.y() + topFrameOffset*2 + topLeft.height());

    QPen pn;
    pn.setColor(Qt::blue);
    painter->setPen(pn);
    QFont font("arial", 14);
    painter->setFont(font);
    painter->drawText(p, "Phonon GStreamer Pipeline");

    if (m_paused) {
        QRect pauseSource = pause.rect();
        QRect pauseDest;
        pauseDest.setX(r.x() + topLeft.width());
        pauseDest.setY(r.bottom() - bottomStretch.height()*3 +1);
        pauseDest.setWidth(32);
        pauseDest.setHeight(32);
        painter->drawPixmap(pauseDest, pause, pauseSource);
        m_playRect = pauseDest;
    } else {
        QRect playSource = play.rect();
        QRect playDest;
        playDest.setX(r.x() + topLeft.width());
        playDest.setY(r.bottom() - bottomStretch.height()*3 +1);
        playDest.setWidth(32);
        playDest.setHeight(32);
        painter->drawPixmap(playDest, play, playSource);
        m_playRect = playDest;
    }

    QRect saveSource = save.rect();
    QRect saveDest;
    saveDest.setX(r.x() + topLeft.width() + m_playRect.width()*2);
    saveDest.setY(r.bottom() - bottomStretch.height()*3 +1);
    saveDest.setWidth(32);
    saveDest.setHeight(32);
    painter->drawPixmap(saveDest, save, saveSource);
    m_saveRect = saveDest;

    painter->restore();
}

void ControlItem::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    if (event->buttons() == Qt::LeftButton) {
        if (m_playRect.contains(event->pos().toPoint())) {
            m_paused = !m_paused;
            update(m_playRect);
        }
    }
}

}} // Phonon::GStreamer
