/*  This file is part of the QbtGStreamer project, a Qt GStreamer Wrapper
    Copyright (C) 2006 Tim Beaulen <tbscope@gmail.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "qbtgstreamerbin.h"

#include <QtDebug>

QbtGStreamerBin::QbtGStreamerBin(const QString &name)
    : d(new QbtGStreamerBinPrivate)
{
    if (name.isEmpty())
        setGstElement(gst_bin_new(NULL));
    else
        setGstElement(gst_bin_new(name.toLocal8Bit().data()));

    d->bin = GST_BIN(gstElement());
    m_object = GST_OBJECT(gstElement());
}

QbtGStreamerBin::~QbtGStreamerBin()
{
}

bool QbtGStreamerBin::addElement(QbtGStreamerElement *element)
{
    return gst_bin_add(d->bin, element->gstElement());
}

bool QbtGStreamerBin::removeElement(QbtGStreamerElement *element)
{
    return gst_bin_remove(d->bin, element->gstElement());
}

QbtGStreamerElement *QbtGStreamerBin::elementByName(const QString &name)
{
    GstElement *e = gst_bin_get_by_name(d->bin, name.toLocal8Bit().data());

    if (!e)
        return 0;

    QbtGStreamerElement *el = new QbtGStreamerElement;
    el->setGstElement(e);

    return el;
}

QbtGStreamerElement *QbtGStreamerBin::elementByNameRecurseUp(const QString &name)
{
    GstElement *e = gst_bin_get_by_name_recurse_up(d->bin, name.toLocal8Bit().data());

    if (!e)
        return 0;

    QbtGStreamerElement *el = new QbtGStreamerElement;
    el->setGstElement(e);

    return el;
}

QList<QbtGStreamerElement *> QbtGStreamerBin::elements()
{
   qDebug() << "QbtGStreamerBin::elements()";

   GstIterator *it = gst_bin_iterate_elements(d->bin);
   QbtGStreamerElement *el;
   QList<QbtGStreamerElement *> l;

   bool done = FALSE;
   gpointer item;

   while (!done) {
     switch (gst_iterator_next(it, &item)) {
       case GST_ITERATOR_OK:
         el = new QbtGStreamerElement;
         el->setGstElement(GST_ELEMENT(item));
         l << el;
         gst_object_unref(item);
         break;
       case GST_ITERATOR_RESYNC:
         //...rollback changes to items...
         gst_iterator_resync(it);
         break;
       case GST_ITERATOR_ERROR:
         //...wrong parameter were given...
         done = TRUE;
         break;
       case GST_ITERATOR_DONE:
         done = TRUE;
         break;
     }
   }
   gst_iterator_free(it);

   qDebug() << "Finished QbtGStreamerBin::elements()";

   return l;
}

GstBin *QbtGStreamerBin::gstBin()
{
    return d->bin;
}

void QbtGStreamerBin::setGstBin(GstBin *bin)
{
    d->bin = bin;
}

#include "qbtgstreamerbin.moc"


