/* NMM - Network-Integrated Multimedia Middleware
 *
 * Copyright (C) 2006
 *               Motama GmbH, Saarbruecken, Germany
 *
 * Maintainer: Bernhard Fuchshumer <fub@motama.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "audiopath.h"
#include "audiooutput.h"
#include "audioeffect.h"
#include "abstractaudiooutput.h"
#include "nmm/base/graph/GraphHandler.hpp"
#include <kdebug.h>

namespace Phonon
{
namespace nmm
{

	AudioPath::AudioPath( QObject* parent )
		: QObject( parent )
		, m_graphHandler( 0 )
		, m_branch(new NMM::Branch())
	{
		kDebug()<<k_funcinfo<<endl;
	}

	AudioPath::~AudioPath()
	{
		kDebug()<<k_funcinfo<<endl;
	}

	bool AudioPath::addOutput( QObject* audioOutput )
	{
		kDebug()<<k_funcinfo<<endl;
		Q_ASSERT( audioOutput );
		AbstractAudioOutput* aao = qobject_cast<AbstractAudioOutput*>( audioOutput );
		Q_ASSERT( aao );
		Q_ASSERT( !m_outputs.contains( aao ) );
		m_outputs.append( aao );


		AudioOutput *ao = qobject_cast<AudioOutput*>( aao );
		connect(ao, SIGNAL(endTrack()), SIGNAL(endTrack()));

		try
		{
			NMM::INode* sink=ao->sinkNode();
			m_branch->addSink(sink);
		}
		catch(NMM::Exception& e)
		{
			kError()<<"Could not add NMM audio sink to branch. Reason was:"<<e.getComment().c_str()<<endl;
		}

		if (m_graphHandler && ! m_branch->isInserted())
			internal_insertInto();
		return true;
	}

	bool AudioPath::removeOutput( QObject* audioOutput )
	{
		kDebug()<<k_funcinfo<<"\n\t WARNING NMM backend does not supprt removing outputs yet"<<endl;
		Q_ASSERT( audioOutput );
		AbstractAudioOutput* ao = qobject_cast<AbstractAudioOutput*>( audioOutput );
		Q_ASSERT( ao );
		Q_ASSERT( m_outputs.removeAll( ao ) > 0 );
		return true;
	}

	bool AudioPath::insertEffect( QObject* newEffect, QObject* insertBefore )
	{
		Q_ASSERT( newEffect );
		AudioEffect* ae = qobject_cast<AudioEffect*>( newEffect );
		Q_ASSERT( ae );
		AudioEffect* before = 0;
		if( insertBefore )
		{
			before = qobject_cast<AudioEffect*>( insertBefore );
			Q_ASSERT( before );
			if( !m_effects.contains( before ) )
				return false;
			m_effects.insert( m_effects.indexOf( before ), ae );
		}
		else
			m_effects.append( ae );

		return true;
	}

	bool AudioPath::removeEffect( QObject* effect )
	{
		Q_ASSERT( effect );
		AudioEffect* ae = qobject_cast<AudioEffect*>( effect );
		Q_ASSERT( ae );
		if( m_effects.removeAll( ae ) > 0 )
			return true;
		return false;
	}

	bool AudioPath::insertInto(NMM::GraphHandler* graph)
	{
		kDebug()<<k_funcinfo<<endl;
		m_graphHandler=graph;
		m_branch->setDemuxTag(graph->getAudioStreamName(0));
		
		if (m_outputs.empty())
			return true; //addition scheduled for later, we don't have outputs now
		return internal_insertInto();
	}

	bool AudioPath::internal_insertInto()
	{
		kDebug()<<k_funcinfo<<endl;

		try
		{
			m_graphHandler->addBranch(m_branch);
		}
		catch(NMM::Exception& e)
		{
			kError()<<"Could not add NMM audio branch. Reason was:"<<e.getComment().c_str()<<endl;
		}

	}
			
	int AudioPath::getBranchId()
	{
		return m_branch->getId();
	}

}
}

#include "audiopath.moc"
// vim: sw=4 ts=4 tw=80 noet
