/* This file is part of Noatun

  Copyright 2004-2007 by Stefan Gehn <mETz81@web.de>
  Copyright 2005-2006 by Charles Samuels <charles@kde.org>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include <noatun/global.h>

#include <noatun/config.h>
#include <noatun/frontend.h>
#include <noatun/player.h>
#include <noatun/playlist.h>
#include <noatun/plugininterfaces.h>
#include <noatun/pluginloader.h>

#include "preferencesdialog.h"

#include <qlocale.h>
#include <qtimer.h>

#include <kcomponentdata.h>
#include <kdebug.h>
#include <klocale.h>

namespace Noatun
{

class Global::Private
{
public:
	Private(const KComponentData &c) : componentData(c) { }

public:
	const KComponentData &componentData;
	Config               *config;
	PreferencesDialog    *preferencesDialog;
	PluginHandler        *pluginHandler;
	Playlist             *playlist;
	Frontend             *frontend;
	Player               *player;
};


// -----------------------------------------------------------------------------


Global::Global(QObject *parent, const KComponentData &componentData)
	: QObject(parent), d(new Global::Private(componentData))
{
	Q_ASSERT(parent);
	setObjectName("Noatun_Global");

	d->config            = 0L; // lazy init
	d->preferencesDialog = 0L; // lazy init
	d->pluginHandler     = new PluginHandler(this);
	d->playlist          = new Playlist(this);
	d->frontend          = new Frontend(this);
	d->player            = new Player(this);

	connect(d->pluginHandler, SIGNAL(allPluginsUnloaded()), this, SIGNAL(shutdown()));
}

Global::~Global()
{
	kDebug(66666) ;
	delete d->config;
	d->config = 0;
	delete d;
}

void Global::quit()
{
	kDebug(66666) << "Called, this will be the end of Noatun ;)";
	pluginHandler()->unloadAll();
}

Playlist *Global::playlist() const
{
	Q_ASSERT(d->playlist);
	return d->playlist;
}

Frontend *Global::frontend() const
{
	Q_ASSERT(d->frontend);
	return d->frontend;
}

Player *Global::player() const
{
	Q_ASSERT(d->player);
	return d->player;
}

PluginHandler *Global::pluginHandler() const
{
	Q_ASSERT(d->pluginHandler);
	return d->pluginHandler;
}

PreferencesDialog *Global::preferencesDialog()
{
	if (!d->preferencesDialog)
		d->preferencesDialog = new PreferencesDialog(this);
	return d->preferencesDialog;
}

Config *Global::config()
{
	if (!d->config)
		d->config = new Config(d->componentData);
	return d->config;
}



QString formatDuration(int duration)
{
	duration /= 1000; // convert from milliseconds to seconds
	int days    = duration / (60 * 60 * 24);
	int hours   = duration / (60 * 60) % 24;
	int minutes = duration / 60 % 60;
	int seconds = duration % 60;

	KLocale *loc = KGlobal::locale();
	const QChar zeroDigit = QLocale::system().zeroDigit();
	KLocalizedString str;

	if (days > 0)
	{
		QString dayStr = i18np("%1 day", "%1 days", days);
		str = ki18nc("<negativeSign><day(s)> <hours>:<minutes>:<seconds>",
			"%1%2 %3:%4:%5")
			.subs(duration < 0 ? loc->negativeSign() : QString())
			.subs(dayStr)
			.subs(hours, 2, 10, zeroDigit)
			.subs(minutes, 2, 10, zeroDigit)
			.subs(seconds, 2, 10, zeroDigit);
	}
	else if (hours > 0)
	{
		str = ki18nc("<negativeSign><hours>:<minutes>:<seconds>", "%1%2:%3:%4")
			.subs(duration < 0 ? loc->negativeSign() : QString())
			.subs(hours, 2, 10, zeroDigit)
			.subs(minutes, 2, 10, zeroDigit)
			.subs(seconds, 2, 10, zeroDigit);
	}
	else
	{
		str = ki18nc("<negativeSign><minutes>:<seconds>", "%1%2:%3")
			.subs(duration < 0 ? loc->negativeSign() : QString())
			.subs(minutes, 2, 10, zeroDigit)
			.subs(seconds, 2, 10, zeroDigit);
	}
	return str.toString();
}

} // namespace Noatun

#include "global.moc"
