/*
  This file is part of the KOrganizer alarm daemon.

  Copyright (c) 2000,2003 Cornelius Schumacher <schumacher@kde.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

  As a special exception, permission is given to link this program
  with any edition of Qt, and distribute the resulting executable,
  without including the source code for Qt in the source distribution.
*/

#include "alarmdialog.h"
#include "koeventviewer.h"
#include <korganizer_interface.h>

#include <kcal/event.h>

#include <Phonon/MediaObject>
#include <kiconloader.h>
#include <klocale.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <knotification.h>
#include <kcombobox.h>
#include <kwindowsystem.h>
#include <klockfile.h>
#include <kpushbutton.h>
#include <ktoolinvocation.h>
#include <kglobal.h>
#include <kvbox.h>

#include <QLabel>
#include <QFile>
#include <QSpinBox>
#include <QLayout>
#include <QPushButton>
#include <QDataStream>
#include <QVBoxLayout>
#include <QBoxLayout>
#include <QtDBus>
#include <QProcess>
#include "alarmdialog.moc"

AlarmDialog::AlarmDialog( QWidget *parent )
  : KDialog( parent/*, Qt::WType_TopLevel | Qt::WStyle_Customize | Qt::WStyle_StaysOnTop |
                     Qt::WStyle_DialogBorder,*/ ),
                 mSuspendTimer(this)
{
  KIconLoader::global()->addAppDir( "kdepim" );

  QWidget *topBox = new QWidget( this);
  setMainWidget( topBox );
  setCaption( i18n("Reminder") );
  setButtons( Ok | User1 | User2 );
  setDefaultButton( User1 );
  setButtonText( User1, i18n("Suspend") );
  setButtonText( User2, i18n("Edit...") );
  QBoxLayout *topLayout = new QVBoxLayout( topBox );
  topLayout->setSpacing( spacingHint() );

  QLabel *label = new QLabel( i18n("The following events triggered reminders:"),
                              topBox );
  topLayout->addWidget( label );

  mEventViewer = new KOEventViewer( topBox );
  topLayout->addWidget( mEventViewer );

  KHBox *suspendBox = new KHBox( topBox );
  suspendBox->setSpacing( spacingHint() );
  topLayout->addWidget( suspendBox );

  QLabel *l = new QLabel( i18n("Suspend &duration:"), suspendBox );
  mSuspendSpin = new QSpinBox( suspendBox );
  mSuspendSpin->setRange( 1, 9999 );
  mSuspendSpin->setValue( 5 );  // default suspend duration
  l->setBuddy( mSuspendSpin );

  mSuspendUnit = new KComboBox( suspendBox );
  mSuspendUnit->addItem( i18n("minute(s)") );
  mSuspendUnit->addItem( i18n("hour(s)") );
  mSuspendUnit->addItem( i18n("day(s)") );
  mSuspendUnit->addItem( i18n("week(s)") );

  // showButton( User2/*3*/, false );

  setMinimumSize( 300, 200 );
  connect(this,SIGNAL(okClicked()),this, SLOT(slotOk()));
  connect(this,SIGNAL(user1Clicked()),this,SLOT(slotUser1()));
  connect(this,SIGNAL(user2Clicked()),this,SLOT(slotUser2()));
}

AlarmDialog::~AlarmDialog()
{
  delete mIncidence;
}

void AlarmDialog::setIncidence( Incidence *incidence )
{
  mIncidence = incidence->clone();
  mEventViewer->appendIncidence( mIncidence );
}

void AlarmDialog::setRemindAt( const QDateTime &dt )
{
  mRemindAt = dt;
}

void AlarmDialog::slotOk()
{
  accept();
  emit finishedSignal( this );
}

void AlarmDialog::slotUser1()
{
  if ( !isVisible() )
    return;

  int unit=1;
  switch ( mSuspendUnit->currentIndex() ) {
    case 3: // weeks
      unit *=  7;
    case 2: // days
      unit *= 24;
    case 1: // hours
      unit *= 60;
    case 0: // minutes
      unit *= 60;
    default:
      break;
  }

  setTimer( unit * mSuspendSpin->value() );
  accept();
}

void AlarmDialog::setTimer( int seconds )
{
  connect( &mSuspendTimer, SIGNAL( timeout() ), SLOT( show() ) );
  mSuspendTimer.setSingleShot( true );
  mSuspendTimer.start( 1000 * seconds );
  mRemindAt = QDateTime::currentDateTime();
  mRemindAt = mRemindAt.addSecs( seconds );
}

void AlarmDialog::slotUser2()
{
  if ( !QDBusConnection::sessionBus().interface()->isServiceRegistered( "org.kde.korganizer" ) ) {
    if ( KToolInvocation::startServiceByDesktopName( "korganizer", QString() ) )
      KMessageBox::error( 0, i18n("Could not start KOrganizer.") );
  }
  org::kde::korganizer::Korganizer korganizer("org.kde.korganizer", "/Korganizer", QDBusConnection::sessionBus());
  korganizer.editIncidence(mIncidence->uid());

#ifdef Q_OS_X11
#ifdef __GNUC__
#warning "kde4: verify it when kontact will not crash"
#endif
  // get desktop # where korganizer (or kontact) runs
  QString object = QDBusConnection::sessionBus().interface()->isServiceRegistered( "org.kde.kontact" ) ?
           "kontact/mainwindow_1" : "korganizer/MainWindow_1";
  QDBusInterface korganizerObj("org.kde.korganizer", '/'+object);
  QDBusReply<int> reply = korganizerObj.call( "winId" );
  if ( reply.isValid() ) {
    int window = reply;
    int desktop = KWindowSystem::windowInfo( window, NET::WMDesktop ).desktop();

    if ( KWindowSystem::currentDesktop() == desktop ) {
      KWindowSystem::minimizeWindow( winId(), false );
    }
    else
      KWindowSystem::setCurrentDesktop( desktop );

    KWindowSystem::activateWindow( KWindowSystem::transientFor( window ) );
  }
#endif
}

void AlarmDialog::show()
{
  KDialog::show();
  KWindowSystem::setState( winId(), NET::KeepAbove );
  KWindowSystem::setOnAllDesktops( winId(), true );
  eventNotification();
}

void AlarmDialog::suspendValueChanged()
{
  setButtonFocus( User1 );
}

void AlarmDialog::suspendUnitChanged()
{
  setButtonFocus( User1 );
  setButtonFocus( User2 );
}

void AlarmDialog::eventNotification()
{
  bool beeped = false;
  Alarm::List alarms = mIncidence->alarms();
  Alarm::List::ConstIterator it;
  for ( it = alarms.begin(); it != alarms.end(); ++it ) {
    Alarm *alarm = *it;
// FIXME: Check whether this should be done for all multiple alarms
    if (alarm->type() == Alarm::Procedure) {
// FIXME: Add a message box asking whether the procedure should really be executed
      kDebug(5890) <<"Starting program: '" << alarm->programFile() <<"'";
      QProcess::startDetached( QFile::encodeName(alarm->programFile()));
    }
    else if (alarm->type() == Alarm::Audio) {
      beeped = true;
      Phonon::MediaObject* player = Phonon::createPlayer( Phonon::NotificationCategory, alarm->audioFile() );
      player->setParent( this );
      connect( player, SIGNAL( finished() ), player, SLOT( deleteLater() ) );
      player->play();
    }
  }

  if ( !beeped ) {
    KNotification::beep();
  }
}

void AlarmDialog::wakeUp()
{
  if ( mRemindAt <= QDateTime::currentDateTime() )
    show();
  else
    setTimer( QDateTime::currentDateTime().secsTo( mRemindAt ) );
}

void AlarmDialog::slotSave()
{
  KSharedConfig::Ptr config = KGlobal::config();
  // KLockFile::Ptr lock = config->lockFile();
  // if ( lock.data()->lock() != KLockFile::LockOK )
  //   return;

  KConfigGroup generalConfig( config, "General" );
  int numReminders = generalConfig.readEntry("Reminders", 0);
  generalConfig.writeEntry( "Reminders", ++numReminders );

  KConfigGroup incidenceConfig( config, QString("Incidence-%1").arg(numReminders) );
  incidenceConfig.writeEntry( "UID", mIncidence->uid() );
  incidenceConfig.writeEntry( "RemindAt", mRemindAt );
  config->sync();
  // lock.data()->unlock();
}

