/* This file is part of the KDE project
   Copyright (C) 2003-2004 Nadeem Hasan <nhasan@kde.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "weatherbutton.h"

#include <qpainter.h>
//Added by qt3to4:
#include <QPixmap>
#include <QResizeEvent>
#include <QEvent>

#include <kcursor.h>
#include <kglobalsettings.h>
#include <kiconeffect.h>
#include <kicontheme.h>
#include <kstandarddirs.h>

WeatherButton::WeatherButton( QWidget *parent, const char *name )
    : Q3Button( parent, name ), m_highlight( false )
{
    setBackgroundOrigin( AncestorOrigin );

    connect( KGlobalSettings::self(), SIGNAL( settingsChanged( int ) ),
       SLOT( slotSettingsChanged( int ) ) );
    connect( KGlobalSettings::self(), SIGNAL( iconChanged( int ) ),
       SLOT( slotIconChanged( int ) ) );

    slotSettingsChanged( KGlobalSettings::SETTINGS_MOUSE );
}

void WeatherButton::drawButton( QPainter *p )
{
    if ( isDown() || isOn() )
    {
        p->setPen( Qt::black );
        p->drawLine( 0, 0, width()-1, 0 );
        p->drawLine( 0, 0, 0, height()-1 );
        p->setPen( palette().color( QPalette::Light) );
        p->drawLine( 0, height()-1, width()-1, height()-1 );
        p->drawLine( width()-1, 0, width()-1, height()-1 );
    }

    drawButtonLabel( p );
}

void WeatherButton::drawButtonLabel( QPainter *p )
{
    if ( pixmap() )
    {
        QPixmap pix = m_highlight? m_activeIcon : m_normalIcon;

        if ( isOn() || isDown() )
            p->translate( 2, 2 );

        p->drawPixmap( pixmapOrigin(), pix );
    }
}

void WeatherButton::setPixmap( const QPixmap &pix )
{
    Q3Button::setPixmap( pix );
    generateIcons();
}

void WeatherButton::generateIcons()
{
    if ( !pixmap() )
        return;

    QImage image = pixmap()->toImage();
    image = image.smoothScale( pixmapSize(), Qt::KeepAspectRatio );

    KIconEffect effect;

    m_normalIcon = QPixmap::fromImage( effect.apply( image, K3Icon::Panel, K3Icon::DefaultState ) );
    m_activeIcon = QPixmap::fromImage( effect.apply( image, K3Icon::Panel, K3Icon::ActiveState ) );
}

void WeatherButton::slotSettingsChanged( int category )
{
    if ( category != KGlobalSettings::SETTINGS_MOUSE ) return;

    bool changeCursor = KGlobalSettings::changeCursorOverIcon();

    if ( changeCursor )
        setCursor( Qt::PointingHandCursor );
    else
        unsetCursor();
}

void WeatherButton::slotIconChanged( int group )
{
    if ( group != K3Icon::Panel )
        return;

    generateIcons();
    repaint( );
}

void WeatherButton::enterEvent( QEvent *e )
{
    m_highlight = true;

    repaint( );
    Q3Button::enterEvent( e );
}

void WeatherButton::leaveEvent( QEvent *e )
{
    m_highlight = false;

    repaint( );
    Q3Button::enterEvent( e );
}

void WeatherButton::resizeEvent( QResizeEvent * )
{
    generateIcons();
}

QPoint WeatherButton::pixmapOrigin() const
{
    QSize point = margin()/2;
    QPoint origin( point.width(), point.height() );

    return origin;
}

#include "weatherbutton.moc"

// vim:ts=4:sw=4:et
