/*
 *   khexedit - Versatile hex editor
 *   Copyright (C) 1999  Espen Sand, espensa@online.no
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */
#include "dialog.h"

#include <ctype.h>
#include <iostream>
#include <limits.h>

#include <QButtonGroup>
#include <QFileInfo>
#include <QLayout>
#include <QValidator>
#include <q3buttongroup.h>

#include <kfiledialog.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kseparator.h>
#include <KStandardGuiItem>

#include <QPushButton>
#include <kglobal.h>

static const QStringList &formatStrings( void );
static const QStringList &operationStrings( void );


CGotoDialog::CGotoDialog( QWidget *parent )
  :KDialog( parent )
{
    setCaption(  i18n("Goto Offset") );
    setButtons( KDialog::Ok | KDialog::Cancel );
    setDefaultButton( KDialog::Ok );
    setModal( false );

  setMainWidget( new QFrame( this ) );
  QVBoxLayout *topLayout = new QVBoxLayout( mainWidget() );
  topLayout->setSpacing( spacingHint() );
  topLayout->setMargin( 0 );

  QVBoxLayout *vbox = new QVBoxLayout();
  topLayout->addLayout( vbox );

  mComboBox = new KComboBox( mainWidget() );
  mComboBox->setEditable( true );
  mComboBox->setMaxCount( 10 );
  mComboBox->setInsertPolicy( KComboBox::InsertAtTop );
  mComboBox->setMinimumWidth( fontMetrics().maxWidth()*17 );

  QLabel *label = new QLabel( i18n("O&ffset:"), mainWidget() );
  label->setBuddy( mComboBox );

  vbox->addWidget( label );
  vbox->addWidget( mComboBox );

  Q3ButtonGroup *group = new Q3ButtonGroup( i18n("Options"), mainWidget() );
  topLayout->addWidget( group, 10 );

  QGridLayout *gbox = new QGridLayout( group );
  gbox->setSpacing( spacingHint() );
  gbox->addItem( new QSpacerItem( 0, fontMetrics().lineSpacing() ), 0, 0 );
  mCheckFromCursor = new QCheckBox( i18n("&From cursor"), group );
  gbox->addWidget( mCheckFromCursor, 1, 0 );
  mCheckBackward = new QCheckBox( i18n("&Backwards"), group );
  gbox->addWidget( mCheckBackward, 1, 1 );
  mCheckVisible = new QCheckBox( i18n("&Stay visible"), group );
  gbox->addWidget( mCheckVisible, 2, 0 );
  gbox->setRowStretch( 3, 10 );

  KConfig &config = *KGlobal::config();
  KConfigGroup gr = config.group("Goto Dialog");
  mCheckFromCursor->setChecked( gr.readEntry( "FromCursor", false ) );
  mCheckVisible->setChecked( gr.readEntry( "StayVisible", true ) );
  mCheckBackward->setChecked( gr.readEntry( "Backwards", false ) );
}



CGotoDialog::~CGotoDialog( void )
{
  KConfig &config = *KGlobal::config();
  KConfigGroup gr = config.group("Goto Dialog");
  gr.writeEntry( "FromCursor", mCheckFromCursor->isChecked() );
  gr.writeEntry( "StayVisible", mCheckVisible->isChecked() );
  gr.writeEntry( "Backwards", mCheckBackward->isChecked() );
  gr.sync();
}



void CGotoDialog::showEvent( QShowEvent *e )
{
  KDialog::showEvent(e);
  mComboBox->setFocus();
}



//
// Format of input string:
// 0x<HexNumber>|<DecimalNumber>s<Bit>
// s = :,. or space
//
void CGotoDialog::slotOk( void )
{
  uint offset;
  bool success = stringToOffset( mComboBox->currentText(), offset );
  if( success == false )
  {
    showEntryFailure( this, QString("") );
    return;
  }

  if( mCheckVisible->isChecked() == false )
  {
    hide();
  }
  emit gotoOffset( offset, 7, mCheckFromCursor->isChecked(),
		   mCheckBackward->isChecked() == true ? false : true );

  #if 0
  const char *p = mComboBox->currentText();
  if( strlen( p ) == 0 )
  {
    return;
  }

  //
  // Skip any whitespaces in front of string
  //
  for( ; *p != 0 && isspace( *p ) ; p++ );

  uint offset, bit;
  int match;
  if( strncmp( p, "0x", 2 ) == 0 || strncmp( p, "0X", 2 ) == 0 )
  {
    match = sscanf( p+2, "%x", &offset );
  }
  else
  {
    match = sscanf( p, "%u", &offset );
  }

  if( match == 0 )
  {
    return;
  }

  bit = 7;

  p = strpbrk( p, ":,. " );
  if( p != 0 )
  {
    match = sscanf( p+1, "%u", &bit );
    if( match == 0 )
    {
      return;
    }
    if( bit > 7 ) { bit = 7; }
  }
  #endif
}



CFindDialog::CFindDialog( QWidget *parent )
  :KDialog( parent )
{
    setCaption( i18n( "Find" ) );
    setButtons( KDialog::Ok | KDialog::Cancel );
    setDefaultButton( KDialog::Ok );
    setModal( false );

  setMainWidget( new QFrame( this ) );
  QVBoxLayout *topLayout = new QVBoxLayout( mainWidget() );
  topLayout->setSpacing( spacingHint() );
  topLayout->setMargin( 0 );

  QVBoxLayout *vbox = new QVBoxLayout();
  topLayout->addLayout( vbox );

  mSelector = new KComboBox( mainWidget() );
  mSelector->setMinimumWidth( fontMetrics().maxWidth()*17 );
  mSelector->addItems( formatStrings() );
  connect( mSelector, SIGNAL(activated(int)), SLOT(selectorChanged(int)) );

  QLabel *label = new QLabel( i18n("Fo&rmat:"), mainWidget() );
  label->setBuddy( mSelector );

  vbox->addWidget( label );
  vbox->addWidget( mSelector );

  mInput = new KLineEdit( mainWidget() );
  mInput->setMinimumWidth( fontMetrics().maxWidth()*17 );
  connect( mInput, SIGNAL(textChanged(const QString&)),
	   SLOT(inputChanged(const QString&)) );
  mFindValidator = new CHexValidator( this, CHexValidator::regularText );
  mInput->setValidator( mFindValidator );

  label = new QLabel( i18n("F&ind:"), mainWidget() );
  label->setBuddy( mInput );

  vbox->addWidget( label );
  vbox->addWidget( mInput );

  Q3ButtonGroup *group = new Q3ButtonGroup( i18n("Options"), mainWidget() );
  if( group == 0 ) { return; }

  QGridLayout *gbox = new QGridLayout( group );
  gbox->setSpacing( spacingHint() );
  gbox->addItem( new QSpacerItem( 0, fontMetrics().lineSpacing() ), 0, 0 );

  mCheckFromCursor = new QCheckBox( i18n("&From cursor"), group );
  gbox->addWidget( mCheckFromCursor, 1, 0 );
  mCheckBackward = new QCheckBox( i18n("&Backwards"), group );
  gbox->addWidget( mCheckBackward, 1, 1 );
  mCheckInSelection = new QCheckBox( i18n("&In selection"), group );
  gbox->addWidget( mCheckInSelection, 2, 0 );
  mCheckUseNavigator = new QCheckBox( i18n("&Use navigator"),group);
  gbox->addWidget( mCheckUseNavigator, 2, 1 );
  mCheckIgnoreCase = new QCheckBox( i18n("Ignore c&ase"),group);
  gbox->addWidget( mCheckIgnoreCase, 3, 0 );
  gbox->setRowStretch( 4, 10 );

  KConfig &config = *KGlobal::config();
  KConfigGroup gr = config.group("Find Dialog");
  mCheckFromCursor->setChecked( gr.readEntry( "FromCursor", true ) );
  mCheckInSelection->setChecked( gr.readEntry( "InSelection", false) );
  mCheckIgnoreCase->setChecked( gr.readEntry( "IgnoreCase", false ) );
  mCheckBackward->setChecked( gr.readEntry( "Backwards", false ) );
  mCheckUseNavigator->setChecked( gr.readEntry( "UseNavigator", true));
  uint val = gr.readEntry( "Format", 0 );
  mSelector->setCurrentIndex(qMin(4u,val) );
  selectorChanged( mSelector->currentIndex() );
  enableButtonOk(!mInput->text().isEmpty());
}


CFindDialog::~CFindDialog( void )
{
  KConfig &config = *KGlobal::config();
  KConfigGroup gr = config.group("Find Dialog");
  gr.writeEntry( "FromCursor", mCheckFromCursor->isChecked() );
  gr.writeEntry( "InSelection", mCheckInSelection->isChecked() );
  gr.writeEntry( "IgnoreCase", mCheckIgnoreCase->isChecked() );
  gr.writeEntry( "Backwards", mCheckBackward->isChecked() );
  gr.writeEntry( "UseNavigator", mCheckUseNavigator->isChecked() );
  gr.writeEntry( "Format", mSelector->currentIndex() );
  gr.sync();
}


void CFindDialog::selectorChanged( int index )
{
  mFindValidator->setState( (CHexValidator::EState)index );
  mInput->setText( mFindString[ index ] );
  mCheckIgnoreCase->setEnabled( index == 4 );
}


void CFindDialog::inputChanged( const QString &text )
{
  mFindString[ mSelector->currentIndex() ] = text;
  mFindValidator->convert( mFindData,
    mFindString[ mSelector->currentIndex() ] );
  enableButtonOk(!text.isEmpty());
}


void CFindDialog::showEvent( QShowEvent *e )
{
  KDialog::showEvent(e);
  mInput->setFocus();
}


bool CFindDialog::isEmpty( void )
{
  return( mFindData.isEmpty() );
}


void CFindDialog::slotOk( void )
{
  if( isEmpty() == true )
  {
    showEntryFailure( this, QString("") );
    return;
  }

  SSearchControl sc;
  sc.key = mFindData;
  sc.keyType = mSelector->currentIndex();
  sc.fromCursor = mCheckFromCursor->isChecked();
  sc.inSelection = mCheckInSelection->isChecked();
  sc.forward = mCheckBackward->isChecked() == true ? false : true;
  sc.ignoreCase = mCheckIgnoreCase->isEnabled() && mCheckIgnoreCase->isChecked();

  hide();
  emit findData( sc, Find_First, mCheckUseNavigator->isChecked() );
}


void CFindDialog::findAgain( EOperation operation )
{
  if( isEmpty() == true )
  {
    showEntryFailure( this, QString("") );
    return;
  }

  SSearchControl sc;
  sc.key = mFindData;
  sc.fromCursor = true;
  sc.inSelection = mCheckInSelection->isChecked();
  sc.ignoreCase = mCheckIgnoreCase->isEnabled() && mCheckIgnoreCase->isChecked();
  if( operation == find_Next )
  {
    sc.forward = true;
  }
  else if( operation == find_Previous )
  {
    sc.forward = false;
  }
  else
  {
    sc.forward = mCheckBackward->isChecked() == true ? false : true;
  }

  hide();
  emit findData( sc, Find_Next, false );
}



CFindNavigatorDialog::CFindNavigatorDialog( QWidget *parent )
  :KDialog( parent )
{
    setCaption( i18n("Find (Navigator)") );
    setButtons( KDialog::User3|KDialog::User2|KDialog::User1|KDialog::Close );
    setButtonText( KDialog::User1, i18n("New &Key") );
    setButtonText( KDialog::User2, i18n("&Next") );
    setButtonText( KDialog::User3, i18n("&Previous") );
    setModal( false );

  setMainWidget( new QFrame( this ) );
  QBoxLayout *topLayout = new QVBoxLayout( mainWidget() );
  topLayout->setSpacing( spacingHint() );
  topLayout->setMargin( 0 );

  topLayout->addSpacing( spacingHint() ); // A little bit extra space

  QHBoxLayout *hbox = new QHBoxLayout();
  topLayout->addLayout( hbox );

  QLabel *label = new QLabel( i18n("Searching for:"), mainWidget() );
  hbox->addWidget( label );

  mKey = new KLineEdit( mainWidget() );
  mKey->setMinimumWidth( fontMetrics().width("M") * 20 );
  mKey->setFocusPolicy( Qt::NoFocus );
  hbox->addWidget( mKey );

  topLayout->addSpacing( spacingHint() ); // A little bit extra space
  topLayout->addStretch(10);
}


CFindNavigatorDialog::~CFindNavigatorDialog( void )
{
}


void CFindNavigatorDialog::defineData( SSearchControl &sc )
{
  mSearchControl = sc;
  mSearchControl.key = sc.key;

  if( mSearchControl.key.isEmpty() )
  {
    mKey->setText("");
    return;
  }

  if( mSearchControl.keyType == 0 )
  {
    QString str;
    for( int i=0; i<mSearchControl.key.size(); i++ )
    {
      str += mSearchControl.key[i];
    }
    mKey->setText( str );

  }
  else if( mSearchControl.keyType == 1 )
  {
    QString str("0x ");
    for( int i=0; i<mSearchControl.key.size(); i++ )
    {
      str += QString().sprintf("%02X ", (unsigned char)mSearchControl.key[i]);
    }
    mKey->setText( str );
  }
  else if( mSearchControl.keyType == 2 )
  {
    QString str;
    for( int i=0; i<mSearchControl.key.size(); i++ )
    {
      str += QString().sprintf("%03o ", (unsigned char)mSearchControl.key[i]);
    }
    mKey->setText( str );
  }
  else
  {
    char buf[10];
    memset( buf, 0, sizeof( buf ) ); buf[8] = ' ';

    QString str;
    for( int i=0; i<mSearchControl.key.size(); i++ )
    {
      unsigned char data = (unsigned char)mSearchControl.key[i];
      for( int j = 0; j < 8; j++ )
      {
	buf[7-j] = (data&(1<<j)) ? '1' : '0';
      }
      str += buf;
    }
    mKey->setText( str );
  }
}


void CFindNavigatorDialog::slotUser3( void ) // Previous
{
  done( repPrevious );
}


void CFindNavigatorDialog::slotUser2( void ) // Next
{
  done( repNext );
}


void CFindNavigatorDialog::slotUser1( void ) // New key
{
  done( repNewKey );
}


void CFindNavigatorDialog::slotClose( void )
{
  done( repClose );
}


void CFindNavigatorDialog::done( int resultCode )
{
  setResult( resultCode );
  if( resultCode == repClose || resultCode == repNewKey )
  {
    if( resultCode == repNewKey )
    {
      emit makeKey();
    }
    hide();
    return;
  }

  mSearchControl.forward = resultCode == repNext ? true : false;
  emit findData( mSearchControl, Find_Next, true );
}




CReplaceDialog::CReplaceDialog( QWidget *parent )
  :KDialog( parent )
{
    setCaption( i18n("Find & Replace") );
    setButtons(  KDialog::Ok|KDialog::Cancel );
    setDefaultButton( KDialog::Ok );
    setModal( false );
  
  setMainWidget( new QFrame( this ) );
  QVBoxLayout *topLayout = new QVBoxLayout( mainWidget() );
  topLayout->setSpacing( spacingHint() );
  topLayout->setMargin( 0 );

  QVBoxLayout *vbox = new QVBoxLayout();
  topLayout->addLayout( vbox );


  mFindSelector = new KComboBox( mainWidget() );
  mFindSelector->setMinimumWidth( fontMetrics().maxWidth()*17 );
  mFindSelector->addItems( formatStrings() );
  connect( mFindSelector, SIGNAL(activated(int)),
	   SLOT(findSelectorChanged(int)) );

  QLabel *label = new QLabel( i18n("Fo&rmat (find):"), mainWidget() );
  label->setBuddy( mFindSelector );

  vbox->addWidget( label );
  vbox->addWidget( mFindSelector );

  mFindInput = new KLineEdit( mainWidget() );
  mFindInput->setMinimumWidth( fontMetrics().maxWidth()*17 );
  mFindValidator = new CHexValidator( this, CHexValidator::regularText );
  mFindInput->setValidator( mFindValidator );
  connect( mFindInput, SIGNAL(textChanged(const QString&)),
	   SLOT(findInputChanged(const QString&)) );

  label = new QLabel( i18n("F&ind:"), mainWidget() );
  label->setBuddy( mFindInput );

  vbox->addWidget( label );
  vbox->addWidget( mFindInput  );

  mReplaceSelector = new KComboBox( mainWidget() );
  mReplaceSelector->setMinimumWidth( fontMetrics().maxWidth()*17 );
  mReplaceSelector->addItems( formatStrings() );
  connect( mReplaceSelector, SIGNAL(activated(int)),
	   SLOT(replaceSelectorChanged(int)) );

  label = new QLabel( i18n("For&mat (replace):"), mainWidget() );
  label->setFixedHeight( label->sizeHint().height() );
  label->setBuddy( mReplaceSelector );

  vbox->addWidget( label );
  vbox->addWidget( mReplaceSelector );

  mReplaceInput = new KLineEdit( mainWidget() );
  mReplaceInput->setMinimumWidth( fontMetrics().maxWidth()*17 );
  mReplaceValidator = new CHexValidator( this, CHexValidator::regularText );
  if( mReplaceValidator == 0 ) { return; }
  mReplaceInput->setValidator( mReplaceValidator );
  connect( mReplaceInput, SIGNAL(textChanged(const QString&)),
	   SLOT(replaceInputChanged(const QString&)) );

  label = new QLabel( i18n("Rep&lace:"), mainWidget() );
  label->setBuddy( mReplaceInput );
  label->setFixedHeight( label->sizeHint().height() );

  vbox->addWidget( label );
  vbox->addWidget( mReplaceInput );

  Q3ButtonGroup *group = new Q3ButtonGroup( i18n("Options"), mainWidget() );
  topLayout->addWidget( group, 10 );

  QGridLayout *gbox = new QGridLayout( group );
  gbox->setSpacing( spacingHint() );
  gbox->addItem( new QSpacerItem( 0, fontMetrics().lineSpacing() ), 0, 0 );
  mCheckFromCursor = new QCheckBox( i18n("&From cursor"), group );
  gbox->addWidget( mCheckFromCursor, 1, 0 );
  mCheckBackward = new QCheckBox( i18n("&Backwards"), group );
  gbox->addWidget( mCheckBackward, 1, 1 );
  mCheckInSelection = new QCheckBox( i18n("&In selection"), group );
  gbox->addWidget( mCheckInSelection, 2, 0 );
  mCheckPrompt = new QCheckBox( i18n("&Prompt"), group );
  gbox->addWidget( mCheckPrompt, 2, 1 );
  mCheckIgnoreCase = new QCheckBox( i18n("Ignore c&ase"), group );
  gbox->addWidget( mCheckIgnoreCase, 3, 0 );
  gbox->setRowStretch( 4, 10 );

  KConfig &config = *KGlobal::config();
  KConfigGroup gr = config.group("Replace Dialog");
  mCheckFromCursor->setChecked( gr.readEntry( "FromCursor", true ) );
  mCheckInSelection->setChecked( gr.readEntry( "InSelection", false) );
  mCheckIgnoreCase->setChecked( gr.readEntry( "IgnoreCase", false ) );
  mCheckBackward->setChecked( gr.readEntry( "Backwards", false ) );
  mCheckPrompt->setChecked( gr.readEntry( "Prompt", true));
  uint val = gr.readEntry( "FindFormat", 0 );
  mFindSelector->setCurrentIndex(qMin(4u,val) );
  findSelectorChanged( mFindSelector->currentIndex() );
  val = gr.readEntry( "ReplaceFormat", 0 );
  mReplaceSelector->setCurrentIndex(qMin(4u,val) );
  replaceSelectorChanged( mReplaceSelector->currentIndex() );
  enableButtonOk(!mFindInput->text().isEmpty());
}


CReplaceDialog::~CReplaceDialog( void )
{
  KConfig &config = *KGlobal::config();
  KConfigGroup gr = config.group("Replace Dialog");
  gr.writeEntry( "FromCursor", mCheckFromCursor->isChecked() );
  gr.writeEntry( "InSelection", mCheckInSelection->isChecked() );
  gr.writeEntry( "IgnoreCase", mCheckIgnoreCase->isChecked() );
  gr.writeEntry( "Backwards", mCheckBackward->isChecked() );
  gr.writeEntry( "Prompt", mCheckPrompt->isChecked() );
  gr.writeEntry( "FindFormat", mFindSelector->currentIndex() );
  gr.writeEntry( "ReplaceFormat", mReplaceSelector->currentIndex() );
  gr.sync();
}


void CReplaceDialog::findSelectorChanged( int index )
{
  mFindValidator->setState( (CHexValidator::EState)index );
  mFindInput->setText( mFindString[ index ] );
  mCheckIgnoreCase->setEnabled( index == 4 );
}


void CReplaceDialog::findInputChanged( const QString &text )
{
  mFindString[ mFindSelector->currentIndex() ] = text;
  mFindValidator->convert( mFindData,
    mFindString[ mFindSelector->currentIndex() ] );
  enableButtonOk(!text.isEmpty());
}


void CReplaceDialog::replaceSelectorChanged( int index )
{
  mReplaceValidator->setState( (CHexValidator::EState)index );
  mReplaceInput->setText( mReplaceString[ index ] );
}


void CReplaceDialog::replaceInputChanged( const QString &text )
{
  mReplaceString[ mReplaceSelector->currentIndex() ] = text;
  mReplaceValidator->convert( mReplaceData,
    mReplaceString[ mReplaceSelector->currentIndex() ] );
}


void CReplaceDialog::showEvent( QShowEvent *e )
{
  KDialog::showEvent(e);
  mFindInput->setFocus();
}


void CReplaceDialog::slotOk( void )
{
  if( mFindData.isEmpty() == true )
  {
    showEntryFailure( this, QString("") );
    return;
  }

  if( mFindData == mReplaceData )
  {
    showEntryFailure( this,i18n("Source and target values can not be equal."));
    return;
  }

  hide();

  SSearchControl sc;
  sc.key = mFindData;
  sc.val = mReplaceData;
  sc.fromCursor = mCheckFromCursor->isChecked();
  sc.inSelection = mCheckInSelection->isChecked();
  sc.forward = mCheckBackward->isChecked() == true ? false : true;

  sc.ignoreCase = mCheckIgnoreCase->isEnabled() && mCheckIgnoreCase->isChecked();

  emit replaceData( sc, mCheckPrompt->isChecked() ? Replace_First: Replace_AllInit );
}





CReplacePromptDialog::CReplacePromptDialog( QWidget *parent )
  :KDialog( parent )
{
    setCaption( i18n("Find & Replace") );
    setButtons( KDialog::User3|KDialog::User2|KDialog::User1|KDialog::Close );
    setDefaultButton( KDialog::User2 );
    setButtonText( KDialog::User1, i18n("Replace &All") );
    setButtonText( KDialog::User2, i18n("Do Not Replace") );
    setButtonText( KDialog::User3, i18n("Replace") );
    setModal( false );

  setMainWidget( new QFrame( this ) );
  QBoxLayout *topLayout = new QVBoxLayout( mainWidget() );
  topLayout->setSpacing( spacingHint() );
  topLayout->setMargin( 0 );

  topLayout->addSpacing( spacingHint() ); // A little bit extra space

  QLabel* label = new QLabel( i18n("Replace marked data at cursor position?"),
                              mainWidget() );
  topLayout->addWidget( label );

  topLayout->addSpacing( spacingHint() ); // A little bit extra space
  topLayout->addStretch(10);
}


CReplacePromptDialog::~CReplacePromptDialog( void )
{
}


void CReplacePromptDialog::defineData( SSearchControl &sc )
{
  mSearchControl = sc;
  mSearchControl.key = sc.key;
  mSearchControl.val = sc.val;
}


void CReplacePromptDialog::slotUser3( void )
{
  done( repYes );
}


void CReplacePromptDialog::slotUser2( void )
{
  done( repNo );
}


void CReplacePromptDialog::slotUser1( void )
{
  done( repAll );
}


void CReplacePromptDialog::slotClose( void )
{
  done( repClose );
}


void CReplacePromptDialog::done( int returnCode )
{
  if( returnCode == repClose )
  {
    hide();
  }
  else if( returnCode == repYes )
  {
    emit replaceData( mSearchControl, Replace_Next );
  }
  else if( returnCode == repNo )
  {
    emit replaceData( mSearchControl, Replace_Ignore );
  }
  else
  {
    emit replaceData( mSearchControl, Replace_All );
  }
}




CFilterDialog::CFilterDialog( QWidget *parent )
  :KDialog( parent)
{
    setCaption( i18n("Binary Filter") );
    setButtons( KDialog::Ok|KDialog::Cancel );
    setDefaultButton( KDialog::Ok );
    setModal( false );
  
  setMainWidget( new QFrame( this ) );
  QVBoxLayout *topLayout = new QVBoxLayout( mainWidget() );
  if( topLayout == 0 ) { return; }
  topLayout->setSpacing( spacingHint() );
  topLayout->setMargin( 0 );

  QVBoxLayout *vbox = new QVBoxLayout();
  if( vbox == 0 ) { return; }
  topLayout->addLayout( vbox );


  mOperationSelector = new KComboBox( mainWidget() );
  if( mOperationSelector == 0 ) { return; }
  mOperationSelector->setMinimumWidth( fontMetrics().maxWidth()*17 );
  mOperationSelector->addItems( operationStrings() );
  connect( mOperationSelector, SIGNAL(activated(int)),
	   SLOT(operationSelectorChanged(int)) );

  QLabel *label = new QLabel( i18n("O&peration:"), mainWidget() );
  if( label == 0 ) { return; }
  label->setBuddy( mOperationSelector );

  vbox->addWidget( label );
  vbox->addWidget( mOperationSelector );

  KSeparator *separator = new KSeparator( mainWidget() );
  separator->setOrientation( Qt::Horizontal );
  vbox->addWidget( separator );


  mWidgetStack = new QStackedWidget ( mainWidget() );
  if( mWidgetStack == 0 ) { return; }
  vbox->addWidget( mWidgetStack );

  makeEmptyLayout();
  makeOperandLayout();
  makeBitSwapLayout();
  makeRotateLayout();
  mWidgetStack->setCurrentIndex( (int)OperandPage );


  Q3ButtonGroup *group = new Q3ButtonGroup( i18n("Options"), mainWidget() );
  if( group == 0 ) { return; }
  topLayout->addWidget( group, 10 );

  QGridLayout *gbox = new QGridLayout( group );
  if( gbox == 0 ) { return; }
  gbox->setSpacing( spacingHint() );
  gbox->addItem( new QSpacerItem( 0, fontMetrics().lineSpacing() ), 0, 0 );
  mCheckFromCursor = new QCheckBox( i18n("&From cursor"), group );
  gbox->addWidget( mCheckFromCursor, 1, 0 );
  mCheckBackward = new QCheckBox( i18n("&Backwards"), group );
  gbox->addWidget( mCheckBackward, 1, 1 );
  mCheckInSelection = new QCheckBox( i18n("&In selection"), group );
  gbox->addWidget( mCheckInSelection, 2, 0 );
  mCheckVisible = new QCheckBox( i18n("&Stay visible"), group );
  gbox->addWidget( mCheckVisible, 2, 1 );
  gbox->setRowStretch( 3, 10 );
}


CFilterDialog::~CFilterDialog( void )
{
  delete mOperandValidator;
}


void CFilterDialog::makeEmptyLayout( void )
{
  QFrame *page = new QFrame( mainWidget() );
  if( page == 0 ) { return; }
  mWidgetStack->insertWidget( EmptyPage, page );
}


void CFilterDialog::makeOperandLayout( void )
{
  QString text;

  QFrame *page = new QFrame( mainWidget() );
  if( page == 0 ) { return; }
  mWidgetStack->insertWidget( OperandPage,page );

  QVBoxLayout *vbox = new QVBoxLayout( page );
  vbox->setSpacing( spacingHint() );
  vbox->setMargin( 0 );

  mOperandSelector = new KComboBox( page );
  if( mOperandSelector == 0 ) { return; }
  mOperandSelector->setFixedHeight( mOperandSelector->sizeHint().height());
  mOperandSelector->setMinimumWidth( fontMetrics().width("M")*20 );
  mOperandSelector->addItems( formatStrings() );
  connect( mOperandSelector, SIGNAL(activated(int)),
	   SLOT(operandSelectorChanged(int)) );

  text = i18n("Fo&rmat (operand):");
  mOperandFormatLabel = new QLabel( text, page );
  if( mOperandFormatLabel == 0 ) { return; }
  mOperandFormatLabel->setBuddy( mOperandSelector );

  vbox->addWidget( mOperandFormatLabel );
  vbox->addWidget( mOperandSelector );

  mOperandInput = new KLineEdit( page );
  if( mOperandInput == 0 ) { return; }
  mOperandInput->setMinimumWidth( fontMetrics().width("M") * 20 );
  mOperandValidator = new CHexValidator( this, CHexValidator::regularText );
  if( mOperandValidator == 0 ) { return; }
  mOperandInput->setValidator( mOperandValidator );
  connect( mOperandInput, SIGNAL(textChanged(const QString&)),
	   SLOT(operandInputChanged(const QString&)) );

  mOperandInputLabel = new QLabel( i18n("O&perand:"), page );
  if( mOperandInputLabel == 0 ) { return; }
  mOperandInputLabel->setBuddy( mOperandInput );

  vbox->addWidget( mOperandInputLabel );
  vbox->addWidget( mOperandInput  );
  vbox->addSpacing( 1 );
}


void CFilterDialog::makeBitSwapLayout( void )
{
  QString text;

  QFrame *page = new QFrame( mainWidget() );
  mWidgetStack->insertWidget( BitSwapPage,page );

  QVBoxLayout *vbox = new QVBoxLayout( page );
  vbox->setSpacing( spacingHint() );
  vbox->setMargin( 0 );

  text = i18n("Swap rule");
  QLabel *label = new QLabel( text, page );
  label->setFixedHeight( label->sizeHint().height() );
  vbox->addWidget( label );

  mByteWidget = new CByteWidget( page );
  vbox->addWidget( mByteWidget );

  QHBoxLayout *hbox = new QHBoxLayout();
  vbox->addLayout( hbox );

  text = i18n("&Reset");
  QPushButton *resetButton = new QPushButton( text, page );
  resetButton->setFixedHeight( resetButton->sizeHint().height() );
  connect( resetButton, SIGNAL(clicked()), mByteWidget, SLOT(reset()) );

  hbox->addWidget( resetButton );
  hbox->addStretch( 10 );
}


void CFilterDialog::makeRotateLayout( void )
{
  QString text;

  QFrame *page = new QFrame( mainWidget() );
  mWidgetStack->insertWidget( RotatePage,page );

  QVBoxLayout *vbox = new QVBoxLayout( page );
  vbox->setSpacing( spacingHint() );
  vbox->setMargin( 0 );

  mGroupSpin = new QSpinBox( page );
  mGroupSpin->setMinimumWidth( fontMetrics().width("M")*20 );
  mGroupSpin->setRange(1, INT_MAX );

  text = i18n("&Group size [bytes]");
  QLabel *label = new QLabel( text, page );
  label->setBuddy( mGroupSpin );

  vbox->addWidget( label );
  vbox->addWidget( mGroupSpin );

  mBitSpin = new QSpinBox( page );
  mBitSpin->setMinimumWidth( fontMetrics().width("M")*20 );
  mBitSpin->setRange(INT_MIN, INT_MAX);

  text = i18n("S&hift size [bits]");
  label = new QLabel( text, page );
  label->setBuddy( mBitSpin );

  vbox->addWidget( label );
  vbox->addWidget( mBitSpin );
}



void CFilterDialog::showEvent( QShowEvent *e )
{
  KDialog::showEvent(e);
  mOperandInput->setFocus();
}



void CFilterDialog::slotOk( void )
{
  SFilterControl fc;
  switch( mOperationSelector->currentIndex() )
  {
    case SFilterControl::OperandAndData:
    case SFilterControl::OperandOrData:
    case SFilterControl::OperandXorData:
      if( mOperandData.isEmpty() == true )
      {
	showEntryFailure( this, QString("") );
	return;
      }
      fc.operand = mOperandData;
    break;

    case SFilterControl::InvertData:
    case SFilterControl::ReverseData:
    break;

    case SFilterControl::RotateData:
    case SFilterControl::ShiftData:
      fc.rotate[0] = mGroupSpin->value();
      fc.rotate[1] = mBitSpin->value();
      if( fc.rotate[1] == 0 )
      {
	QString msg = i18n("Shift size is zero.");
	showEntryFailure( this, msg );
	return;
      }
    break;

    case SFilterControl::SwapBits:
      if( mByteWidget->flag( fc.operand ) == false )
      {
	QString msg = i18n("Swap rule does not define any swapping.");
	showEntryFailure( this, msg );
	return;
      }
    break;

    default:
      return;
    break;

  }

  if( mCheckVisible->isChecked() == false )
  {
    hide();
  }

  fc.operation = (SFilterControl::Operation)mOperationSelector->currentIndex();
  fc.fromCursor = mCheckFromCursor->isChecked();
  fc.inSelection = mCheckInSelection->isChecked();
  if( mCheckBackward->isEnabled() == true )
  {
    fc.forward = mCheckBackward->isChecked() == true ? false : true;
  }
  else
  {
    fc.forward = true;
  }

  emit filterData( fc );
}


void CFilterDialog::operandSelectorChanged( int index )
{
  mOperandValidator->setState( (CHexValidator::EState)index );
  mOperandInput->setText( mOperandString[ index ] );
}


void CFilterDialog::operandInputChanged( const QString &text )
{
  mOperandString[ mOperandSelector->currentIndex() ] = text;
  mOperandValidator->convert( mOperandData,
    mOperandString[ mOperandSelector->currentIndex() ] );
}


void CFilterDialog::operationSelectorChanged( int index )
{
  if( index <= 2 )
  {
    mWidgetStack->setCurrentIndex( OperandPage );
    mCheckBackward->setEnabled( true );
  }
  else if( index <= 4 )
  {
    mWidgetStack->setCurrentIndex( EmptyPage );
    mCheckBackward->setEnabled( true );
  }
  else if( index <= 6 )
  {
    mWidgetStack->setCurrentIndex( RotatePage );
    mCheckBackward->setEnabled( false );
  }
  else
  {
    mWidgetStack->setCurrentIndex( BitSwapPage );
    mCheckBackward->setEnabled( true );
  }
}




CInsertDialog::CInsertDialog( QWidget *parent )
  :KDialog( parent)
{
    setCaption( i18n("Insert Pattern") );
    setButtons( KDialog::Ok|KDialog::Cancel );
    setDefaultButton( KDialog::Ok );
    setModal( false );
  setButtonGuiItem( Ok, KGuiItem(i18n("&Insert")) );

  setMainWidget( new QFrame( this ) );
  QVBoxLayout *topLayout = new QVBoxLayout( mainWidget() );
  topLayout->setSpacing( spacingHint() );
  topLayout->setMargin( 0 );
  QVBoxLayout *vbox = new QVBoxLayout();
  topLayout->addLayout( vbox );

  mSizeBox = new QSpinBox( mainWidget() );
  mSizeBox->setMinimumWidth( fontMetrics().maxWidth()*17 );
  mSizeBox->setRange( 1, INT_MAX );
  mSizeBox->setValue( 1 );

  QLabel *label = new QLabel( i18n("&Size:"), mainWidget() );
  label->setBuddy( mSizeBox );

  vbox->addWidget( label );
  vbox->addWidget( mSizeBox );

  mPatternSelector = new KComboBox( mainWidget() );
  mPatternSelector->setMinimumWidth( fontMetrics().maxWidth()*17 );
  mPatternSelector->addItems( formatStrings() );
  connect( mPatternSelector, SIGNAL(activated(int)),
	   SLOT(patternSelectorChanged(int)) );

  label = new QLabel( i18n("Fo&rmat (pattern):"), mainWidget() );
  label->setBuddy( mPatternSelector );

  vbox->addWidget( label );
  vbox->addWidget( mPatternSelector );

  mPatternInput = new KLineEdit( mainWidget() );
  mPatternInput->setMinimumWidth( fontMetrics().maxWidth()*17 );
  mPatternValidator = new CHexValidator( this, CHexValidator::regularText );
  if( mPatternValidator == 0 ) { return; }
  mPatternInput->setValidator( mPatternValidator );
  connect( mPatternInput, SIGNAL(textChanged(const QString&)),
	   SLOT(patternInputChanged(const QString&)) );

  label = new QLabel( i18n("&Pattern:"), mainWidget() );
  label->setBuddy( mPatternInput );

  vbox->addWidget( label );
  vbox->addWidget( mPatternInput );

  mOffsetInput = new KLineEdit( mainWidget() );
  mOffsetInput->setMinimumWidth( fontMetrics().maxWidth()*17 );

  mOffsetLabel = new QLabel( i18n("&Offset:"), mainWidget() );
  mOffsetLabel->setBuddy( mOffsetInput );

  vbox->addWidget( mOffsetLabel );
  vbox->addWidget( mOffsetInput );

  Q3ButtonGroup *group = new Q3ButtonGroup( i18n("Options"), mainWidget() );
  topLayout->addWidget( group, 10 );


  QGridLayout *gbox = new QGridLayout( group );
  gbox->setSpacing( spacingHint() );
  gbox->addItem( new QSpacerItem( 0, fontMetrics().lineSpacing() ), 0, 0 );

  mCheckPattern = new QCheckBox( i18n("R&epeat pattern"), group );
  gbox->addWidget( mCheckPattern, 1, 0 );
  mCheckOnCursor = new QCheckBox( i18n("&Insert on cursor position"), group );
  gbox->addWidget( mCheckOnCursor, 2, 0 );
  connect( mCheckOnCursor, SIGNAL(clicked()), SLOT(cursorCheck()) );
  gbox->setRowStretch( 3, 10 );

  KConfig &config = *KGlobal::config();
  KConfigGroup gr = config.group("Insert Pattern Dialog");
  mCheckPattern->setChecked( gr.readEntry( "RepeatPattern", false ) );
  mCheckOnCursor->setChecked( gr.readEntry( "InsertOnCursor", false) );
  cursorCheck();
  uint val = gr.readEntry( "Format", 0 );
  mPatternSelector->setCurrentIndex( qMin(4u,val) );
  patternSelectorChanged( mPatternSelector->currentIndex() );
}


CInsertDialog::~CInsertDialog( void )
{
  KConfig &config = *KGlobal::config();
  KConfigGroup gr = config.group("Insert Pattern Dialog");
  gr.writeEntry( "RepeatPattern", mCheckPattern->isChecked() );
  gr.writeEntry( "InsertOnCursor", mCheckOnCursor->isChecked() );
  gr.writeEntry( "Format", mPatternSelector->currentIndex() );
  gr.sync();
}


void CInsertDialog::showEvent( QShowEvent *e )
{
  KDialog::showEvent(e);
  mPatternInput->setFocus();
}


void CInsertDialog::patternSelectorChanged( int index )
{
  mPatternValidator->setState( (CHexValidator::EState)index );
  mPatternInput->setText( mPatternString[ index ] );
}


void CInsertDialog::patternInputChanged( const QString &text )
{
  mPatternString[ mPatternSelector->currentIndex() ] = text;
  mPatternValidator->convert( mPatternData,
    mPatternString[ mPatternSelector->currentIndex() ] );
}


void CInsertDialog::slotOk( void )
{
  if( mPatternData.isEmpty() == true )
  {
    showEntryFailure( this, QString("") );
    return;
  }

  SInsertData id;
  id.size = mSizeBox->value();
  id.repeatPattern = mCheckPattern->isChecked();
  id.onCursor = mCheckOnCursor->isChecked();
  id.pattern = mPatternData;

  if( id.onCursor == false )
  {
    bool success = stringToOffset( mOffsetInput->text(), id.offset );
    if( success == false )
    {
      showEntryFailure( this, QString("") );
      return;
    }
  }

  hide();
  execute( id );
}


void CInsertDialog::cursorCheck( void )
{
  bool state = mCheckOnCursor->isChecked() == true ? false : true;
  mOffsetLabel->setEnabled( state );
  mOffsetInput->setEnabled( state );
}




void centerDialog( QWidget *widget, QWidget *centerParent )
{
  if( centerParent == 0 || widget == 0 )
  {
    return;
  }

  QPoint point = centerParent->mapToGlobal( QPoint(0,0) );
  QRect pos    = centerParent->geometry();

  widget->setGeometry( point.x() + pos.width()/2  - widget->width()/2,
		       point.y() + pos.height()/2 - widget->height()/2,
		       widget->width(), widget->height() );
}


void centerDialogBottom( QWidget *widget, QWidget *centerParent )
{
  if( centerParent == 0 || widget == 0 )
  {
    return;
  }

  QPoint point = centerParent->mapToGlobal( QPoint(0,0) );
  QRect pos    = centerParent->geometry();

  widget->setGeometry( point.x() + pos.width()/2  - widget->width()/2,
		       point.y() + pos.height() - widget->height(),
		       widget->width(), widget->height() );
}


void comboMatchText( KComboBox *combo, const QString &text )
{
  for( int i=0; i < combo->count(); i++ )
  {
    if( combo->itemText(i) == text )
    {
      combo->setCurrentIndex(i);
      return;
    }
  }
  combo->setCurrentIndex(0);
}





void showEntryFailure( QWidget *parent, const QString &msg )
{
  QString message;
  message += i18n("Your request can not be processed.");
  message += '\n';
  if( msg.isNull() || msg.isEmpty() )
  {
    message += i18n("Examine argument(s) and try again.");
  }
  else
  {
    message += msg;
  }
  KMessageBox::sorry( parent, message, i18n("Invalid argument(s)") );
}


bool verifyFileDestnation( QWidget *parent, const QString &title,
			   const QString &path )
{
  if( path.isEmpty() == true )
  {
    QString msg = i18n("You must specify a destination file.");
    KMessageBox::sorry( parent, msg, title );
    return( false );
  }

  QFileInfo info( path );
  if( info.exists() == true )
  {
    if( info.isDir() == true )
    {
      QString msg = i18n("You have specified an existing folder.");
      KMessageBox::sorry( parent, msg, title );
      return( false );
    }

    if( info.isWritable() == false )
    {
      QString msg = i18n("You do not have write permission to this file.");
      KMessageBox::sorry( parent, msg, title );
      return( false );
    }

    QString msg = i18n( ""
      "You have specified an existing file.\n"
      "Overwrite current file?" );
    int reply = KMessageBox::warningContinueCancel( parent, msg, title, KStandardGuiItem::overwrite() );
    if( reply != KMessageBox::Continue )
    {
      return( false );
    }
  }

  return( true );
}



bool stringToOffset( const QString & text, uint &offset )
{
  if( text.isEmpty() )
  {
    return( false );
  }

  const char * p = text.toAscii(); // ####: Is this correct?

  //
  // Skip any whitespaces in front of string
  //
  for( ; *p != 0 && isspace( *p ) ; p++ );

  int match = 0;
  bool space = false;
  if( strncmp( p, "0x", 2 ) == 0 || strncmp( p, "0X", 2 ) == 0 )
  {
    for( const char *q = p+2; *q != 0; q++ )
    {
      if( isxdigit( *q ) == 0 || space == true )
      {
	if( isspace( *q ) == 0 )
	{
	  return( false );
	}
	space = true;
      }
    }
    match = sscanf( p+2, "%x", &offset );
  }
  else
  {
    for( const char *q = p; *q != 0; q++ )
    {
      if( isdigit( *q ) == 0 || space == true )
      {
	if( isspace( *q ) == 0 )
	{
	  return( false );
	}
	space = true;
      }
    }
    match = sscanf( p, "%u", &offset );
  }

  if( match == 0 )
  {
    return( false );
  }

  return( true );
}


static const QStringList &formatStrings( void )
{
  static QStringList list;
  if( list.isEmpty() == true )
  {
    list.append( i18n( "Hexadecimal" ) );
    list.append( i18n( "Decimal" ) );
    list.append( i18n( "Octal" ) );
    list.append( i18n( "Binary" ) );
    list.append( i18n( "Regular Text" ) );
  }
  return( list );
}


static const QStringList &operationStrings( void )
{
  static QStringList list;
  if( list.isEmpty() == true )
  {
    list.append( i18n( "operand AND data" ) );
    list.append( i18n( "operand OR data" ) );
    list.append( i18n( "operand XOR data" ) );
    list.append( i18n( "INVERT data" ) );
    list.append( i18n( "REVERSE data" ) );
    list.append( i18n( "ROTATE data" ) );
    list.append( i18n( "SHIFT data" ) );
    list.append( i18n( "Swap Individual Bits" ) );
  }

  return( list );
}





#include "dialog.moc"
