// $Id: imap4.cc,v 1.140.2.6 2004/03/19 22:57:53 tilladam Exp $
/**********************************************************************
 *
 *   imap4.cc  - IMAP4rev1 KIOSlave
 *   Copyright (C) 2001-2002  Michael Haeckel <haeckel@kde.org>
 *   Copyright (C) 1999  John Corey
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *   Send comments and bug fixes to jcorey@fruity.ath.cx
 *
 *********************************************************************/

/**
 * @class IMAP4Protocol
 * @note References:
 *   - RFC 2060 - Internet Message Access Protocol - Version 4rev1 - December 1996
 *   - RFC 2192 - IMAP URL Scheme - September 1997
 *   - RFC 1731 - IMAP Authentication Mechanisms - December 1994
 *              (Discusses KERBEROSv4, GSSAPI, and S/Key)
 *   - RFC 2195 - IMAP/POP AUTHorize Extension for Simple Challenge/Response
 *            - September 1997 (CRAM-MD5 authentication method)
 *   - RFC 2104 - HMAC: Keyed-Hashing for Message Authentication - February 1997
 *   .
 * Supported URLs:
 *   - imap://server/ - Prompt for user/pass, list all folders in home directory
 *   - imap://user:pass@server/ - Uses LOGIN to log in
 *   - imap://user;AUTH=method:pass@server/ - Uses AUTHENTICATE to log in
 *   - imap://server/folder/ - List messages in folder
 *
 * @note API notes:
 *   Not recieving the required write access for a folder means
 *       ERR_CANNOT_OPEN_FOR_WRITING.
 *  ERR_DOES_NOT_EXIST is reserved for folders.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "imap4.h"

#include "rfcdecoder.h"

#include <sys/stat.h>

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <qbuffer.h>
#include <qdatetime.h>
#include <kprotocolmanager.h>
#include <kmessagebox.h>
#include <kdebug.h>
#include <kio/connection.h>
#include <kio/slaveinterface.h>
#include <kio/passdlg.h>
#include <klocale.h>

using namespace KIO;

extern "C"
{
  void sigalrm_handler (int);
  int kdemain (int argc, char **argv);
}

int
kdemain (int argc, char **argv)
{
  kdDebug(7116) << "IMAP4::kdemain" << endl;

  KInstance instance ("kio_imap4");
  if (argc != 4)
  {
    fprintf(stderr, "Usage: kio_imap4 protocol domain-socket1 domain-socket2\n");
    ::exit (-1);
  }

  //set debug handler

  IMAP4Protocol *slave;
  if (strcasecmp (argv[1], "imaps") == 0)
    slave = new IMAP4Protocol (argv[2], argv[3], true);
  else if (strcasecmp (argv[1], "imap") == 0)
    slave = new IMAP4Protocol (argv[2], argv[3], false);
  else
    abort ();
  slave->dispatchLoop ();
  delete slave;

  return 0;
}

void
sigchld_handler (int signo)
{
  int pid, status;

  while (true && signo == SIGCHLD)
  {
    pid = waitpid (-1, &status, WNOHANG);
    if (pid <= 0)
    {
      // Reinstall signal handler, since Linux resets to default after
      // the signal occurred ( BSD handles it different, but it should do
      // no harm ).
      signal (SIGCHLD, sigchld_handler);
      return;
    }
  }
}

static const QString hidePass(KURL aUrl)
{
  aUrl.setPass(QString::null);
  return KURL::decode_string(aUrl.prettyURL());
}

IMAP4Protocol::IMAP4Protocol (const QCString & pool, const QCString & app, bool isSSL):TCPSlaveBase ((isSSL ? 993 : 143), (isSSL ? "imaps" : "imap4"), pool,
              app, isSSL), imapParser (),
mimeIO ()
{
  mySSL = isSSL;
  readBuffer[0] = 0x00;
  relayEnabled = false;
  readBufferLen = 0;
  cacheOutput = false;
  mTimeOfLastNoop = QDateTime();
}

IMAP4Protocol::~IMAP4Protocol ()
{
  closeDescriptor();
  kdDebug(7116) << "IMAP4: Finishing" << endl;
}

void
IMAP4Protocol::get (const KURL & _url)
{
  if (!makeLogin()) return;
  kdDebug(7116) << "IMAP4::get -  " << hidePass(_url) << endl;
  QString aBox, aSequence, aType, aSection, aValidity, aDelimiter;
  enum IMAP_TYPE aEnum =
    parseURL (_url, aBox, aSection, aType, aSequence, aValidity, aDelimiter);

  if (aSequence == "0:0" && getState() == ISTATE_SELECT)
  {
    imapCommand *cmd = doCommand (imapCommand::clientNoop());
    completeQueue.removeRef(cmd);
  }

  if (aSequence.isEmpty ())
  {
    aSequence = "1:*";
  }

  mProcessedSize = 0;
  imapCommand *cmd = NULL;
  if (!assureBox (aBox, true)) return;

#ifdef USE_VALIDITY
  if (selectInfo.uidValidityAvailable () && !aValidity.isEmpty ()
      && selectInfo.uidValidity () != aValidity.toULong ())
  {
    // this url is stale
    error (ERR_COULD_NOT_READ, hidePass(_url));
  }
  else
#endif
  {
    if (aSection.find ("STRUCTURE", 0, false) != -1)
    {
      aSection = "BODYSTRUCTURE";
    }
    else if (aSection.find ("ENVELOPE", 0, false) != -1)
    {
      aSection = "UID ENVELOPE";
      if (hasCapability("IMAP4rev1")) {
        aSection += " BODY.PEEK[HEADER.FIELDS (REFERENCES)]";
      } else {
        // imap4 does not know HEADER.FIELDS
        aSection += " RFC822.HEADER.LINES (REFERENCES)";
      }
    }
    else if (aSection == "HEADER")
    {
      aSection = "UID RFC822.HEADER";
    }
    else if (aSection.find ("BODY.PEEK[", 0, false) != -1) 
    {
      if (aSection.find ("BODY.PEEK[]", 0, false) != -1 &&
          !hasCapability("IMAP4rev1")) {
        // imap4 does not know BODY.PEEK[]
        aSection.replace("BODY.PEEK[]", "RFC822.PEEK");
      }
      aSection.prepend("UID RFC822.SIZE INTERNALDATE FLAGS ");
    }
	else
    {
      if (aSection.isEmpty()) 
         aSection = "UID RFC822";
      else if (aSection == "FLAGS" ) 
         ; /*aSection = "UID FLAGS";*/
      else
         aSection = "UID BODY[" + aSection + "]";
    }
    if (aEnum == ITYPE_BOX || aEnum == ITYPE_DIR_AND_BOX)
    {
      if( aSection != "FLAGS" ) aSection += " RFC822.SIZE INTERNALDATE FLAGS";

      // write the digest header
      outputLine
        ("Content-Type: multipart/digest; boundary=\"IMAPDIGEST\"\r\n");
      if (selectInfo.recentAvailable ())
        outputLineStr ("X-Recent: " +
                       QString ().setNum (selectInfo.recent ()) + "\r\n");
      if (selectInfo.countAvailable ())
        outputLineStr ("X-Count: " + QString ().setNum (selectInfo.count ()) +
                       "\r\n");
      if (selectInfo.unseenAvailable ())
        outputLineStr ("X-Unseen: " +
                       QString ().setNum (selectInfo.unseen ()) + "\r\n");
      if (selectInfo.uidValidityAvailable ())
        outputLineStr ("X-uidValidity: " +
                       QString ().setNum (selectInfo.uidValidity ()) +
                       "\r\n");
      if (selectInfo.uidNextAvailable ())
        outputLineStr ("X-UidNext: " +
                       QString ().setNum (selectInfo.uidNext ()) + "\r\n");
      if (selectInfo.flagsAvailable ())
        outputLineStr ("X-Flags: " + QString ().setNum (selectInfo.flags ()) +
                       "\r\n");
      if (selectInfo.permanentFlagsAvailable ())
        outputLineStr ("X-PermanentFlags: " +
                       QString ().setNum (selectInfo.permanentFlags ()) +
                       "\r\n");
      if (selectInfo.readWriteAvailable ())
        outputLineStr (QString ("X-Access: ") +
                       (selectInfo.readWrite ()? "Read/Write" : "Read only") +
                       "\r\n");
      outputLine ("\r\n");
    }

    if (aEnum == ITYPE_MSG)
      relayEnabled = true;

    if (aSequence != "0:0")
    {
      if (getLastHandled())
        getLastHandled()->clear();
      cmd = sendCommand (imapCommand::clientFetch (aSequence, aSection));
      int res;
      do
      {
        while (!(res = parseLoop()));
        if (res == -1) break;

        mailHeader *lastone = NULL;
        imapCache *cache;
        cache = getLastHandled ();
        if (cache)
          lastone = cache->getHeader ();

        if (!cmd->isComplete ())
        {
          if ((aSection.find ("BODYSTRUCTURE", 0, false) != -1)
                    || (aSection.find ("FLAGS", 0, false) != -1)
                    || (aSection.find ("UID", 0, false) != -1)
                    || (aSection.find ("ENVELOPE", 0, false) != -1)
                    || (aSection.find ("BODY.PEEK[0]", 0, false) != -1
                        && (aEnum == ITYPE_BOX || aEnum == ITYPE_DIR_AND_BOX)))
          {
            if (aEnum == ITYPE_BOX || aEnum == ITYPE_DIR_AND_BOX)
            {
              // write the mime header (default is here message/rfc822)
              outputLine ("--IMAPDIGEST\r\n");
              cacheOutput = true;
              if (cache->getUid () != 0)
                outputLineStr ("X-UID: " +
                               QString ().setNum (cache->getUid ()) + "\r\n");
              if (cache->getSize () != 0)
                outputLineStr ("X-Length: " +
                               QString ().setNum (cache->getSize ()) + "\r\n");
              if (!cache->getDate ().isEmpty())
                outputLineStr ("X-Date: " + cache->getDate () + "\r\n");
              if (cache->getFlags () != 0)
                outputLineStr ("X-Flags: " +
                               QString ().setNum (cache->getFlags ()) + "\r\n");
            } else cacheOutput = true;
            if( lastone ) lastone->outputPart (*this);
            cacheOutput = false;
            flushOutput();
          }
        }
      }
      while (!cmd->isComplete ());
      if (aEnum == ITYPE_BOX || aEnum == ITYPE_DIR_AND_BOX)
      {
        // write the end boundary
        outputLine ("--IMAPDIGEST--\r\n");
      }

      completeQueue.removeRef (cmd);
    }
  }

  // just to keep everybody happy when no data arrived
  data (QByteArray ());

  finished ();
  relayEnabled = false;
  kdDebug(7116) << "IMAP4::get -  finished" << endl;
}

void
IMAP4Protocol::listDir (const KURL & _url)
{
  kdDebug(7116) << "IMAP4::listDir - " << hidePass(_url) << endl;

  if (_url.path().isEmpty())
  {
    KURL url = _url;
    url.setPath("/");
    redirection( url );
    finished();
    return;
  }

  QString myBox, mySequence, myLType, mySection, myValidity, myDelimiter;
  enum IMAP_TYPE myType =
    parseURL (_url, myBox, mySection, myLType, mySequence, myValidity,
      myDelimiter);

  if (!makeLogin()) return;

  if (myType == ITYPE_DIR || myType == ITYPE_DIR_AND_BOX)
  {
    QString listStr = myBox;
    imapCommand *cmd;

    if (!listStr.isEmpty ())
      listStr += myDelimiter;
    listStr += "%";
    cmd =
      doCommand (imapCommand::clientList ("", listStr, myLType == "LSUB"));
    if (cmd->result () == "OK")
    {
      QString mailboxName;
      UDSEntry entry;
      UDSAtom atom;
      KURL aURL = _url;
      if (aURL.path().find(";") != -1)
        aURL.setPath(aURL.path().left(aURL.path().find(";")));

      kdDebug(7116) << "IMAP4Protocol::listDir - got " << listResponses.count () << endl;

      if (myLType == "LSUB")
      {
        // fire the same command as LIST to check if the box really exists
        QValueList<imapList> listResponsesSave = listResponses;
        doCommand (imapCommand::clientList ("", listStr, false));
        for (QValueListIterator < imapList > it = listResponsesSave.begin ();
            it != listResponsesSave.end (); ++it)
        {
          bool boxOk = false;
          for (QValueListIterator < imapList > it2 = listResponses.begin ();
              it2 != listResponses.end (); ++it2)
          {
            if ((*it2).name() == (*it).name())
            {
              boxOk = true;
              // copy the flags from the LIST-command
              (*it) = (*it2);
              break;
            }
          }
          if (boxOk)
            doListEntry (aURL, myBox, (*it));
          else
            kdDebug(7116) << "IMAP4Protocol::listDir - suppress " << (*it).name() << endl;
        }
        listResponses = listResponsesSave;
      }
      else
      {
        for (QValueListIterator < imapList > it = listResponses.begin ();
            it != listResponses.end (); ++it)
        {
          doListEntry (aURL, myBox, (*it));
        }
      }
      entry.clear ();
      listEntry (entry, true);
    }
    else
    {
      error (ERR_CANNOT_ENTER_DIRECTORY, hidePass(_url));
    }
    completeQueue.removeRef (cmd);
  }
  if ((myType == ITYPE_BOX || myType == ITYPE_DIR_AND_BOX)
      && myLType != "LIST" && myLType != "LSUB")
  {
    if (!_url.query ().isEmpty ())
    {
      QString query = KURL::decode_string (_url.query ());
      query = query.right (query.length () - 1);
      if (!query.isEmpty())
      {
        imapCommand *cmd = NULL;

        if (!assureBox (myBox, true)) return;

        if (!selectInfo.countAvailable() || selectInfo.count())
        {
          cmd = doCommand (imapCommand::clientSearch (query));
          if (cmd->result() != "OK")
          {
            error(ERR_UNSUPPORTED_ACTION, hidePass(_url));
            completeQueue.removeRef (cmd);
            return;
          }
          completeQueue.removeRef (cmd);

          QStringList list = getResults ();
          int stretch = 0;

          if (selectInfo.uidNextAvailable ())
            stretch = QString ().setNum (selectInfo.uidNext ()).length ();
          UDSEntry entry;
          imapCache fake;

          for (QStringList::Iterator it = list.begin (); it != list.end ();
               ++it)
          {
            fake.setUid((*it).toULong());
            doListEntry (_url, stretch, &fake);
          }
          entry.clear ();
          listEntry (entry, true);
        }
      }
    }
    else
    {
      if (!assureBox (myBox, true)) return;

      kdDebug(7116) << "IMAP4: select returned:" << endl;
      if (selectInfo.recentAvailable ())
        kdDebug(7116) << "Recent: " << selectInfo.recent () << "d" << endl;
      if (selectInfo.countAvailable ())
        kdDebug(7116) << "Count: " << selectInfo.count () << "d" << endl;
      if (selectInfo.unseenAvailable ())
        kdDebug(7116) << "Unseen: " << selectInfo.unseen () << "d" << endl;
      if (selectInfo.uidValidityAvailable ())
        kdDebug(7116) << "uidValidity: " << selectInfo.uidValidity () << "d" << endl;
      if (selectInfo.flagsAvailable ())
        kdDebug(7116) << "Flags: " << selectInfo.flags () << "d" << endl;
      if (selectInfo.permanentFlagsAvailable ())
        kdDebug(7116) << "PermanentFlags: " << selectInfo.permanentFlags () << "d" << endl;
      if (selectInfo.readWriteAvailable ())
        kdDebug(7116) << "Access: " << (selectInfo.readWrite ()? "Read/Write" : "Read only") << endl;

#ifdef USE_VALIDITY
      if (selectInfo.uidValidityAvailable ()
          && selectInfo.uidValidity () != myValidity.toULong ())
      {
        //redirect
        KURL newUrl = _url;

        newUrl.setPath ("/" + myBox + ";UIDVALIDITY=" +
                        QString ().setNum (selectInfo.uidValidity ()));
        kdDebug(7116) << "IMAP4::listDir - redirecting to " << hidePass(newUrl) << endl;
        redirection (newUrl);


      }
      else
#endif
      if (selectInfo.count () > 0)
      {
        int stretch = 0;

        if (selectInfo.uidNextAvailable ())
          stretch = QString ().setNum (selectInfo.uidNext ()).length ();
        //        kdDebug(7116) << selectInfo.uidNext() << "d used to stretch " << stretch << endl;
        UDSEntry entry;

        if (mySequence.isEmpty()) mySequence = "1:*";

        bool withSubject = mySection.isEmpty();
        if (mySection.isEmpty()) mySection = "UID RFC822.SIZE ENVELOPE";

        bool withFlags = mySection.upper().find("FLAGS") != -1;
        imapCommand *fetch =
          sendCommand (imapCommand::
                       clientFetch (mySequence, mySection));
      imapCache *cache;
        do
        {
          while (!parseLoop ());

          cache = getLastHandled ();

          if (cache && !fetch->isComplete())
            doListEntry (_url, stretch, cache, withFlags, withSubject);
        }
        while (!fetch->isComplete ());
        entry.clear ();
        listEntry (entry, true);
      }
    }
  }

  kdDebug(7116) << "IMAP4Protcol::listDir - Finishing listDir" << endl;
  finished ();
}

void
IMAP4Protocol::setHost (const QString & _host, int _port,
                        const QString & _user, const QString & _pass)
{
  if (myHost != _host || myPort != _port || myUser != _user)
  {
    if (!myHost.isEmpty ())
      closeConnection ();
    myHost = _host;
    myPort = _port;
    myUser = _user;
    myPass = _pass;
  }
}

void
IMAP4Protocol::parseRelay (const QByteArray & buffer)
{
  if (relayEnabled)  {
    data( buffer );
    mProcessedSize += buffer.size();
    processedSize( mProcessedSize );
  }
}

void
IMAP4Protocol::parseRelay (ulong len)
{
  if (relayEnabled)
    totalSize (len);
}


bool IMAP4Protocol::parseRead(QByteArray & buffer, ulong len, ulong relay)
{
  char buf[8192];
  while (buffer.size() < len)
  {
    ssize_t readLen = myRead(buf, QMIN(len - buffer.size(), sizeof(buf) - 1));
    if (readLen == 0)
    {
      error (ERR_CONNECTION_BROKEN, myHost);
      setState(ISTATE_CONNECT);
      closeConnection();
      return FALSE;
    }
    if (relay > buffer.size())
    {
      QByteArray relayData;
      ssize_t relbuf = relay - buffer.size();
      int currentRelay = QMIN(relbuf, readLen);
      relayData.setRawData(buf, currentRelay);
      parseRelay(relayData);
      relayData.resetRawData(buf, currentRelay);
    }
    {
      QBuffer stream (buffer);
      stream.open (IO_WriteOnly);
      stream.at (buffer.size ());
      stream.writeBlock (buf, readLen);
      stream.close ();
    }
  }
  return (buffer.size() == len);
}


bool IMAP4Protocol::parseReadLine (QByteArray & buffer, ulong relay)
{
  if (myHost.isEmpty()) return FALSE;

  while (true) {
    ssize_t copyLen = 0;
    if (readBufferLen > 0)
    {
      while (copyLen < readBufferLen && readBuffer[copyLen] != '\n') copyLen++;
      if (copyLen < readBufferLen) copyLen++;
      if (relay > 0)
      {
        QByteArray relayData;

        if (copyLen < (ssize_t) relay)
          relay = copyLen;
        relayData.setRawData (readBuffer, relay);
        parseRelay (relayData);
        relayData.resetRawData (readBuffer, relay);
        kdDebug(7116) << "relayed : " << relay << "d" << endl;
      }
      // append to buffer
      {
        QBuffer stream (buffer);

        stream.open (IO_WriteOnly);
        stream.at (buffer.size ());
        stream.writeBlock (readBuffer, copyLen);
        stream.close ();
//        kdDebug(7116) << "appended " << copyLen << "d got now " << buffer.size() << endl;
      }

      readBufferLen -= copyLen;
      if (readBufferLen) memcpy(readBuffer, &readBuffer[copyLen], readBufferLen);
      if (buffer[buffer.size() - 1] == '\n') return TRUE;
    }
    if (!isConnectionValid())
    {
      error (ERR_CONNECTION_BROKEN, myHost);
      setState(ISTATE_CONNECT);
      closeConnection();
      return FALSE;
    }
    waitForResponse(600);
    readBufferLen = read(readBuffer, IMAP_BUFFER - 1);
    if (readBufferLen == 0)
    {
      error (ERR_CONNECTION_BROKEN, myHost);
      setState(ISTATE_CONNECT);
      closeConnection();
      return FALSE;
    }
  }
}


void
IMAP4Protocol::mimetype (const KURL & _url)
{
  kdDebug(7116) << "IMAP4::mimetype - " << hidePass(_url) << endl;
  QString aBox, aSequence, aType, aSection, aValidity, aDelimiter;

  mimeType (getMimeType(parseURL (_url, aBox, aSection, aType, aSequence,
            aValidity, aDelimiter)));
  finished ();
}

void
IMAP4Protocol::setSubURL (const KURL & _url)
{
  kdDebug(7116) << "IMAP4::setSubURL - " << hidePass(_url) << endl;
  KIO::TCPSlaveBase::setSubURL (_url);
}

void
IMAP4Protocol::put (const KURL & _url, int, bool, bool)
{
  kdDebug(7116) << "IMAP4::put - " << hidePass(_url) << endl;
//  KIO::TCPSlaveBase::put(_url,permissions,overwrite,resume);
  QString aBox, aSequence, aLType, aSection, aValidity, aDelimiter;
  enum IMAP_TYPE aType =
    parseURL (_url, aBox, aSection, aLType, aSequence, aValidity, aDelimiter);

  // see if it is a box
  if (aType != ITYPE_BOX && aType != ITYPE_DIR_AND_BOX)
  {
    if (aBox[aBox.length () - 1] == '/')
      aBox = aBox.right (aBox.length () - 1);
    imapCommand *cmd = doCommand (imapCommand::clientCreate (aBox));

    if (cmd->result () != "OK")
      error (ERR_COULD_NOT_WRITE, hidePass(_url));
    completeQueue.removeRef (cmd);
  }
  else
  {
    QPtrList < QByteArray > bufferList;
    int length = 0;

    int result;
    // Loop until we got 'dataEnd'
    do
    {
      QByteArray *buffer = new QByteArray ();
      dataReq ();               // Request for data
      result = readData (*buffer);
      if (result > 0)
      {
        bufferList.append (buffer);
        length += result;
      } else {
	delete buffer;
      }
    }
    while (result > 0);

    if (result != 0)
    {
      error (ERR_ABORTED, hidePass(_url));
      finished ();
      return;
    }

    imapCommand *cmd =
      sendCommand (imapCommand::clientAppend (aBox, aSection, length));
    while (!parseLoop ());

    // see if server is waiting
    if (!cmd->isComplete () && !getContinuation ().isEmpty ())
    {
      bool sendOk = true;
      ulong wrote = 0;

      QByteArray *buffer;
      while (!bufferList.isEmpty () && sendOk)
      {
        buffer = bufferList.take (0);

        sendOk =
          (write (buffer->data (), buffer->size ()) ==
           (ssize_t) buffer->size ());
        wrote += buffer->size ();
        delete buffer;
        if (!sendOk)
        {
          error (ERR_CONNECTION_BROKEN, myHost);
          completeQueue.removeRef (cmd);
          setState(ISTATE_CONNECT);
          closeConnection();
          return;
        }
      }
      parseWriteLine ("");
      while (!cmd->isComplete ())
        parseLoop ();
      if (cmd->result () != "OK")
        error (ERR_COULD_NOT_WRITE, myHost);
      else
      {
        if (hasCapability("UIDPLUS"))
        {
          QString uid = cmd->resultInfo();
          if (uid.find("APPENDUID") != -1)
          {
            uid = uid.section(" ", 2, 2);
            uid = uid.left(uid.length()-1);
            infoMessage("UID "+uid);
          }
        }
        // MUST reselect to get the new message
        else if (aBox == getCurrentBox ())
        {
          cmd =
            doCommand (imapCommand::
                       clientSelect (aBox, !selectInfo.readWrite ()));
          completeQueue.removeRef (cmd);
        }
      }
    }
    else
    {
      error (ERR_COULD_NOT_WRITE, myHost);
    }

    completeQueue.removeRef (cmd);
  }

  finished ();
}

void
IMAP4Protocol::mkdir (const KURL & _url, int)
{
  kdDebug(7116) << "IMAP4::mkdir - " << hidePass(_url) << endl;
  QString path = _url.path();
  int slash = path.findRev('/', (path.at(path.length() - 1) == '/') ?
    (int)path.length() - 2 : -1);
  KURL parentUrl = _url;
  QString newBox;
  if (slash != -1)
  {
    parentUrl.setPath(path.left(slash) + ";TYPE=LIST");
    newBox = path.mid(slash + 1);
  }
  QString aBox, aSequence, aLType, aSection, aValidity, aDelimiter;
  parseURL(parentUrl, aBox, aSection, aLType, aSequence, aValidity, aDelimiter);
/*  if (aBox[aBox.length () - 1] != '/')
    aBox += "/"; */
  newBox = (newBox.isEmpty()) ? aBox : aBox + aDelimiter + newBox;
  imapCommand *cmd = doCommand (imapCommand::clientCreate(newBox));

  if (cmd->result () != "OK")
  {
    error (ERR_COULD_NOT_MKDIR, hidePass(_url));
    completeQueue.removeRef (cmd);
    return;
  }
  completeQueue.removeRef (cmd);

  enum IMAP_TYPE type =
    parseURL(_url, aBox, aSection, aLType, aSequence, aValidity, aDelimiter);
  kdDebug(7116) << "IMAP4::mkdir - parseURL of " << _url << " returned type " << type << endl;
  if (type == ITYPE_BOX)
  {
    if (messageBox(QuestionYesNo,
      i18n("What do you want to store in this folder?"), i18n("Create Folder"),
      i18n("&Messages"), i18n("&Subfolders")) == KMessageBox::No)
    {
      cmd = doCommand(imapCommand::clientDelete(newBox));
      completeQueue.removeRef (cmd);
      cmd = doCommand(imapCommand::clientCreate(newBox + aDelimiter));
      if (cmd->result () != "OK")
      {
        error (ERR_COULD_NOT_MKDIR, hidePass(_url));
        completeQueue.removeRef (cmd);
        return;
      }
      completeQueue.removeRef (cmd);
    }
  }

  cmd = doCommand(imapCommand::clientSubscribe(newBox));
  completeQueue.removeRef(cmd);

  finished ();
}

void
IMAP4Protocol::copy (const KURL & src, const KURL & dest, int, bool overwrite)
{
  kdDebug(7116) << "IMAP4::copy - [" << (overwrite ? "Overwrite" : "NoOverwrite") << "] " << hidePass(src) << " -> " << hidePass(dest) << endl;
  QString sBox, sSequence, sLType, sSection, sValidity, sDelimiter;
  QString dBox, dSequence, dLType, dSection, dValidity, dDelimiter;
  enum IMAP_TYPE sType =
    parseURL (src, sBox, sSection, sLType, sSequence, sValidity, sDelimiter);
  enum IMAP_TYPE dType =
    parseURL (dest, dBox, dSection, dLType, dSequence, dValidity, dDelimiter);

  // see if we have to create anything
  if (dType != ITYPE_BOX && dType != ITYPE_DIR_AND_BOX)
  {
    // this might be konqueror
    int sub = dBox.find (sBox);

    // might be moving to upper folder
    if (sub > 0)
    {
      KURL testDir = dest;

      QString subDir = dBox.right (dBox.length () - dBox.findRev ("/"));
      QString topDir = dBox.left (sub);
      testDir.setPath ("/" + topDir);
      dType =
        parseURL (testDir, topDir, dSection, dLType, dSequence, dValidity,
          dDelimiter);

      kdDebug(7116) << "IMAP4::copy - checking this destination " << topDir << endl;
      // see if this is what the user wants
      if (dType == ITYPE_BOX || dType == ITYPE_DIR_AND_BOX)
      {
        kdDebug(7116) << "IMAP4::copy - assuming this destination " << topDir << endl;
        dBox = topDir;
      }
      else
      {

        // maybe if we create a new mailbox
        topDir = "/" + topDir + subDir;
        testDir.setPath (topDir);
        kdDebug(7116) << "IMAP4::copy - checking this destination " << topDir << endl;
        dType =
          parseURL (testDir, topDir, dSection, dLType, dSequence, dValidity,
            dDelimiter);
        if (dType != ITYPE_BOX && dType != ITYPE_DIR_AND_BOX)
        {
          // ok then we'll create a mailbox
          imapCommand *cmd = doCommand (imapCommand::clientCreate (topDir));

          // on success we'll use it, else we'll just try to create the given dir
          if (cmd->result () == "OK")
          {
            kdDebug(7116) << "IMAP4::copy - assuming this destination " << topDir << endl;
            dType = ITYPE_BOX;
            dBox = topDir;
          }
          else
          {
            completeQueue.removeRef (cmd);
            cmd = doCommand (imapCommand::clientCreate (dBox));
            if (cmd->result () == "OK")
              dType = ITYPE_BOX;
            else
              error (ERR_COULD_NOT_WRITE, hidePass(dest));
          }
          completeQueue.removeRef (cmd);
        }
      }

    }
  }
  if (sType == ITYPE_MSG || sType == ITYPE_BOX)
  {
    //select the source box
    if (!assureBox(sBox, true)) return;
    kdDebug(7116) << "IMAP4::copy - " << sBox << " -> " << dBox << endl;

    //issue copy command
    imapCommand *cmd =
      doCommand (imapCommand::clientCopy (dBox, sSequence));
    if (cmd->result () != "OK")
    {
      error (ERR_COULD_NOT_WRITE, hidePass(dest));
    } else {
      if (hasCapability("UIDPLUS"))
      {
        QString uid = cmd->resultInfo();
        if (uid.find("COPYUID") != -1)
        {
          uid = uid.section(" ", 2, 3);
          uid = uid.left(uid.length()-1);
          infoMessage("UID "+uid);
        }
      }
    }
    completeQueue.removeRef (cmd);
  }
  else
  {
    error (ERR_ACCESS_DENIED, hidePass(src));
  }
  finished ();
}

void
IMAP4Protocol::del (const KURL & _url, bool isFile)
{
  kdDebug(7116) << "IMAP4::del - [" << (isFile ? "File" : "NoFile") << "] " << hidePass(_url) << endl;
  QString aBox, aSequence, aLType, aSection, aValidity, aDelimiter;
  enum IMAP_TYPE aType =
    parseURL (_url, aBox, aSection, aLType, aSequence, aValidity, aDelimiter);

  switch (aType)
  {
  case ITYPE_BOX:
  case ITYPE_DIR_AND_BOX:
    if (!aSequence.isEmpty ())
    {
      if (aSequence == "*")
      {
        if (!assureBox (aBox, false)) return;
        imapCommand *cmd = doCommand (imapCommand::clientExpunge ());
        if (cmd->result () != "OK")
          error (ERR_CANNOT_DELETE, hidePass(_url));
        completeQueue.removeRef (cmd);
      }
      else
      {
        // if open for read/write
        if (!assureBox (aBox, false)) return;
        imapCommand *cmd =
          doCommand (imapCommand::
                     clientStore (aSequence, "+FLAGS.SILENT", "\\DELETED"));
        if (cmd->result () != "OK")
          error (ERR_CANNOT_DELETE, hidePass(_url));
        completeQueue.removeRef (cmd);
      }
    }
    else
    {
      if (getCurrentBox() == aBox)
      {
        imapCommand *cmd = doCommand(imapCommand::clientClose());
        completeQueue.removeRef(cmd);
        setState(ISTATE_LOGIN);
      }
      // We unsubscribe, otherwise we get ghost folders on UW-IMAP
      imapCommand *cmd = doCommand(imapCommand::clientUnsubscribe(aBox));
      completeQueue.removeRef(cmd);
      cmd = doCommand(imapCommand::clientDelete (aBox));
      // If this doesn't work, we try to empty the mailbox first
      if (cmd->result () != "OK")
      {
        completeQueue.removeRef(cmd);
        if (!assureBox(aBox, false)) return;
        bool stillOk = true;
        if (stillOk)
        {
          imapCommand *cmd = doCommand(
            imapCommand::clientStore("1:*", "+FLAGS.SILENT", "\\DELETED"));
          if (cmd->result () != "OK") stillOk = false;
          completeQueue.removeRef(cmd);
        }
        if (stillOk)
        {
          imapCommand *cmd = doCommand(imapCommand::clientClose());
          if (cmd->result () != "OK") stillOk = false;
          completeQueue.removeRef(cmd);
          setState(ISTATE_LOGIN);
        }
        if (stillOk)
        {
          imapCommand *cmd = doCommand (imapCommand::clientDelete(aBox));
          if (cmd->result () != "OK") stillOk = false;
          completeQueue.removeRef(cmd);
        }
        if (!stillOk)
        {
          error (ERR_COULD_NOT_RMDIR, hidePass(_url));
          return;
        }
      } else {
        completeQueue.removeRef (cmd);
      }
    }
    break;

  case ITYPE_DIR:
    {
      imapCommand *cmd = doCommand (imapCommand::clientDelete (aBox));
      if (cmd->result () != "OK")
        error (ERR_COULD_NOT_RMDIR, hidePass(_url));
      completeQueue.removeRef (cmd);
    }
    break;

  case ITYPE_MSG:
    {
      // if open for read/write
      if (!assureBox (aBox, false)) return;
      imapCommand *cmd =
        doCommand (imapCommand::
                   clientStore (aSequence, "+FLAGS.SILENT", "\\DELETED"));
      if (cmd->result () != "OK")
        error (ERR_CANNOT_DELETE, hidePass(_url));
      completeQueue.removeRef (cmd);
    }
    break;

  case ITYPE_UNKNOWN:
    error (ERR_CANNOT_DELETE, hidePass(_url));
    break;
  }
  finished ();
}

void
IMAP4Protocol::special (const QByteArray & aData)
{
  if (!makeLogin()) return;

  kdDebug(7116) << "IMAP4Protocol::special" << endl;
  QDataStream stream(aData, IO_ReadOnly);

  int tmp;
  stream >> tmp;

  if (tmp  == 'C')
  {
    KURL src;
    KURL dest;
    stream >> src >> dest;
    copy(src, dest, 0, FALSE);
  }
  else if (tmp == 'c')
  {
    infoMessage(imapCapabilities.join(" "));
    finished();
  }
  else if (tmp == 'N')
  {
    imapCommand *cmd = doCommand(imapCommand::clientNoop());
    completeQueue.removeRef (cmd);
    finished();
  }
  else if (tmp == 'U')
  {
    // unsubscribe
    KURL _url;
    stream >> _url;
    QString aBox, aSequence, aLType, aSection, aValidity, aDelimiter;
    parseURL (_url, aBox, aSection, aLType, aSequence, aValidity, aDelimiter);
    imapCommand *cmd = doCommand(imapCommand::clientUnsubscribe(aBox));
    if (cmd->result () != "OK")
    {
      error(ERR_NO_CONTENT, i18n("Unsubscribe of folder %1 "
            "failed.").arg(hidePass(_url)));
      return;
    }
    completeQueue.removeRef (cmd);
    finished();
  }
  else if (tmp == 'u')
  {
    // subscribe
    KURL _url;
    stream >> _url;
    QString aBox, aSequence, aLType, aSection, aValidity, aDelimiter;
    parseURL (_url, aBox, aSection, aLType, aSequence, aValidity, aDelimiter);
    imapCommand *cmd = doCommand(imapCommand::clientSubscribe(aBox));
    if (cmd->result () != "OK")
    {
      error(ERR_NO_CONTENT, i18n("Subscribe of folder %1 "
            "failed.").arg(hidePass(_url)));
      return;
    }
    completeQueue.removeRef (cmd);
    finished();
  }
  else
  {
    // status ('S')
    KURL _url;
    QCString newFlags;
    stream >> _url >> newFlags;

    QString aBox, aSequence, aLType, aSection, aValidity, aDelimiter;
    parseURL (_url, aBox, aSection, aLType, aSequence, aValidity, aDelimiter);
    if (!assureBox(aBox, false)) return;
    imapCommand *cmd = doCommand (imapCommand::
      clientStore (aSequence, "-FLAGS.SILENT",
      "\\SEEN \\ANSWERED \\FLAGGED \\DRAFT"));
    if (cmd->result () != "OK")
    {
      error(ERR_NO_CONTENT, i18n("Changing the flags of message %1 "
        "failed.").arg(hidePass(_url)));
      return;
    }
    completeQueue.removeRef (cmd);
    if (!newFlags.isEmpty())
    {
      cmd = doCommand (imapCommand::
        clientStore (aSequence, "+FLAGS.SILENT", newFlags));
      if (cmd->result () != "OK")
      {
        error(ERR_NO_CONTENT, i18n("Changing the flags of message %1 "
          "failed.").arg(hidePass(_url)));
        return;
      }
      completeQueue.removeRef (cmd);
    }
    finished();
  }
}

void
IMAP4Protocol::rename (const KURL & src, const KURL & dest, bool overwrite)
{
  kdDebug(7116) << "IMAP4::rename - [" << (overwrite ? "Overwrite" : "NoOverwrite") << "] " << hidePass(src) << " -> " << hidePass(dest) << endl;
  QString sBox, sSequence, sLType, sSection, sValidity, sDelimiter;
  QString dBox, dSequence, dLType, dSection, dValidity, dDelimiter;
  enum IMAP_TYPE sType =
    parseURL (src, sBox, sSection, sLType, sSequence, sValidity, sDelimiter);
  enum IMAP_TYPE dType =
    parseURL (dest, dBox, dSection, dLType, dSequence, dValidity, dDelimiter);

  if (dType == ITYPE_UNKNOWN)
  {
    switch (sType)
    {
    case ITYPE_BOX:
    case ITYPE_DIR:
    case ITYPE_DIR_AND_BOX:
      {
        imapCommand *cmd = doCommand (imapCommand::clientRename (sBox, dBox));
        if (cmd->result () != "OK")
          error (ERR_CANNOT_RENAME, cmd->result ());
        completeQueue.removeRef (cmd);
      }
      break;

    case ITYPE_MSG:
    case ITYPE_UNKNOWN:
      error (ERR_CANNOT_RENAME, hidePass(src));
      break;
    }
  }
  else
  {
    error (ERR_CANNOT_RENAME, hidePass(src));
  }
  finished ();
}

void
IMAP4Protocol::slave_status ()
{
  kdDebug(7116) << "IMAP4::slave_status" << endl;
//  KIO::TCPSlaveBase::slave_status();
  slaveStatus (myHost, !(getState () == ISTATE_NO));
//  slaveStatus(QString::null,false);
}

void
IMAP4Protocol::dispatch (int command, const QByteArray & data)
{
  kdDebug(7116) << "IMAP4::dispatch - command=" << command << endl;
  KIO::TCPSlaveBase::dispatch (command, data);
}

void
IMAP4Protocol::stat (const KURL & _url)
{
  kdDebug(7116) << "IMAP4::stat - " << hidePass(_url) << endl;
  QString aBox, aSequence, aLType, aSection, aValidity, aDelimiter;
  enum IMAP_TYPE aType =
    parseURL (_url, aBox, aSection, aLType, aSequence, aValidity, aDelimiter);

  UDSEntry entry;
  UDSAtom atom;

  atom.m_uds = UDS_NAME;
  atom.m_str = aBox;
  entry.append (atom);

  if (!aSection.isEmpty())
  {
    if (getState() == ISTATE_SELECT && aBox == getCurrentBox())
    {
      imapCommand *cmd = doCommand (imapCommand::clientClose());
      bool ok = cmd->result() == "OK";
      completeQueue.removeRef(cmd);
      if (!ok)
      {
        error(ERR_NO_CONTENT, i18n("Unable to close mailbox."));
        return;
      }
      setState(ISTATE_LOGIN);
    }
    imapCommand *cmd = doCommand(imapCommand::clientStatus(aBox, aSection));
    bool ok = cmd->result() == "OK";
    QString cmdInfo = cmd->resultInfo();
    completeQueue.removeRef(cmd);
    if (!ok)
    {
      bool found = false;
      cmd = doCommand (imapCommand::clientList ("", aBox));
      if (cmd->result () == "OK")
      {
        for (QValueListIterator < imapList > it = listResponses.begin ();
             it != listResponses.end (); ++it)
        {
          if (aBox == (*it).name ()) found = true;
        }
      }
      completeQueue.removeRef (cmd);
      if (found)
        error(ERR_NO_CONTENT, i18n("Unable to get information about folder %1. The server replied: %2").arg(aBox).arg(cmdInfo));
      else
        error(KIO::ERR_DOES_NOT_EXIST, aBox);
      return;
    }
    if ((aSection == "UIDNEXT" && getStatus().uidNextAvailable())
      || (aSection == "UNSEEN" && getStatus().unseenAvailable()))
    {
      atom.m_uds = UDS_SIZE;
      atom.m_str = QString::null;
      atom.m_long = (aSection == "UIDNEXT") ? getStatus().uidNext()
        : getStatus().unseen();
      entry.append(atom);
    }
  } else
  if (aType == ITYPE_BOX || aType == ITYPE_DIR_AND_BOX || aType == ITYPE_MSG)
  {
    ulong validity = 0;
    // see if the box is already in select/examine state
    if (aBox == getCurrentBox ())
      validity = selectInfo.uidValidity ();
    else
    {
      // do a status lookup on the box
      // only do this if the box is not selected
      // the server might change the validity for new select/examine
      imapCommand *cmd =
        doCommand (imapCommand::clientStatus (aBox, "UIDVALIDITY"));
      completeQueue.removeRef (cmd);
      validity = getStatus ().uidValidity ();
    }
    validity = 0;               // temporary

    if (aType == ITYPE_BOX || aType == ITYPE_DIR_AND_BOX)
    {
      // has no or an invalid uidvalidity
      if (validity > 0 && validity != aValidity.toULong ())
      {
        //redirect
        KURL newUrl = _url;

        newUrl.setPath ("/" + aBox + ";UIDVALIDITY=" +
                        QString ().setNum (validity));
        kdDebug(7116) << "IMAP4::stat - redirecting to " << hidePass(newUrl) << endl;
        redirection (newUrl);
      }
    }
    else if (aType == ITYPE_MSG)
    {
      //must determine if this message exists
      //cause konqueror will check this on paste operations

      // has an invalid uidvalidity
      // or no messages in box
      if (validity > 0 && validity != aValidity.toULong ())
      {
        aType = ITYPE_UNKNOWN;
        kdDebug(7116) << "IMAP4::stat - url has invalid validity [" << validity << "d] " << hidePass(_url) << endl;
      }
    }
  }

  atom.m_uds = UDS_MIME_TYPE;
  atom.m_str = getMimeType (aType);
  entry.append (atom);

  kdDebug(7116) << "IMAP4: stat: " << atom.m_str << endl;
  switch (aType)
  {
  case ITYPE_DIR:
    atom.m_uds = UDS_FILE_TYPE;
    atom.m_str = "";
    atom.m_long = S_IFDIR;
    entry.append (atom);
    break;

  case ITYPE_BOX:
  case ITYPE_DIR_AND_BOX:
    atom.m_uds = UDS_FILE_TYPE;
    atom.m_str = "";
    atom.m_long = S_IFDIR;
    entry.append (atom);
    break;

  case ITYPE_MSG:
    atom.m_uds = UDS_FILE_TYPE;
    atom.m_str = "";
    atom.m_long = S_IFREG;
    entry.append (atom);
    break;

  case ITYPE_UNKNOWN:
    error (ERR_DOES_NOT_EXIST, hidePass(_url));
    break;
  }

  statEntry (entry);
  kdDebug(7116) << "IMAP4::stat - Finishing stat" << endl;
  finished ();
}

void IMAP4Protocol::openConnection()
{
  if (makeLogin()) connected();
}

void IMAP4Protocol::closeConnection()
{
  if (getState() == ISTATE_NO) return;
  if (getState() == ISTATE_SELECT && metaData("expunge") == "auto")
  {
    imapCommand *cmd = doCommand (imapCommand::clientExpunge());
    completeQueue.removeRef (cmd);
  }
  if (getState() != ISTATE_CONNECT)
  {
    imapCommand *cmd = doCommand (imapCommand::clientLogout());
    completeQueue.removeRef (cmd);
  }
  closeDescriptor();
  setState(ISTATE_NO);
  completeQueue.clear();
  sentQueue.clear();
  lastHandled = NULL;
  currentBox = QString::null;
  readBufferLen = 0;
}

bool IMAP4Protocol::makeLogin ()
{
  if (getState () == ISTATE_LOGIN || getState () == ISTATE_SELECT)
    return true;

  kdDebug(7116) << "IMAP4::makeLogin - checking login" << endl;
  bool alreadyConnected = getState() == ISTATE_CONNECT;
  if (alreadyConnected || connectToHost (myHost.latin1(), myPort))
  {
//      fcntl (m_iSock, F_SETFL, (fcntl (m_iSock, F_GETFL) | O_NDELAY));

    setState(ISTATE_CONNECT);

    myAuth = metaData("auth");
    myTLS  = metaData("tls");

    imapCommand *cmd;

    unhandled.clear ();
    if (!alreadyConnected) while (!parseLoop ());    //get greeting
    QString greeting;
    if (!unhandled.isEmpty()) greeting = unhandled.first().stripWhiteSpace();
    unhandled.clear ();       //get rid of it
    cmd = doCommand (new imapCommand ("CAPABILITY", ""));

    kdDebug(7116) << "IMAP4: setHost: capability" << endl;
    for (QStringList::Iterator it = imapCapabilities.begin ();
         it != imapCapabilities.end (); ++it)
    {
      kdDebug(7116) << "'" << (*it) << "'" << endl;
    }
    completeQueue.removeRef (cmd);

    if (!hasCapability("IMAP4") && !hasCapability("IMAP4rev1"))
    {
      error(ERR_COULD_NOT_LOGIN, i18n("The server %1 supports neither "
        "IMAP4 nor IMAP4rev1.\nIt identified itself with: %2")
        .arg(myHost).arg(greeting));
      closeConnection();
      return false;
    }

    if (metaData("nologin") == "on") return TRUE;

    if (myTLS == "on" && !hasCapability(QString("STARTTLS")))
    {
      error(ERR_COULD_NOT_LOGIN, i18n("The server does not support TLS.\n"
        "Disable this security feature to connect unencrypted."));
      closeConnection();
      return false;
    }
    if ((myTLS == "on" || (canUseTLS() && myTLS != "off")) &&
      hasCapability(QString("STARTTLS")))
    {
      imapCommand *cmd = doCommand (imapCommand::clientStartTLS());
      if (cmd->result () == "OK")
      {
        completeQueue.removeRef(cmd);
        int tlsrc = startTLS();
        if (tlsrc == 1)
        {
          kdDebug() << "TLS mode has been enabled." << endl;
          imapCommand *cmd2 = doCommand (new imapCommand ("CAPABILITY", ""));
          for (QStringList::Iterator it = imapCapabilities.begin ();
                                     it != imapCapabilities.end (); ++it)
          {
            kdDebug(7116) << "'" << (*it) << "'" << endl;
          }
          completeQueue.removeRef (cmd2);
        } else {
          kdDebug() << "TLS mode setup has failed.  Aborting." << endl;
          error (ERR_COULD_NOT_LOGIN, i18n("Starting TLS failed."));
          closeConnection();
          return false;
        }
      } else completeQueue.removeRef(cmd);
    }

    if (!myAuth.isEmpty () && myAuth != "*"
        && !hasCapability (QString ("AUTH=") + myAuth))
    {
      error (ERR_COULD_NOT_LOGIN, i18n("The authentication method %1 is not "
        "supported by the server.").arg(myAuth));
      closeConnection();
      return false;
    }

    kdDebug(7116) << "IMAP4::makeLogin - attempting login" << endl;

    KIO::AuthInfo authInfo;
    authInfo.username = myUser;
    authInfo.password = myPass;
    authInfo.prompt = i18n ("Username and password for your IMAP account:");
    if (myUser.isEmpty () || myPass.isEmpty ()) {
      if(openPassDlg (authInfo)) {
        myUser = authInfo.username;
        myPass = authInfo.password;
      }
    }

    kdDebug(7116) << "IMAP4::makeLogin - open_PassDlg: user=" << myUser << " pass=xx" << endl;

    QString resultInfo;
    if (myAuth.isEmpty () || myAuth == "*")
    {
      if (QString(myUser.utf8()) != myUser)
        error(KIO::ERR_COULD_NOT_LOGIN, i18n("In IMAP clear text login, "
          "only US-ASCII characters are possible. Please use a different "
          "authentication method that your server supports or try to get a "
          "different username."));
      else if (QString(myPass.utf8()) != myPass)
        error(KIO::ERR_COULD_NOT_LOGIN, i18n("In IMAP clear text login, "
          "only US-ASCII characters are possible. Please use a different "
          "authentication method that your server supports or change your "
          "password."));
      else if (!clientLogin (myUser, myPass, resultInfo))
        error(KIO::ERR_COULD_NOT_LOGIN, i18n("Unable to login. Probably the "
        "password is wrong.\nThe server replied:\n%1").arg(resultInfo));
    }
    else
    {
      if (!clientAuthenticate (myUser, myPass, myAuth, mySSL, resultInfo))
        error(KIO::ERR_COULD_NOT_LOGIN, i18n("Unable to authenticate via %1.\n"
        "The server replied:\n%2").arg(myAuth).arg(resultInfo));
    }
  }

  return getState() == ISTATE_LOGIN;
}

void
IMAP4Protocol::parseWriteLine (const QString & aStr)
{
  QCString writer = aStr.utf8 ();

  // append CRLF if necessary
  if (!writer.length() || (writer[writer.length () - 1] != '\n'))
    writer += "\r\n";

  // write it
  write (writer.data (), writer.length ());
}

QString
IMAP4Protocol::getMimeType (enum IMAP_TYPE aType)
{
  QString retVal = "unknown/unknown";

  switch (aType)
  {
  case ITYPE_UNKNOWN:
    retVal = "unknown/unknown";
    break;

  case ITYPE_DIR:
    retVal = "inode/directory";
    break;

  case ITYPE_BOX:
    retVal = "message/digest";
    break;

  case ITYPE_DIR_AND_BOX:
    retVal = "message/directory";
    break;

  case ITYPE_MSG:
    retVal = "message/rfc822-imap";
    break;
  }

  return retVal;
}



void
IMAP4Protocol::doListEntry (const KURL & _url, int stretch, imapCache * cache,
  bool withFlags, bool withSubject)
{
  if (cache)
  {
    UDSEntry entry;
    UDSAtom atom;
    KURL aURL = _url;
    aURL.setQuery (QString::null);

    entry.clear ();

    atom.m_uds = UDS_NAME;
    atom.m_str = QString::number(cache->getUid());
    atom.m_long = 0;
    if (stretch > 0)
    {
      atom.m_str = "0000000000000000" + atom.m_str;
      atom.m_str = atom.m_str.right (stretch);
    }
    if (withSubject)
    {
      mailHeader *header = cache->getHeader();
      if (header)
        atom.m_str += " " + header->getSubject();
    }
//    atom.m_str.prepend(";UID=");
    entry.append (atom);

    atom.m_uds = UDS_URL;
    atom.m_str = aURL.url(0, 106); // utf-8
    if (atom.m_str[atom.m_str.length () - 1] != '/')
      atom.m_str += "/";
    atom.m_str += ";UID=" + QString::number(cache->getUid());
    atom.m_long = 0;
    entry.append (atom);

    atom.m_uds = UDS_FILE_TYPE;
    atom.m_str = "";
    atom.m_long = S_IFREG;
    entry.append (atom);

    atom.m_uds = UDS_SIZE;
    atom.m_long = cache->getSize();
    entry.append (atom);

    atom.m_uds = UDS_MIME_TYPE;
    atom.m_str = "message/rfc822-imap";
    atom.m_long = 0;
    entry.append (atom);

    atom.m_uds = UDS_USER;
    atom.m_str = myUser;
    entry.append (atom);

    atom.m_uds = KIO::UDS_ACCESS;
    atom.m_long = (withFlags) ? cache->getFlags() : S_IRUSR | S_IXUSR | S_IWUSR;
    entry.append (atom);

    listEntry (entry, false);
  }
}

void
IMAP4Protocol::doListEntry (const KURL & _url, const QString & myBox,
                            const imapList & item)
{
  KURL aURL = _url;
  aURL.setQuery (QString::null);
  UDSEntry entry;
  UDSAtom atom;
  int hdLen = item.hierarchyDelimiter().length();

  {
    QString mailboxName = item.name ();

    // some beautification
    if (mailboxName.find (myBox) == 0)
    {
      mailboxName =
        mailboxName.right (mailboxName.length () - myBox.length ());
    }
    if (mailboxName[0] == '/')
        mailboxName = mailboxName.right (mailboxName.length () - 1);
    if (mailboxName.left(hdLen) == item.hierarchyDelimiter())
      mailboxName = mailboxName.right(mailboxName.length () - hdLen);
    if (mailboxName.right(hdLen) == item.hierarchyDelimiter())
      mailboxName = mailboxName.left(mailboxName.length () - hdLen);

    atom.m_uds = UDS_NAME;
    atom.m_str = mailboxName;

    // konqueror will die with an assertion failure otherwise
    if (atom.m_str.isEmpty ())
      atom.m_str = "..";

    if (!atom.m_str.isEmpty ())
    {
      atom.m_long = 0;
      entry.append (atom);

      if (!item.noSelect ())
      {
        atom.m_uds = UDS_MIME_TYPE;
        if (!item.noInferiors ())
        {
          atom.m_str = "message/directory";
        } else {
          atom.m_str = "message/digest";
        }
        atom.m_long = 0;
        entry.append (atom);
        mailboxName += "/";

        // explicitly set this as a directory for KFileDialog
        atom.m_uds = UDS_FILE_TYPE;
        atom.m_str = "";
        atom.m_long = S_IFDIR;
        entry.append (atom);
      }
      else if (!item.noInferiors ())
      {
        atom.m_uds = UDS_MIME_TYPE;
        atom.m_str = "inode/directory";
        atom.m_long = 0;
        entry.append (atom);
        mailboxName += "/";

        // explicitly set this as a directory for KFileDialog
        atom.m_uds = UDS_FILE_TYPE;
        atom.m_str = "";
        atom.m_long = S_IFDIR;
        entry.append (atom);
      }
      else
      {
        atom.m_uds = UDS_MIME_TYPE;
        atom.m_str = "unknown/unknown";
        atom.m_long = 0;
        entry.append (atom);
      }

      atom.m_uds = UDS_URL;
      QString path = aURL.path();
      atom.m_str = aURL.url (0, 106); // utf-8
      if (path.right(1) == "/" && !path.isEmpty() && path != "/")
        path = path.left(path.length() - 1);
      if (!path.isEmpty() && path != "/"
        && path.right(hdLen) != item.hierarchyDelimiter())
        path += item.hierarchyDelimiter();
      path += mailboxName;
      aURL.setPath(path);
      atom.m_str = aURL.url(0, 106); // utf-8
      atom.m_long = 0;
      entry.append (atom);

      atom.m_uds = UDS_USER;
      atom.m_str = myUser;
      entry.append (atom);

      atom.m_uds = KIO::UDS_ACCESS;
      atom.m_long = S_IRUSR | S_IXUSR | S_IWUSR;
      entry.append (atom);

      listEntry (entry, false);
    }
  }
}

enum IMAP_TYPE
IMAP4Protocol::parseURL (const KURL & _url, QString & _box,
                         QString & _section, QString & _type, QString & _uid,
                         QString & _validity, QString & _hierarchyDelimiter)
{
  enum IMAP_TYPE retVal;
  retVal = ITYPE_UNKNOWN;
  _hierarchyDelimiter = QString();

  imapParser::parseURL (_url, _box, _section, _type, _uid, _validity);
//  kdDebug(7116) << "URL: query - '" << KURL::decode_string(_url.query()) << "'" << endl;

  if (!_box.isEmpty ())
  {
    kdDebug(7116) << "IMAP4::parseURL: box " << _box << endl;

    // Hack for UW-IMAP news folders
    if (_box.left(5) == "#news")
    {
      _hierarchyDelimiter = ".";
      retVal = ITYPE_DIR_AND_BOX;
    } else

    if (makeLogin ())
    {
      if (getCurrentBox () != _box || _type == "LIST" || _type == "LSUB")
      {
        imapCommand *cmd;

        cmd = doCommand (imapCommand::clientList ("", _box));
        if (cmd->result () == "OK")
        {
          for (QValueListIterator < imapList > it = listResponses.begin ();
               it != listResponses.end (); ++it)
          {
            //kdDebug(7116) << "IMAP4::parseURL - checking " << _box << " to " << (*it).name() << endl;
            if (_box == (*it).name ())
            {
              _hierarchyDelimiter = (*it).hierarchyDelimiter();
              if ((*it).noSelect ())
              {
                retVal = ITYPE_DIR;
              }
              else if ((*it).noInferiors ())
              {
                retVal = ITYPE_BOX;
              }
              else
              {
                retVal = ITYPE_DIR_AND_BOX;
              }
            }
          }
          // if we got no list response for the box it's probably some kind of prefix
          // e.g. #mh
          if (retVal == ITYPE_UNKNOWN)
            retVal = ITYPE_DIR;
        } else {
          kdDebug(7116) << "IMAP4::parseURL - got error for " << _box << endl;
        }
        completeQueue.removeRef (cmd);
      }
      else
      {
        retVal = ITYPE_BOX;
      }
    }
    else
      kdDebug(7116) << "IMAP4::parseURL: no login!" << endl;

  }
  else
  {
    kdDebug(7116) << "IMAP4: parseURL: box [root]" << endl;
    retVal = ITYPE_DIR;
  }

  //see if it is a real sequence or a simple uid
  if (retVal == ITYPE_BOX || retVal == ITYPE_DIR_AND_BOX)
  {
    if (!_uid.isEmpty ())
    {
      if (_uid.find (":") == -1 && _uid.find (",") == -1
          && _uid.find ("*") == -1)
        retVal = ITYPE_MSG;
    }
  }
  if ( _hierarchyDelimiter.isEmpty() && 
       (_type == "LIST" || _type == "LSUB") )
  {
    // try to reconstruct the delimiter from the URL
    if (!_box.isEmpty())
    {
      int start = _url.path().findRev(_box);
      if (start != -1)
        _hierarchyDelimiter = _url.path().mid(start-1, start);
      kdDebug(7116) << "IMAP4::parseURL - reconstructed delimiter:" << _hierarchyDelimiter
        << " from URL " << _url.path() << endl;
    }
    if (_hierarchyDelimiter.isEmpty())
      _hierarchyDelimiter = "/";
  }

  return retVal;
}

int
IMAP4Protocol::outputLine (const QCString & _str)
{
  if (cacheOutput)
  {
    QBuffer stream(outputCache);
    stream.open(IO_WriteOnly);
    stream.at(outputCache.size());
    stream.writeBlock(_str.data(), _str.length());
    stream.close();
    return 0;
  }

  QByteArray temp;
  bool relay = relayEnabled;

  relayEnabled = true;
  temp.setRawData (_str.data (), _str.length ());
  parseRelay (temp);
  temp.resetRawData (_str.data (), _str.length ());

  relayEnabled = relay;
  return 0;
}

void IMAP4Protocol::flushOutput()
{
  // send out cached data to the application
  if (outputCache.isEmpty()) return;
  data(outputCache);
  mProcessedSize += outputCache.size();
  processedSize( mProcessedSize );
  outputCache.resize(0);
}

ssize_t IMAP4Protocol::myRead(void *data, ssize_t len)
{
  if (readBufferLen)
  {
    ssize_t copyLen = (len < readBufferLen) ? len : readBufferLen;
    memcpy(data, readBuffer, copyLen);
    readBufferLen -= copyLen;
    if (readBufferLen) memcpy(readBuffer, &readBuffer[copyLen], readBufferLen);
    return copyLen;
  }
  if (!isConnectionValid()) return 0;
  waitForResponse(600);
  return read(data, len);
}

bool
IMAP4Protocol::assureBox (const QString & aBox, bool readonly)
{
  if (aBox.isEmpty()) return false;
  
  imapCommand *cmd = NULL;

  if (aBox != getCurrentBox () || (!getSelected().readWrite() && !readonly))
  {
    // open the box with the appropriate mode
    kdDebug(7116) << "IMAP4Protocol::assureBox - opening box" << endl;
    selectInfo = imapInfo();
    cmd = doCommand (imapCommand::clientSelect (aBox, readonly));
    bool ok = cmd->result() == "OK";
    QString cmdInfo = cmd->resultInfo();
    completeQueue.removeRef (cmd);

    if (!ok)
    {
      bool found = false;
      cmd = doCommand (imapCommand::clientList ("", aBox));
      if (cmd->result () == "OK")
      {
        for (QValueListIterator < imapList > it = listResponses.begin ();
             it != listResponses.end (); ++it)
        {
          if (aBox == (*it).name ()) found = true;
        }
      }
      completeQueue.removeRef (cmd);
      if (found)
        error(ERR_NO_CONTENT, i18n("Unable to open folder %1. The server replied: %2").arg(aBox).arg(cmdInfo));
      else
        error(KIO::ERR_DOES_NOT_EXIST, aBox);
      return false;
    }
  }
  else
  {
    // Give the server a chance to deliver updates every ten seconds.
    // Doing this means a server roundtrip and since assureBox is called
    // after every mail, we do it with a timeout.
    kdDebug(7116) << "IMAP4Protocol::assureBox - reusing box" << endl;
    if ( mTimeOfLastNoop.secsTo( QDateTime::currentDateTime() ) > 10 ) {
      cmd = doCommand (imapCommand::clientNoop ());
      completeQueue.removeRef (cmd);
      mTimeOfLastNoop = QDateTime::currentDateTime();
      kdDebug(7116) << "IMAP4Protocol::assureBox - noop timer fired" << endl;
    }
  }

  // if it is the mode we want
  if (!getSelected().readWrite() && !readonly)
  {
    error(KIO::ERR_CANNOT_OPEN_FOR_WRITING, aBox);
    return false;
  }

  return true;
}
