// -*- c++ -*-
/***************************************************************************
                           controlinterface.h
                           ------------------
    begin                : Sun Nov 21 2004
    copyright            : (C) 2004 by Dirk Ziegelmeier
    email                : dziegel@gmx.de
 ***************************************************************************/

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <qstring.h>
#include <qobject.h>
#include <qvariant.h>
#include <qobjectlist.h>

#ifndef __CONTROLINTERFACE_H__
#define __CONTROLINTERFACE_H__

class Control : public QObject
{
    Q_OBJECT

public:
    enum ControlType { Control_Int, Control_Bool, Control_Menu, Control_Button };

    Control(const QString& uiName, const QString& internalName, ControlType _type = Control_Int);
    ~Control();

    virtual const QVariant valueAsQVariant() const { return QVariant(); }
    
    typedef QObjectList ControlList;
    
    QString uiName;
    QString internalName;
    bool advanced;
    ControlType controlType;

public slots:
    virtual bool setValue(const QVariant& /*unused*/) { return false; }
    virtual bool setDefault() { return false; }
};
    
class IntegerControl : public Control
{
    Q_OBJECT

public:
    IntegerControl(const QString& uiName, const QString& internalName);
    virtual ~IntegerControl();

    int defaultValue;
    int minimumValue;
    int maximumValue;
    int step;

    virtual int value() const = 0;
    virtual const QVariant valueAsQVariant() const { return QVariant(value()); }
    virtual bool setValue(const QVariant& value) { return setValue(value.toInt()); }
    virtual bool setDefault() { return setValue(defaultValue); }

signals:
    void changed(int value);

public slots:
    virtual bool setValue(int value);

protected:
    virtual bool doSetValue(int value) = 0;
};

class BooleanControl : public Control
{
    Q_OBJECT

public:
    BooleanControl(const QString& uiName, const QString& internalName);
    virtual ~BooleanControl();

    bool defaultValue;

    virtual bool value() const = 0;
    virtual const QVariant valueAsQVariant() const { return QVariant(value(), 0); }
    virtual bool setValue(const QVariant& value) { return setValue(value.toBool()); }
    virtual bool setDefault() { return setValue(defaultValue); }

signals:
    void changed(bool value);

public slots:
    virtual bool setValue(bool value);

protected:
    virtual bool doSetValue(bool value) = 0;
};

class MenuControl : public Control
{
    Q_OBJECT

public:
    MenuControl(const QString& uiName, const QString& internalName);
    virtual ~MenuControl();

    QString defaultValue;
    QStringList choices;

    virtual const QString value() const = 0;
    virtual const QVariant valueAsQVariant() const { return QVariant(value()); }
    virtual bool setValue(const QVariant& value) { return setValue(value.toString()); }
    virtual bool setDefault() { return setValue(defaultValue); }

signals:
    void changed(const QString& value);

public slots:
    virtual bool setValue(const QString& value);

protected:
    virtual bool doSetValue(const QString& value) = 0;
};

class ButtonControl : public Control
{
    Q_OBJECT

public:
    ButtonControl(const QString& uiName, const QString& internalName);
    virtual ~ButtonControl();

    virtual bool setValue(const QVariant& /*unused*/) { return click(); }
    virtual bool setDefault() { return true; }

public slots:
    virtual bool click() = 0;
};

#endif
