// -*- c++ -*-

/*
 *
 * Copyright (C) 2002 George Staikos <staikos@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */



#ifndef __KDETVSRC_PLUGIN_H
#define __KDETVSRC_PLUGIN_H

#include <qobject.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qmap.h>
#include <qcolor.h>
#include "kdetvpluginbase.h"
#include "controlinterface.h"
#include "channel.h"

class KdetvSourcePluginPrivate;
class Qimage;

/**
 * Video source plugin base class
 */
class KdetvSourcePlugin : public KdetvPluginBase
{
    Q_OBJECT

public:
    KdetvSourcePlugin(Kdetv *ktv, const QString& cfgkey, QObject *parent = 0, const char* name = 0);
    virtual ~KdetvSourcePlugin();

    enum PluginType { TYPE_ANALOG, TYPE_DVB };
    virtual PluginType pluginType() const { return TYPE_ANALOG; }

    /**
     * To get various values.
     */
    virtual int frequency() = 0;
    virtual int signal() = 0;
    virtual bool muted() = 0;
    virtual Control::ControlList& controls() = 0;

    /**
     * True if the CURRENT RUNNING device is a tuner device.
     */
    virtual bool isTuner() = 0;
    virtual QColor colourKey() = 0;

    /**
     * True if this device is a tuner device.
     * Do not override this.  Provide this information in _tuners.
     */
    bool isTuner(const QString &dev);

    virtual const QStringList& deviceList() const { return _devices;    }
    virtual const QStringList& audioModes() const { return _audioModes; }
    virtual const QStringList& broadcastedAudioModes() = 0;
    virtual const QString&     defaultAudioMode() = 0;
    virtual const QStringList& sourceList(const QString& dev) const   { return _sources[dev];   }
    virtual const QStringList& encodingList(const QString& dev) const { return _encodings[dev]; }
    virtual const QString& device() const    { return _device;    }
    virtual const QString& source() const    { return _source;    }
    virtual const QString& encoding() const  { return _encoding;  }

    // return 0 on success.  This is called to indicate that you should probe
    // for devices and populate _devices, _sources, _encodings, _audioModes and _tuners.
    // It may be called more than once.
    virtual int probeDevices() = 0;

    // Set the widget that we display on
    void setWidget(QWidget *w) { _widget = w; }

    virtual bool videoPlaying() const = 0;

    virtual bool canVideoDesktop() const { return false; }
    virtual bool canGrabStill() const    { return false; }
    virtual bool grabStill( QImage* )    { return false; }

    virtual bool supportsImageFiltering() const { return false; }

public slots:
    /**
     * Sets the device to use.
     */
    virtual int setDevice( const QString &dev ) = 0;

    /**
     * Sets the channel properties.
     */
    virtual int setChannelProperties(const Channel::PropertyList& properties) = 0;

    /**
     * Sets the source to use.
     */
    virtual int setSource( const QString &src ) = 0;

    /**
     * Set the norm for the plugin.
     */
    virtual int setEncoding( const QString &encoding ) = 0;

    /**
     * Sets the current current channel.
     */
    virtual void setFrequency( int freq ) = 0;

    /**
     * Set audio mode.
     */
    virtual int setAudioMode( const QString &audioMode ) = 0;

    /**
     * Mutes the sound.
     */
    virtual void setMuted( bool muted ) = 0;

    /**
     * Set volume.
     */
    virtual bool setVolume( int left, int right );

    virtual int startVideo() = 0;
    virtual int stopVideo() = 0;

    virtual int setVideoDesktop(bool /* on */) { return -1; };

    /**
     * Hint for the plugin to display video in fullscreen mode
     * (e.g. to change resolution)
     */
    virtual void setFullscreen(bool) {};


signals:
    void errorMessage(const QString& msg);


protected:
    QStringList _devices;
    QStringList _audioModes;
    QMap<QString, QStringList> _sources;
    QMap<QString, QStringList> _encodings;
    QMap<QString, bool> _tuners;
    QString _device, _source, _encoding;
    QWidget *_widget;
    bool _isVideoDesktop, _isTuner;

private:
    KdetvSourcePluginPrivate *d;
};

#endif
