/***************************************************************************
 *   Copyright (C) 2004, 2005 Thomas Nagy                                  *
 *   tnagy2^8@yahoo.fr                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation (see COPYING)            *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 ***************************************************************************/

#include <qheader.h>
#include <qdragobject.h>
#include <qpopupmenu.h>

#include <kmacroexpander.h>
#include <krun.h>
#include <klocale.h>
#include <kapplication.h>
#include <kdebug.h>
#include <kurldrag.h>

#include "DDataItem.h"
#include "settings.h"
#include "linksPopup.h"
#include "urldlg.h"
#include "BoxURL.h"

BoxURL::BoxURL(QWidget *parent) : KListView(parent)
{
	m_item = NULL;
	m_popup = NULL;
	setAcceptDrops(true);

	connect(this, SIGNAL( contextMenu(KListView*, QListViewItem*, const QPoint&) ),
			this, SLOT( popupmenu(KListView*, QListViewItem*, const QPoint&) ) );

	connect(this, SIGNAL(doubleClicked( QListViewItem *, const QPoint &, int )),
			this, SLOT( launchSelectedURL() ) );

	connect(this, SIGNAL( datachanged() ), this, SLOT( updateItem() ) );
	
	header()->hide();

	addColumn(i18n("Links"));
	setResizeMode(KListView::AllColumns);
}

void BoxURL::dragEnterEvent(QDragEnterEvent* event)
{
	if (!m_item) event->accept(false);
	else event->accept(KURLDrag::canDecode(event));
}

void BoxURL::dropEvent(QDropEvent* event)
{
	if (!m_item) return;
	if (!KURLDrag::canDecode(event)) return;

	KURL::List ulist;
	KURLDrag::decode(event, ulist);
	for (unsigned int i=0; i<ulist.count(); i++)
		addSingleURL( ulist[i].url() );
	updateItem();
}

void BoxURL::setCurrentItem(DDataItem* item)
{
	if (!item)
	{
		clear();
		setEnabled(false);
		m_item = item;
	}
	else
	{
		setEnabled(true);
		if ( m_item != item ) 
		{
			m_item = item;
			updateFromItem();
		}
	}
}

void BoxURL::updateItem()
{
	if (!m_item) return;
	m_item->m_urllist.clear();

	QListViewItemIterator it(this);
	while (it.current())
	{
		BoxURLItem* item = (BoxURLItem*) (*it);
		m_item->m_urllist.append( item->m_object );
		++it;
	}
}

void BoxURL::updateFromItem()
{
	if (!m_item) return;

	clear();
	for (unsigned int i=0; i<m_item->m_urllist.count(); i++)
	{
		addItem( m_item->m_urllist[i] );
	}
}

void BoxURL::requestAddURL()
{
	URLObject obj;
	urldlg *dlg = new urldlg(this, obj);

	if (KDialogBase::Accepted == dlg->exec())
	{
		addItem( obj );
		emit datachanged();
	}
	delete dlg;
}

void BoxURL::requestChangeURL()
{
	BoxURLItem* item = (BoxURLItem*) selectedItem();
	if (!item) return;
	
	urldlg *dlg = new urldlg(this, item->m_object);
	if (KDialogBase::Accepted == dlg->exec())
	{
		item->updateCaption();
		emit datachanged();
	}
}

void BoxURL::addItem(URLObject& obj)
{
	if (!obj.m_caption.length() && !obj.m_url.length()) return;
	
	BoxURLItem* lstitem = new BoxURLItem(this);
	lstitem->m_object = obj;
	lstitem->updateCaption();
}

void BoxURL::addSingleURL( const QString & url )
{
	URLObject obj;
	obj.m_caption = QString::null;
	obj.m_url = url;
	addItem(obj);
}

void BoxURL::popupmenu(KListView*l, QListViewItem*i, const QPoint&p)
{
	if (m_popup) delete m_popup;
	m_popup = new linksPopup(this);
	connect(m_popup, SIGNAL(activated(int)), this, SLOT(popupMenuSel(int)));
	m_popup->exec(p);
}

void BoxURL::popupMenuSel(int val)
{
	switch (val)
	{
		case linksPopup::e_add:
			requestAddURL();
			break;
		case linksPopup::e_modify:
			requestChangeURL();
			break;
		case linksPopup::e_launch:
			launchSelectedURL();
			break;
		case linksPopup::e_remove:
			delete selectedItem();
			emit datachanged();
			break;
		default:
			break;
	}
}

void BoxURL::launchSelectedURL()
{
	if (!selectedItem()) return;
	BoxURLItem* item = (BoxURLItem*) selectedItem();
	launchURL( item->m_object.m_url );
}

void BoxURL::launchURL(const QString& url)
{
	QMap<QChar,QString> map;
	map.insert('s', url);
	QString command = Settings::launchCommand();
	KRun::runCommand( KMacroExpander::expandMacrosShellQuote( command, map ) );
}

void BoxURLItem::updateCaption()
{
	if (!m_object.m_caption.length()) setText(0, m_object.m_url);
	else setText(0, m_object.m_caption+" ["+m_object.m_url+"]");
}

#include "BoxURL.moc"
