/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : Handle communications with flickr.com
 * 
 * Copyright 2005 by M. Asselstine

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */
#include "flickrcomm.h"

#include <qdom.h>
#include <krun.h>
#include <qdom.h>
#include <qurl.h>
#include <qfile.h>
#include <qimage.h>
#include <qregexp.h>
#include <qbuffer.h>
#include <qstring.h>
#include <klocale.h>
#include <qwmatrix.h>
#include <kimageio.h>
#include <qstrlist.h>
#include <qcstring.h> 
#include <kmdcodec.h>
#include <qtextstream.h>
#include <qstringlist.h>
#include <kapplication.h>

#include "photolistview.h"

#include <algorithm>

/*!
    \fn FlickrComm::FlickrComm(QObject *parent, const char *name)
 */
FlickrComm::FlickrComm(QObject *parent, const char *name)
 : QObject(parent, name),
 m_APIKey("c0134cf226b1187e3d79e4e1be03d1bf"),
 m_publicKey("f92fb243a918a3d2"),
 m_userToken("")
{
    // variable inits
    m_requests.clear();
    m_incomingData.clear();
    m_MD5Context = new KMD5("");
}

/*!
    \fn FlickrComm::~FlickrComm()
 */
FlickrComm::~FlickrComm()
{
    delete m_MD5Context;
}


/*!
    \fn FlickrComm::generateMD5(const ArgMap &)
 */
QString FlickrComm::generateMD5(const ArgMap &args)
{
    QString str;
    ArgMap::ConstIterator it;    
    
    // create the string from our arguments
    for( it = args.constBegin(); it != args.constEnd(); ++it )
    {
        str += it.key() + it.data();
    }
      
    // generate MD5, we have to make use of the QCString returned by the call
    // to QString::utf8 in order to properly account for unicode issues
    m_MD5Context->reset();
    m_MD5Context->update(m_publicKey);
    m_MD5Context->update(str.utf8());
    
    return m_MD5Context->hexDigest().data();
}


/*!
    \fn FlickrComm::assembleArgs(const ArgMap &)
 */
QString FlickrComm::assembleArgs(const ArgMap &args)
{
    QString result;
    ArgMap::ConstIterator it;    
    
    // create the request URL  XXX=YYY&ZZZ=AAA...
    for( it = args.constBegin(); it != args.constEnd(); ++it )
    {
        if( !result.isEmpty() )
        {  
            result += "&";
        }
        result += it.key() + "=" + it.data();
    }
    return result;
}


/*!
    \fn FlickrComm::validateHTTPResponse(const QString &)
 */
QString FlickrComm::validateHTTPResponse(const QString &str)
{
    QString err;
    QDomNode node;
    QDomElement root;
    QDomDocument doc("response");
    
    // pump str into the XML document
    if( !doc.setContent(str) )
    {
        return i18n("Unrecognizable response from Flickr.com");
    }
    
    // start at document root, first node
    root = doc.documentElement();
    node = root.firstChild();
    
    // check for fail response
    if( root.attribute("stat", "fail") == "fail")
    {
        // go through each node, should only be one
        while( !node.isNull() )
        {
            // we got our frob, good to go
            if( node.isElement() && node.nodeName() == "err" )
            {
                // get error text
                QDomElement elem = node.toElement();
                err = elem.attribute("msg", i18n("Unknown"));
            }
            node = node.nextSibling();
        }
    }
    return err;
}


/*!
    \fn FlickrComm::setUserToken(const QString &)
 */
void FlickrComm::setUserToken(const QString &str)
{
    m_userToken = str;
}


/*!
    \fn FlickrComm::sendRequest(ArgMap)
 */
KIO::TransferJob* FlickrComm::sendRequest(ArgMap &args)
{
    QString url = "http://www.flickr.com/services/rest/?";
        
    // setup request specific arguments
    args["api_key"] = m_APIKey;
    args["api_sig"] = generateMD5(args);
    
    // formulate complete URL
    url += assembleArgs(args);
    
    // send request
    KIO::TransferJob *job = KIO::http_post(url, QByteArray(0), false);
    job->addMetaData("content-type", "Content-Type: application/x-www-form-urlencoded");
    
    connect(job,SIGNAL(result(KIO::Job*)),this,SLOT(jobResult(KIO::Job*)));
    connect(job,SIGNAL(data(KIO::Job*,const QByteArray&)),this,SLOT(jobData(KIO::Job*,const QByteArray&)));
    
    return job;
}


/*!
    \fn FlickrComm::sendFROBRequest()
 */
void FlickrComm::sendFROBRequest()
{
    ArgMap args;    
    
    // setup args
    args["method"] = "flickr.auth.getFrob";
    
    // remember request type and make request
    KIO::TransferJob* job = sendRequest(args);
    m_requests[job] = FROB_REQ;
}


/*!
    \fn FlickrComm::sendTokenRequest(const QString &frob)
 */
void FlickrComm::sendTokenRequest(const QString &frob)
{
    ArgMap args;    
    
    // setup args
    args["method"]  = "flickr.auth.getToken";
    args["frob"]    = frob;
    
    // remember request type and make request
    KIO::TransferJob* job = sendRequest(args);
    m_requests[job] = TOKEN_REQ;
}


/*!
    \fn FlickrComm::sendPhotosetsRequest(const QString &user)
 */
void FlickrComm::sendPhotosetsRequest(const QString &user)
{
    ArgMap args;    
    
    // setup args
    args["method"]      = "flickr.photosets.getList";
    args["user_id"]     = user;
    
    // remember request type and make request
    KIO::TransferJob* job = sendRequest(args);
    m_requests[job] = PHOTOSET_REQ;   
}


/*!
    \fn FlickrComm::sendTagsRequest(const QString &token, const QString &user)
 */
void FlickrComm::sendTagsRequest(const QString &token, const QString &user)
{
    ArgMap args;    
    
    // setup args
    args["method"]      = "flickr.tags.getListUser";
    args["user_id"]     = user;
    args["auth_token"]  = token;
    
    // remember request type and make request
    KIO::TransferJob* job = sendRequest(args);
    m_requests[job] = TAGS_REQ;
}


/*!
    \fn FlickrComm::sendUpStatusRequest( const QString &token )
 */
void FlickrComm::sendUpStatusRequest( const QString &token )
{
    ArgMap args;
    
    // setup args
    args["method"]        = "flickr.people.getUploadStatus";
    args["auth_token"]    = token;
    
    // remember request type and make request
    KIO::TransferJob* job = sendRequest(args);
    m_requests[job] = STATUS_REQ;
}


/*!
    \fn FlickrComm::prepareWebAuthentication(const QString &frob)
 */
void FlickrComm::doWebAuthentication(const QString &frob)
{
    ArgMap args;
    QString addr;
    
    // setup browser address string
    addr = "http://flickr.com/services/auth/";
    args["api_key"] = "c0134cf226b1187e3d79e4e1be03d1bf";
    args["perms"]   = "write";
    args["frob"]    = frob;
    args["api_sig"] = generateMD5(args);

    addr += "?" + assembleArgs(args);
    
    // Open the default browser with the address
    new KRun( KURL( addr ) );
}


/*!
    \fn FlickrComm::sendPhoto(const QString &token, PhotoListViewItem *photo)
 */
void FlickrComm::sendPhoto(const QString &token, PhotoListViewItem *photo)
{
    ArgMap args;
    QString encoded;
    QByteArray array;
    QString dd = "--";
    QString crlf = "\r\n";
    QBuffer buffer(array);
    QString url = "http://www.flickr.com/services/upload/?";
    QString boundary = QString("-----") + KApplication::randomString(20);
    
    
    // setup form
    buffer.open(IO_WriteOnly);
    QTextStream textStream(&buffer);
    textStream.setEncoding(QTextStream::UnicodeUTF8);
    
    // setup request specific arguments
    args["api_key"] = m_APIKey;
    textStream << dd << boundary << crlf;
    textStream << "Content-Disposition: form-data; name=\"api_key\"" << crlf << crlf;
    textStream << m_APIKey << crlf;
     
    args["auth_token"] = token;
    textStream << dd << boundary << crlf;
    textStream << "Content-Disposition: form-data; name=\"auth_token\"" << crlf << crlf;
    textStream << token << crlf;
    
    if( !photo->title().isEmpty() )
    {
        args["title"] = photo->title();
        textStream << dd << boundary << crlf;
        textStream << "Content-Disposition: form-data; name=\"title\"" << crlf << crlf;
        textStream << photo->title() << crlf;
    }
    
    if( !photo->description().isEmpty() )
    {
        args["description"] = photo->description();
        textStream << dd << boundary << crlf;
        textStream <<"Content-Disposition: form-data; name=\"description\"" << crlf << crlf;
        textStream << photo->description() << crlf;
    }
    
    if( photo->tags().count() > 0 )
    {
        args["tags"] = photo->tags().join(" ");
        textStream << dd << boundary << crlf;
        textStream << "Content-Disposition: form-data; name=\"tags\"" << crlf << crlf;
        textStream << photo->tags().join(" ") << crlf;
    }
    
    args["is_public"] = (photo->isPublic() ? "1" : "0");
    textStream << dd << boundary << crlf;
    textStream << "Content-Disposition: form-data; name=\"is_public\"" << crlf << crlf;
    textStream << (photo->isPublic( ) ? "1" : "0") << crlf;
    
    args["is_family"] = (photo->isFamily() ? "1" : "0");
    textStream << dd << boundary << crlf;
    textStream << "Content-Disposition: form-data; name=\"is_family\"" << crlf << crlf;
    textStream << (photo->isFamily() ? "1" : "0") << crlf;
    
    args["is_friend"] = (photo->isFriends() ? "1" : "0");
    textStream << dd << boundary << crlf;
    textStream << "Content-Disposition: form-data; name=\"is_friend\"" << crlf << crlf;
    textStream << (photo->isFriends() ? "1" : "0") << crlf;
    
    // We can now generate our MD5
    QString md5 = generateMD5(args);
    
    textStream << dd << boundary << crlf;
    textStream << "Content-Disposition: form-data; name=\"api_sig\"" << crlf << crlf;
    textStream << md5 << crlf;
    
    textStream << dd << boundary << crlf;
    textStream << "Content-Disposition: form-data; name=\"photo\"; filename=\"";
    textStream << photo->url().path() << QCString( "\"" ) << crlf;
    textStream << "Content-Type: " << KImageIO::mimeType(photo->url().path()) << crlf << crlf;
        
    if( photo->size() == i18n("Original") && photo->rotation() == 0 )
    {
        // put the file data into the array
        QFile photoFile(photo->url().path());
        if ( !photoFile.open(IO_ReadOnly) )
            return;
        textStream.writeRawBytes(photoFile.readAll().data(), photoFile.size());
        photoFile.close();
    }
    else
    {    
        QByteArray EXIFData;
        QStrList fmts = QImage::outputFormats();
        QString fmt = QImage::imageFormat(photo->url().path());
            
        // copy the EXIF data header from the photo file
        if( fmt == "JPEG" && fmts.contains("JPEG") )
        {
            QFile photoFile(photo->url().path());
            EXIFData = getEXIFData(photoFile);
            photoFile.close();        
        }
        
        // open the image
        QImage photoImage(photo->url().path());
        
        // size the image
        if( photo->size() == i18n("Square 75x75") )
        {
            photoImage = photoImage.smoothScale(75, 75, QImage::ScaleFree);
        }
        else if( photo->size() == i18n("Thumb 75x100") )
        {
            photoImage = photoImage.smoothScale(75, 100, QImage::ScaleFree);
        }
        else if( photo->size() == i18n("Small 180x240") )
        {
            photoImage = photoImage.smoothScale(180, 240, QImage::ScaleMin);
        }
        else if( photo->size() == i18n("Medium 375x500") )
        {
            photoImage = photoImage.smoothScale(375, 500, QImage::ScaleMin);
        }
        else if( photo->size() == i18n("Large 768x1024") )
        {
            photoImage = photoImage.smoothScale(768, 1024, QImage::ScaleMin);
        }
        
        // rotate the image
        if( photo->rotation() != 0 )
        {
            QWMatrix matrix;
            matrix = matrix.rotate(photo->rotation());
            photoImage = photoImage.xForm(matrix);
        }
        
        // add image data to our form data, maintain format if possible
        if( fmts.contains(fmt) )
        {
            if( !EXIFData.isEmpty() )
                writePhotoWithEXIF(textStream, photoImage, EXIFData);
            else
                photoImage.save(&buffer, fmt);
        }
        else if( fmts.contains("JPEG") )
        {
            photoImage.save(&buffer, "JPEG");
        }
        else if( fmts.contains("PNG") )
        {
           photoImage.save(&buffer, "PNG");
        }
    }
    
    // end form
    textStream << crlf << dd << boundary << dd << crlf << "\0";
    buffer.close();
    
    // send request
    KIO::TransferJob *job = KIO::http_post(url, array, false);
    job->addMetaData("content-type", "Content-Type: multipart/form-data; boundary=" + boundary);
        
    connect(job,SIGNAL(result(KIO::Job*)),this,SLOT(jobResult(KIO::Job*)));
    connect(job,SIGNAL(data(KIO::Job*,const QByteArray&)),this,SLOT(jobData(KIO::Job*,const QByteArray&)));

    m_requests[job] = FILE_UPLOAD;
}


/*!
    \fn FlickrComm::jobResult(KIO::Job *job)
 */
void FlickrComm::jobResult(KIO::Job *job)
{
    QString err;
    KIO::TransferJob *transferJob = dynamic_cast<KIO::TransferJob*>(job);

    
    // Get associated request type
    if( transferJob )
    {
        // Check that the operation did not error
        if( job->error() )
        {
            emit commError(i18n("HTTP request failed. (error: %1)").arg(job->errorString()));
            m_requests.erase(transferJob);
            m_incomingData.erase(transferJob);
            return;
        }
        
        // notify user of specific error
        if( m_requests[transferJob] != NO_REQ && (err = validateHTTPResponse(m_incomingData[transferJob])) != "" )
        {
            emit commError(i18n("HTTP request to Flickr.com failed. (msg: %1)").arg(err));
            m_requests.erase(transferJob);
            m_incomingData.erase(transferJob);
            return;
        }
        
        // process the good response
        switch( m_requests[transferJob] )
        {
            case FROB_REQ:
                handleFrobResponse(m_incomingData[transferJob]);
                break;
            case TOKEN_REQ:
                handleTokenResponse(m_incomingData[transferJob]);
                break;
            case TAGS_REQ:
                handleTagsResponse(m_incomingData[transferJob]);
                break;
            case STATUS_REQ:
                handleStatusResponse(m_incomingData[transferJob]);
                break;
            case FILE_UPLOAD:
                handleUploadResponse(m_incomingData[transferJob]);
                break;
            case PHOTOSET_REQ:
                handlePhotosetResponse(m_incomingData[transferJob]);
                break;
            case CREATE_PHOTOSET_REQ:
                hanldeCreatePhotosetResponse(m_incomingData[transferJob]);
                break;
            default:
                break;
        };
        m_requests.erase(transferJob);
        m_incomingData.erase(transferJob);
    }
}


/*!
    \fn FlickrComm::jobData(KIO::Job *job, const QByteArray &data)
 */
void FlickrComm::jobData(KIO::Job *job, const QByteArray &data)
{
    KIO::TransferJob *transferJob = dynamic_cast<KIO::TransferJob*>(job);
    
    if( transferJob && !data.isEmpty() )
    {
        // Add new data to end of the associated buffer
        m_incomingData[transferJob].append(QString::fromUtf8(data, data.size()));
    }
}


/*!
    \fn FlickrComm::handleFrobResponse(const QString &str)
 */
void FlickrComm::handleFrobResponse(const QString &str)
{
    QString frob = "";
    QDomNode node;
    QDomElement root;
    QDomDocument doc("frobresponse");
    
    // pump str into the XML document
    if( !doc.setContent(str) )
    {
        emit commError(i18n("Invalid response received from Flickr.com."));
        return;
    }
    
    // start at document root, first node
    root = doc.documentElement();
    node = root.firstChild();
    
    // go through each node, should only be one
    while( !node.isNull() )
    {
        // we got our frob, good to go
        if( node.isElement() && node.nodeName() == "frob" )
        {
            QDomElement elem = node.toElement();
            frob = elem.text();
        }
        node = node.nextSibling();
    }
    
    // move on to next step if we have our frob
    if( !frob.isEmpty() )
    {
        emit returnedFrob(frob);
    }
    else
    {
        emit commError(i18n("Flickr.com returned empty 'frob'"));
    }
}


/*!
    \fn FlickrComm::handleTokenResponse(const QString &str)
 */
void FlickrComm::handleTokenResponse(const QString &str)
{
    QString nsid;
    QString token;
    QString perms;
    QDomNode node;
    QDomElement root;
    QString username = i18n("Unknown");
    QDomDocument doc("tokenresponse");
    
    // pump str into the XML document
    if( !doc.setContent(str) )
    {
        emit commError(i18n("Invalid response received from Flickr.com."));
        return;
    }
    
    // start at document root, first node
    root = doc.documentElement();
    node = root.firstChild();
    
    // go through each node, should only be one
    while( !node.isNull() )
    {
        // we got our token
        if( node.isElement() && node.nodeName() == "token" )
        {
            QDomElement elem = node.toElement();
            token = elem.text();
        }
        
        // we got our permission
        if( node.isElement() && node.nodeName() == "perms" )
        {
            QDomElement elem = node.toElement();
            perms = elem.text();
        }
        
        // for the user
        if( node.isElement() && node.nodeName() == "user" )
        {
            QDomElement elem = node.toElement();
            username = elem.attribute("username", i18n("Unknown"));
            nsid = elem.attribute("nsid", "");
        }
        
        // step down the node tree
        if( node.isElement() && node.nodeName() == "auth" )
            node = node.firstChild();
        else
            node = node.nextSibling();
    }

    // add the user and make active
    emit returnedToken(username, token, nsid); 
}


/*!
    \fn FlickrComm::handleTagsResponse(const QString &str)
 */
void FlickrComm::handleTagsResponse(const QString &str)
{
    QDomNode node;
    QStringList tags;
    QDomElement root;
    QDomDocument doc("tagsresponse");
    
    
    // pump str into the XML document
    if( !doc.setContent(str) )
    {
        emit commError(i18n("Invalid response received from Flickr.com."));
        return;
    }
    
    // start at document root, first node
    root = doc.documentElement();
    node = root.firstChild();
    
    // go through each node, should only be one
    while( !node.isNull() )
    {
        // we got a tag
        if( node.isElement() && node.nodeName() == "tag" )
        {
            QDomElement elem = node.toElement();
            
            // put quotes around multi word tags
            if( elem.text().contains(QRegExp("\\s+")) )
                tags += QString("\"" + elem.text() + "\"");
            else
                tags += elem.text();
        }
        
        // step down the node tree
        if( node.isElement() && (node.nodeName() == "who" || node.nodeName() == "tags") )
            node = node.firstChild();
        else
            node = node.nextSibling();
    }

    // add the user and make active
    emit returnedTags(tags); 
}


/*!
    \fn FlickrComm::handleStatusResponse(const QString &str)
 */
void FlickrComm::handleStatusResponse(const QString &str)
{
    QString max;
    QString used;
    QDomNode node;
    QString remaining;
    QDomElement root;
    QDomDocument doc("statusresponse");
    
    // pump str into the XML document
    if( !doc.setContent(str) )
    {
        emit commError(i18n("Invalid response received from Flickr.com."));
        return;
    }
    
    // start at document root, first node
    root = doc.documentElement();
    node = root.firstChild();
    
    // go through each node, should only be one
    while( !node.isNull() )
    {
        // we got a tag
        if( node.isElement() && node.nodeName() == "bandwidth" )
        {
            QDomElement elem = node.toElement();
            max = elem.attribute("max", "");
            used = elem.attribute("used", "");
        }
        
        // step down the node tree
        if( node.isElement() && node.nodeName() == "user" )
            node = node.firstChild();
        else
            node = node.nextSibling();
    }

    // calculate remaining bandwidth
    if( max != "" && used != "" )
    {
        float value;
        
        // remaining in Bytes
        value = max.toFloat() - used.toFloat();
        
        // make a nice string
        if( value > (1024.0 * 1024.0 * 1024.0) )
        {
            remaining.sprintf("%.1f", (value / (1024.0 * 1024.0 * 1024.0)));
            remaining += "GB";
        }
        else if( value > (1024.0 * 1024.0) )
        {
            remaining.sprintf("%.1f", (value / (1024.0 * 1024.0)));
            remaining += "MB";
        }
        else if( value > 1024.0 )
        {
            remaining.sprintf("%.1f", (value / 1024.0));
            remaining += "KB";
        }
        else
        {
            remaining = QString::number(value) + "Bytes";
        }            
    }
    else
    {
        remaining = i18n("Unknown");
    }
    
    // emit the users upload remaining
    emit returnedUploadStatus(remaining);    
}


/*!
    \fn FlickrComm::handleUploadResponse(const QString &str)
 */
void FlickrComm::handleUploadResponse(const QString &str)
{
    QString id;
    QDomNode node;
    QDomElement root;
    QDomDocument doc("uploadresponse");
    
    // pump str into the XML document
    if( !doc.setContent(str) )
    {
        emit commError(i18n("Invalid response received from Flickr.com."));
        return;
    }
    
    // start at document root, first node
    root = doc.documentElement();
    node = root.firstChild();
    
    while( !node.isNull( ) )
    {
        // we got our id, good to go
        if( node.isElement( ) && node.nodeName( ) == "photoid" )
        {
            QDomElement elem = node.toElement();
            id = elem.text( );
        }
        node = node.nextSibling();
    }

    // notify that upload was OK
    emit returnedUploadedOK(id);
}


/*!
    \fn FlickrComm::handlePhotosetResponse(const QString &str)
 */
void FlickrComm::handlePhotosetResponse(const QString &str)
{
    QString id;
    QDomNode node;
    QDomElement root;
    QStringList setTitles;
    QDomDocument doc("photosetsresponse");
    
    
    // pump str into the XML document
    if( !doc.setContent(str) )
    {
        emit commError(i18n("Invalid response received from Flickr.com."));
        return;
    }
    
    // Clear the current photoset data
    m_photoSets.clear();
    
    // start at document root, first node
    root = doc.documentElement();
    node = root.firstChild();
    
    // go through each node, should only be one
    while( !node.isNull() )
    {
        // we got a set, store the id and fish out the title
        if( node.isElement() && node.nodeName() == "photoset" )
        {
            QDomElement elem = node.toElement();            
            id = elem.attribute("id");
            
            elem = elem.elementsByTagName("title").item(0).toElement();
            if( id != QString::null )
            {        
                setTitles.append(elem.text());
                m_photoSets.insert(elem.text(), id);
            }
            
        }
         
        // step down the node tree
        if( node.isElement() && node.nodeName() == "photosets" )
            node = node.firstChild();
        else
            node = node.nextSibling();
    }

    // sort the title alphabetically
    setTitles.sort();
    
    // notify interested widgets that new set titles are available    
    emit returnedPhotosets(setTitles, QString::null); 
}

/*!
    \fn FlickrComm::hanldeCreatePhotosetResponse(const QString &str)
 */
void FlickrComm::hanldeCreatePhotosetResponse(const QString &str)
{
    QString id;
    QDomNode node;
    QString newTitle;
    QDomElement root;
    SetData::Iterator it;
    QStringList setTitles;
    QDomDocument doc("photosetsresponse");
    
    // pump str into the XML document
    if( !doc.setContent(str) )
    {
        emit commError(i18n("Invalid response received from Flickr.com."));
        return;
    }
    
    // start at document root, first node
    root = doc.documentElement();
    node = root.firstChild();
    
    // extract the id from the XML document
    while( !node.isNull() )
    {
        // we got our photoset ID, good to go
        if( node.isElement() && node.nodeName() == "photoset" )
        {
            QDomElement elem = node.toElement();
            id = elem.attribute("id");
        }
        node = node.nextSibling();
    }

    // Do two things at once; create the string list with all the photoset titles and
    // find the SetData entry with no ID, this will be the newly created set. Since we
    // are using a map the title will already be sorted as well so no need to sort.
    for( it = m_photoSets.begin(); it != m_photoSets.end(); ++it )
    {
        setTitles.append(it.key());
            
        // Is this the set we are creating
        if( it.data() == QString::null )
        {
           newTitle = it.key();         // store as active set
           m_photoSets[newTitle] = id;  // store the id
        }
    }    
    
    // notify interested widgets that new set titles are available    
    emit returnedPhotosets(setTitles, newTitle); 
}


/*!
    \fn FlickrComm::createPhotoset(const QString &token, const QString &name, const QString &photoID)
 */
void FlickrComm::createPhotoset(const QString &token, const QString &name, const QString &photoID)
{
    ArgMap args;
    
    // Setup the flickr call
    args["method"]              = "flickr.photosets.create";
    args["title"]               = name;
    args["primary_photo_id"]    = photoID;
    args["auth_token"]          = token;

    // Add the set name to the SetData
    m_photoSets[name] = QString::null;
    
    // Send the request
    KIO::TransferJob* job = sendRequest(args);
    m_requests[job] = CREATE_PHOTOSET_REQ;
}


/*!
    \fn FlickrComm::addPhoto2Photoset(const QString &token, const QString &photoset, const QString &photoID)
 */
void FlickrComm::addPhoto2Photoset(const QString &token, const QString &photoset, const QString &photoID)
{
    // check if the set has to be created
    if( !m_photoSets.contains(photoset) )
    {
        createPhotoset(token,photoset,photoID);
    }
    else    // otherwise just add the photo to the set
    {
        ArgMap args;
        
        // Setup the flickr call
        args["method"]      = "flickr.photosets.addPhoto";
        args["photoset_id"] = m_photoSets[photoset];
        args["photo_id"]    = photoID;
        args["auth_token"]  = token;
    
        // Send the request
        KIO::TransferJob* job = sendRequest(args);
        m_requests[job] = ADD_PHOTO2SET_REQ;
    }
}


/*!
    \fn FlickrComm::abortCurrentRequest()
 */
void FlickrComm::abortCurrentRequest()
{
    QMap<KIO::TransferJob*,ResponseType>::iterator iter;
    
    for( iter = m_requests.begin(); iter != m_requests.end(); ++iter )
    {
        iter.key()->kill();
    }
    m_requests.clear();
    m_incomingData.clear();
}


/*!
    \fn FlickrComm::getEXIFData(QFile &in)
 */
QByteArray FlickrComm::getEXIFData(QFile &in)
{
    char data[6];
    ulong dataLength;
    QByteArray array;
    
    if ( !in.open(IO_ReadOnly) )
        return false;
    
    // Based on data from the web EXIF data will be at the start of the JPEG
    // file. All JPEG files start with 0xFFD8. This might be followed by JFIF
    // marker (0xFFE0) and data and then possible EXIF marker (0xFFE1) if
    // there is EXIF data. The (next byte * 256) + next byte will give the length
    // from the start of the EXIF marker to the end of the EXIF data.
        
    // First ensure this is a valid JPEG file. Previous checks should
    // call QImage::imageFormat() so this is really just confirmation.
    ulong len = in.readBlock(data, 6);
    if( len == 6 && (uchar)data[0] == 0xff && (uchar)data[1] == 0xd8 )
    {
        // check for JFIF marker
        if( (uchar)data[2] == 0xff && (uchar)data[3] == 0xe0 )
        {
            // JFIF data is present move beyond this data, raw data length
            // includes the length bytes but not the marker bytes.
            dataLength = ((uchar)data[4] * 256) + (uchar)data[5] + 2;
            
            // Skip ahead in the file to the next marker. Ignore the two
            // bytes that start the file so that the bytes of interest for
            // the EXIF check are 2 and 3 whether JFIF data is present or not 
            in.at(dataLength);
            
            // read in next (marker + length)
            len = in.readBlock(data, 6);
            if( len != 6 )
                return false;
        }
        
        // check for EXIF marker
        if( (uchar)data[2] == 0xff && (uchar)data[3] == 0xe1 )
        {
            // EXIF data is present find out how big it is (includes marker)
            // Raw data length includes the length bytes but not the marker bytes.
            dataLength = ((uchar)data[4] * 256) + (uchar)data[5] + 2;
            
            // Write to the return array all the EXIF data.
            in.at(in.at()-4);                
            array.resize(dataLength);
            len = in.readBlock(array.data(), dataLength);
            if( len != (dataLength) )
            {
                array.resize(0);
            }
        }
    }
    return array;
}


/*!
    \fn FlickrComm::writePhotoWithEXIF(QTextStream &out, QImage &in, QByteArray &exif)
 */
void FlickrComm::writePhotoWithEXIF(QTextStream &out, QImage &in, QByteArray &exif)
{
    int num2Remove;
    ulong arrayPos;
    ulong dataLength;
    QByteArray array;
    QBuffer buffer(array);
    
    // Save the image to the buffer
    buffer.open(IO_WriteOnly);
    in.save(&buffer, "JPEG");
    buffer.close();

    // Write the JPEG standard first two bytes to the stream
    out.writeRawBytes(&array[0], 2);
    arrayPos = 2;
    
    // Write JFIF data if present
    if( (uchar)array[2] == 0xFF && (uchar)array[3] == 0xE0 )
    {
        // Remember to add 2 bytes since the data length bytes are
        // included but not the marker bytes which also want included
        dataLength = ((uchar)array[4] * 256) + (uchar)array[5] + 2;
        out.writeRawBytes(&array[2], dataLength);
        arrayPos += dataLength;
    }
    
    // Write our passed in EXIF data
    out.writeRawBytes(exif.data(), exif.size());
    
    // Skip current EXIF data if present
    if( (uchar)array[arrayPos] == 0xFF && (uchar)array[arrayPos+1] == 0xE1 )
    {
        // Remember to add 2 bytes since the data length bytes are
        // included but not the marker bytes which also want included
        arrayPos += ((uchar)array[num2Remove+2] * 256) + (uchar)array[num2Remove+3] + 2;
    }
        
    // Write everything after the arrayPos bytes.
    out.writeRawBytes(&array[arrayPos], array.size() - arrayPos);
}


#include "flickrcomm.moc"
