/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : Specialized list to display a list of photos
 * 
 * Copyright 2005 by M. Asselstine

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */
#include "photolistview.h"

#include <qpen.h>
#include <qrect.h>
#include <qmime.h>
#include <qevent.h>
#include <qstring.h>
#include <qwidget.h>
#include <klocale.h>
#include <qwmatrix.h>
#include <qpainter.h>
#include <qpalette.h>
#include <qpopupmenu.h>
#include <kiconloader.h>
#include <kmainwindow.h>
#include <kapplication.h>
#include <qapplication.h>
#include <kxmlguiclient.h>
#include <kstandarddirs.h>
#include <kio/previewjob.h>
#include <kxmlguifactory.h>

#include "kflickrpart.h"


/*!
    \fn PhotoListView::PhotoListView(QWidget *parent, const char *name)
 */
PhotoListView::PhotoListView(QWidget *parent, const char *name)
    : KListView(parent, name)
{
    // disable sorting
    setSorting(-1);
    
    // load the border pixmap
    KStandardDirs *dirs = KApplication::kApplication()->dirs();
    m_border = QPixmap( dirs->findResource( "data", "kflickr/border.png" ) );
    
    // Signal - Slot connections
    connect(this, SIGNAL(selectionChanged()), this, SLOT(slotSelectionUpdate()));
    connect(this, SIGNAL(rightButtonClicked(QListViewItem *, const QPoint &, int)),
            this, SLOT(showRMBMenu(QListViewItem *, const QPoint &, int)));
}


/*!
    \fn PhotoListView::~PhotoListView()
 */
PhotoListView::~PhotoListView()
{
    // kill any pending IO Slave jobs
    KIO::Job *job = m_previewJobs.first();
    while( job != 0L )
    {
        job->kill();
        job = m_previewJobs.next();
    }
}


/*!
    \fn PhotoListView::dragObject()
 */
QDragObject* PhotoListView::dragObject()
{
    if (!currentItem())
        return 0;
        
    return new QStoredDrag("application/x-photolistviewitem", viewport());
}


/*!
    \fn PhotoListView::resizeEvent()
 */
void PhotoListView::resizeEvent(QResizeEvent * e)
{
    // Call base class implementation
    KListView::resizeEvent(e);
    
    // If list is empty, invalidate everything to display
    // the information text as a nice touch
    if( !firstChild() )
    {
        viewport()->repaint(viewport()->rect());
    }        
}


/*!
    \fn PhotoListView::viewportPaintEvent()
 */
void PhotoListView::viewportPaintEvent(QPaintEvent *e)
{
    // Call base class implementation
    KListView::viewportPaintEvent(e);
    
    // If list is empty, invalidate everything to display
    // the information text as a nice touch
    if( !firstChild() )
    {
        QString str;
        QRect rect( 0, 0, 300, 150 );
    
    
        // Initialize painter device
        QPainter painter(viewport());
        
        // Setup the pen
        QPen pen(colorGroup().mid(), 4);
        painter.setPen(pen);
        
        // position our rect
        rect.moveTopLeft(viewport()->rect().center() - QPoint(rect.width()/2, rect.height()/2));
        
        // draw outline rect
        painter.drawRoundRect( rect, 4, 8 );
        
        // draw our text
        str = i18n("KFlickr");   
        QFont font("Helvetica", 32, QFont::Bold);
        painter.setFont(font);
        QRect brect = painter.boundingRect(rect, AlignHCenter|AlignTop, str);
        painter.drawText(rect, AlignHCenter|AlignTop, str);
        
        str = i18n("A Standalone Flickr.com Uploader for KDE\nhttp://kflickr.sourceforge.net");
        font.setPointSize(8);
        painter.setFont(font);
        painter.drawText(rect, AlignHCenter|AlignVCenter, str);
        
        str = i18n("(Drag-n-Drop your photos here)");
        painter.drawText(rect, AlignHCenter|AlignBottom, str);
    }
}
 
 
/*!
    \fn PhotoListView::acceptDrag()
 */
bool PhotoListView::acceptDrag( QDropEvent *event ) const
{
    bool acceptable = false;
    
    if( acceptDrops() && itemsMovable() )
    {
        if( event->provides( "text/plain" ) ||  event->provides( "text/uri-list") )
            acceptable = true;
        else if( event->provides("application/x-photolistviewitem") && event->source() == viewport() )
            acceptable = true;
    }
    return acceptable;
}


/*!
    \fn PhotoListView::slotSelectionUpdate()
 */
void PhotoListView::slotSelectionUpdate()
{
    bool anySelected = FALSE;
    bool singleSelection = FALSE;
    QListViewItem *lvi = firstChild();
    
    
    // while there are valid items
    while( lvi != 0L )
    {
        // check if this item is selected
        if( lvi->isSelected() )
        {
            // if so we have at least one selection
            anySelected = TRUE;
            if( !singleSelection )
            {
                singleSelection = TRUE;
            }
            else
            {
                singleSelection = FALSE;
                break;
            }
        }
        // get the next item to check
        lvi = lvi->itemBelow();
    }
    emit hasValidSelection( anySelected );
    emit hasSingleSelection( singleSelection );
}


/*!
    \fn PhotoListView::removeSelected()
 */
void PhotoListView::removeSelected()
{
    QListViewItem *next;
    QListViewItem *lvi = firstChild();
   
    // while there are valid items
    while( lvi != 0L )
    {
        // store the next item
        next = lvi->itemBelow();
        
        // check if this item is selected
        if( lvi->isSelected() )
        {
            delete lvi;
        }
        
        // move on to next item
        lvi = next;
    }
   
}


/*!
    \fn PhotoListView::lastPhoto()
 */
PhotoListViewItem* PhotoListView::lastPhoto() const
{
    return dynamic_cast<PhotoListViewItem*>(lastItem());
}

/*!
    \fn PhotoListView::firstSelectedItem()
 */
PhotoListViewItem * PhotoListView::firstSelectedPhoto() const
{
    QListViewItem *lvi = firstChild();
     
    // while there are valid items
    while( lvi != 0L )
    {
        // check if this item is selected
        if( lvi->isSelected() )
        {
            // search no further
            break;
        }
        
        // move on to next item
        lvi = lvi->itemBelow();
    }
    return dynamic_cast<PhotoListViewItem*>(lvi);
}


/*!
    \fn PhotoListView::nextSelectedPhoto( PhotoListViewItem *current )
 */
PhotoListViewItem* PhotoListView::nextSelectedPhoto( PhotoListViewItem *current ) const
{
    QListViewItem *lvi = current->itemBelow();
     
    // while there are valid items
    while( lvi != 0L )
    {
        // check if this item is selected
        if( lvi->isSelected() )
        {
            // search no further
            break;
        }
        
        // move on to next item
        lvi = lvi->itemBelow();
    }
    return dynamic_cast<PhotoListViewItem*>(lvi);
}


/*!
    \fn PhotoListView::numSelected()
 */
int PhotoListView::numSelected() const
{
    int count = 0;
    QListViewItem *lvi = firstChild();
     
    // while there are valid items
    while( lvi != 0L )
    {
        // check if this item is selected
        if( lvi->isSelected() )
        {
            // add to count
            ++count;
        }
        
        // move on to next item
        lvi = lvi->itemBelow();
    }
    return count;
}


/*!
    \fn PhotoListView::getNextPhoto( const PhotoListViewItem &current ) const
 */
PhotoListViewItem* PhotoListView::getNextPhoto( PhotoListViewItem *current ) const
{
    PhotoListViewItem *photo;
    
    if( current == 0L )
        photo = dynamic_cast<PhotoListViewItem*>(firstChild());
    else
        photo = dynamic_cast<PhotoListViewItem*>(current->itemBelow());
        
    return photo;        
}


/*!
    \fn PhotoListView::showRMBMenu(QListViewItem*, const QPoint &pt, int)
 */
void PhotoListView::showRMBMenu(QListViewItem*, const QPoint &pt, int)
{
    if( childCount() > 0)
    {
        KXMLGUIClient* gui = static_cast<KXMLGUIClient*>(kflickrPart::Instance());
        QPopupMenu *pop = (QPopupMenu*)gui->factory()->container("listitem_popup", gui);
        pop->popup(pt);
    }
}


/*!
    \fn void PhotoListView::getPreview(const KURL& url)
 */
void PhotoListView::getPreview(const KURL& url)
{
    KURL::List urls;
    urls.append(url);
    getPreviews(urls);
}


/*!
    \fn void PhotoListView::getPreviews(const KURL::List& urls)
 */
void PhotoListView::getPreviews(const KURL::List& urls)
{   
    KIO::PreviewJob *job;
    
    // start the IO slave
    job = KIO::filePreview(urls, 140, 90, 0, 0, true, false);
    
    // make notification connections
    connect(job, SIGNAL(result(KIO::Job *)), this, SLOT(slotResult(KIO::Job *)));
    connect(job, SIGNAL(gotPreview(const KFileItem*, const QPixmap&)),
                 SLOT(gotPreview(const KFileItem*, const QPixmap&)));
    connect(job, SIGNAL(failed(const KFileItem*)), this, SLOT(slotFailed(const KFileItem*)));
    
    // add to jobs DB
    m_previewJobs.append(job);
}


/*!
    \fn PhotoListView::slotResult(KIO::Job *job)
 */
void PhotoListView::slotResult(KIO::Job *job)
{
    // remove from our records
    if( m_previewJobs.contains(job) )
        m_previewJobs.remove(job);
}


/*!
    \fn PhotoListView::slotFailed(const KFileItem *i)
 */
void PhotoListView::slotFailed(const KFileItem *i)
{
    // find list item to match i
    PhotoListViewItem *photo = dynamic_cast<PhotoListViewItem*>(firstChild());
    
    while( photo != 0L )
    {
        if( photo->url() == i->url() )
        {    
            // show broken image icon
            photo->m_preview = SmallIcon("file_broken", KIcon::SizeHuge, KIcon::DefaultState);
            repaintItem(photo);
            
            // Don't put a break here, although probably not used
            // this would prevent having two of the same photos
            // with the same filename.
        }
        photo = dynamic_cast<PhotoListViewItem*>(photo->itemBelow());
    }
}


/*!
    \fn PhotoListView::gotPreview(const KFileItem *i, const QPixmap &pix)
 */
void PhotoListView::gotPreview(const KFileItem *i, const QPixmap &pix)
{
    // find list item to match i
    PhotoListViewItem *photo = dynamic_cast<PhotoListViewItem*>(firstChild());
    
    while( photo != 0L )
    {
        if( photo->url() == i->url() )
        {    
            // show broken image icon
                photo->m_preview = pix;
                repaintItem(photo);

            
            // Don't put a break here, although probably not used
            // this would prevent having two of the same photos
            // with the same filename.
        }
        photo = dynamic_cast<PhotoListViewItem*>(photo->itemBelow());
    }
}


/*!
    \fn PhotoListViewItem::PhotoListViewItem(KListView *parent, const KURL &url)
 */
PhotoListViewItem::PhotoListViewItem(KListView *parent, const KURL &url)
  : QObject(), KListViewItem(parent) 
{
    init(url);
}

/*!
    \fn PhotoListViewItem::PhotoListViewItem(KListView *parent, const KURL &url, QListViewItem *after)
 */
PhotoListViewItem::PhotoListViewItem(KListView *parent, const KURL &url, QListViewItem *after)
  : QObject(), KListViewItem(parent, after) 
{
    init(url);
}

/*!
    \fn PhotoListViewItem::init()
 */
void PhotoListViewItem::init(const KURL &url)
{
    // variable inits
    m_URL = url;
    m_rotation = 0;
    m_public = TRUE;
    m_family = FALSE;
    m_friends = FALSE;
    m_size = i18n("Original");
    m_preview = SmallIcon( "camera", KIcon::SizeHuge, KIcon::DefaultState );
    
    // make dragable
    setDragEnabled(true);
}

PhotoListViewItem::~PhotoListViewItem()
{
}


/*!
    \fn PhotoListViewItem::setHeight( int height )
 */
void PhotoListViewItem::setHeight( int height )
{
    height = 115;
    KListViewItem::setHeight( height );
}


/*!
    \fn PhotoListViewItem::paintCell( QPainter *, const QColorGroup &cg, int column, int width, int alignment )
 */
void PhotoListViewItem::paintCell( QPainter *p, const QColorGroup &cg, int column, int width, int alignment )
{
    QRect cr = QRect( 0, 0, width, height() );
    
    p->setPen( cg.foreground() );
    p->setBackgroundColor( backgroundColor() );
    
    if( isSelected() )
        p->setBackgroundColor( cg.highlight() );
        
    p->eraseRect( cr );
    
    if( column == 0 )
    {
        QSize previewSize = QSize(m_preview.width(), m_preview.height()).boundedTo(QSize(140,90));
        
        p->drawPixmap( 0, 0, static_cast<PhotoListView*>(listView())->getBorderImage() );
        p->drawPixmap( cr.width() / 2 - previewSize.width() / 2,
                       cr.height() / 2 - previewSize.height() / 2,
                       m_preview, 0, 0, 140, 90 );
    }
    else
    {
        QString str;
        QRect br = cr;
        
        if( column == 1 )
            str = accessString();
        else if( column == 2 )
            str = m_title;
        else if( column == 3 )
            str = m_size.section(' ', 0, 0);
        else if( column == 4 )
            str = m_tags.join( " " );
        else if( column == 5 )
            str = m_desc;
           
        setText( column, str );
        KListViewItem::paintCell( p, cg, column, width, alignment );
    }
}


/*!
    \fn PhotoListViewItem::width( const QFontMetrics &, const QListView *, int c ) const
 */
int PhotoListViewItem::width( const QFontMetrics &fm, const QListView *lv, int c ) const
{
    if( c == 0 )
        return 175;
        
    return KListViewItem::width( fm, lv, c );
}


/*!
    \fn PhotoListViewItem::setPublic( bool isPublic )
 */
void PhotoListViewItem::setPublic( bool isPublic )
{
    m_public = isPublic;
    listView()->repaintItem( this );
}


/*!
    \fn PhotoListViewItem::setFamily( bool isFamily )
 */
void PhotoListViewItem::setFamily( bool isFamily )
{
    m_family = isFamily;
    listView()->repaintItem( this );
}


/*!
    \fn PhotoListViewItem::setFriends( bool isFriends )
 */
void PhotoListViewItem::setFriends( bool isFriends )
{
    m_friends = isFriends;
    listView()->repaintItem( this );
}


/*!
    \fn PhotoListViewItem::setDescription( const QString &str )
 */
void PhotoListViewItem::setDescription( const QString &str )
{
    m_desc = str;
    listView()->repaintItem( this );
}


/*!
    \fn PhotoListViewItem::setTags( const QStringList &strlst )
 */
void PhotoListViewItem::setTags( const QStringList &strlst )
{
    m_tags = strlst;
    listView()->repaintItem( this );
}


/*!
    \fn PhotoListViewItem::setTitle( const QString &str )
 */
void PhotoListViewItem::setTitle( const QString &str )
{
    m_title = str;
    listView()->repaintItem( this );
}


/*!
    \fn PhotoListViewItem::setSize(const QString &size)
 */
void PhotoListViewItem::setSize(const QString &size)
{
    m_size = size;
    listView()->repaintItem( this );
}


/*!
    \fn PhotoListViewItem::previewImage()
 */
QPixmap PhotoListViewItem::previewImage() const
{
    return m_preview;
}


/*!
    \fn PhotoListViewItem::title() const
 */
QString PhotoListViewItem::title() const
{
    return m_title;
}


/*!
    \fn PhotoListViewItem::size() const
 */
QString PhotoListViewItem::size() const
{
    return m_size;
}


/*!
    \fn PhotoListViewItem::description() const
 */
QString PhotoListViewItem::description() const
{
    return m_desc;
}


/*!
    \fn PhotoListViewItem::tags() const
 */
QStringList PhotoListViewItem::tags() const
{
    return m_tags;
}


/*!
    \fn PhotoListViewItem::isPublic() const
 */
bool PhotoListViewItem::isPublic() const
{
    return m_public;
}


/*!
    \fn PhotoListViewItem::isFriends() const
 */
bool PhotoListViewItem::isFriends() const
{
    return m_friends;
}


/*!
    \fn PhotoListViewItem::isFamily() const
 */
bool PhotoListViewItem::isFamily() const
{
    return m_family;
}


/*!
    \fn PhotoListViewItem::url()
 */
const KURL& PhotoListViewItem::url() const
{
    return m_URL;
}


/*!
    \fn PhotoListViewItem::rotation()
 */
int PhotoListViewItem::rotation() const
{
    return m_rotation;
}


/*!
    \fn PhotoListViewItem::accessString() const
 */
QString PhotoListViewItem::accessString() const
{
    QString str;
    
    if( m_public )
    {
        str = i18n("Public");
    }
    else
    {
        str = i18n("Priv");
        if( m_family )
            str += i18n("/Fam");
        if( m_friends )
            str += i18n("/Fnds");
    }
    return str;
}


/*!
    \fn PhotoListViewItem::rotatePhoto()
 */
void PhotoListViewItem::rotatePhoto()
{
    if( m_rotation == 0 )
        m_rotation = 270;
    else
        m_rotation -= 90;
        
    QWMatrix matrix;
    matrix = matrix.rotate(-90);
    m_preview = m_preview.xForm(matrix);
    listView()->repaintItem( this );
}

#include "photolistview.moc"
