/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : The main widget, show photo properties
 *
 * Copyright 2005 by M. Asselstine
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include <kdialog.h>
#include <klocale.h>

#include <qlabel.h>
#include <iostream>

#include "kflickrwidget.h"
#include "photolistview.h"
#include "authcomplete.h"
#include "photoproperties.h"
#include "authquestiondlg.h"

#include <krun.h>
#include <qvbox.h>
#include <qlabel.h>
#include <qframe.h>
#include <qevent.h>
#include <qlayout.h>
#include <kconfig.h>
#include <qstring.h>
#include <qheader.h>
#include <kglobal.h>
#include <kguiitem.h>
#include <qvariant.h>
#include <qcombobox.h>
#include <kprogress.h>
#include <qvaluelist.h>
#include <qsizepolicy.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <qscrollview.h>
#include <qdragobject.h>
#include <qpushbutton.h>
#include <kpushbutton.h>
#include <qmessagebox.h>
#include <kapplication.h>

kflickrWidget::kflickrWidget(QWidget* parent, const char* name, WFlags fl)
    : kflickrWidgetUI(parent,name,fl)
    , DCOPObject("kflickrWidgetInterface")
    , m_currentUserIndex(-1)
    , m_progressDlg(0L)
    , m_uploadInProgress(false)
{
    QValueList<int> widths;

    // setup to retrieve saved settings
    KConfig *config = kapp->config();

    // config settings group
    config->setGroup("KFlickrWidget");

    // Stored user data
    m_users->insertStringList(config->readListEntry("users", QStringList()));
    m_userNSIDs= config->readListEntry( "user_nsids", QStringList() );
    m_userTokens = config->readListEntry( "user_tokens", QStringList() );

    m_previousBtn->setAccel(Key_PageUp);
    m_previousBtn->setGuiItem(KGuiItem(i18n("Previous"),"up"));
    m_nextBtn->setAccel(Key_PageDown);
    m_nextBtn->setGuiItem(KGuiItem(i18n("Next"),"down"));

    // Set GUI icons
    m_addBtn->setGuiItem(KGuiItem(i18n("&Add"), "add"));
    m_removeBtn->setGuiItem(KGuiItem(i18n("&Remove"), "remove"));
    m_uploadBtn->setGuiItem(KGuiItem(i18n("&Upload"), "up"));

    // Photo list
    m_photolist = new PhotoListView( m_listHolder, "m_photolist" );
    QVBoxLayout* listLayout = new QVBoxLayout(m_listHolder, 0, 6);
    listLayout->addWidget(m_photolist);

    m_photolist->addColumn( i18n("Photo") );

    m_photolist->header()->setResizeEnabled( false, 0 );
    m_photolist->setAllColumnsShowFocus( TRUE );
    m_photolist->setAcceptDrops( TRUE );
    m_photolist->setDropVisualizer( TRUE );
    m_photolist->setSelectionMode( QListView::Extended );

    // Photo Properties
    m_photoProps = new PhotoProperties(m_propsHolder, "m_photoProps");
    QVBoxLayout* propsLayout = new QVBoxLayout(m_propsHolder, 0, 6);
    propsLayout->addWidget(m_photoProps);

    // widget signals and slots connections
    connect(m_addBtn, SIGNAL(clicked()), SLOT(addPhotos()));
    connect(m_removeBtn, SIGNAL(clicked()), SLOT(removePhotos()));
    connect(m_uploadBtn, SIGNAL(clicked()), SLOT(uploadPhotos()));
    connect(m_authNew, SIGNAL(clicked()), SLOT(authorizeNewUser()));
    connect(m_users, SIGNAL(activated(const QString &)), SLOT(setActiveUser(const QString &)));

    connect(m_photolist, SIGNAL(selectionChanged()), SLOT(editSelected()));
    connect(m_photolist, SIGNAL(selectionChanged()), SLOT(updateAvailableActions()));
    connect(m_photolist, SIGNAL(dropped(QDropEvent*,QListViewItem*)), SLOT(dropSlot(QDropEvent*,QListViewItem*)) );
    connect(m_photolist, SIGNAL(itemAdded(QListViewItem*)), SLOT(updateCount(QListViewItem*)));
    connect(m_photolist, SIGNAL(itemRemoved(QListViewItem*)), SLOT(updateCount(QListViewItem*)));

    connect(m_nextBtn, SIGNAL(clicked()), m_photolist, SLOT(selectNext()));
    connect(m_previousBtn, SIGNAL(clicked()), m_photolist, SLOT(selectPrevious()));

    // Communications signals and slots connections
    connect(&m_comm, SIGNAL(commError(const QString&)), SLOT(handleCommError(const QString&)));
    connect(&m_comm, SIGNAL(returnedFrob(const QString&)), SLOT(doUserAuthentication(const QString&)));
    connect(&m_comm, SIGNAL(returnedToken(const QString&, const QString &, const QString&)), SLOT(addUser(const QString&, const QString&, const QString&)));
    connect(&m_comm, SIGNAL(returnedTags(const QStringList&)), m_photoProps, SLOT(setAvailableTags(const QStringList &)));
    connect(&m_comm, SIGNAL(returnedPhotosets(const QStringList&, const QString&)), m_photoProps, SLOT(setPhotosets(const QStringList&, const QString&)));
    connect(&m_comm, SIGNAL(returnedUploadStatus(const QString&)), SLOT(newBandwidthValue(const QString&)));
    connect(&m_comm, SIGNAL(returnedUploadedOK(const QString&)), SLOT(photoUploadedOK(const QString&)));
    connect(&m_comm, SIGNAL(returnedLicenses(const QStringList&)), m_photoProps, SLOT(setLicenses(const QStringList&)));

    connect(&m_timer, SIGNAL(timeout()), SLOT(refreshCount()));

    // Update the available licenses by making a request to flickr
    m_comm.sendLicensesRequest();

    // Finally set the active user if one already authorized
    if( config->readEntry( "current_user", QString( "" ) ) != "" )
        setActiveUser( config->readEntry( "current_user" ) );
}

kflickrWidget::~kflickrWidget()
{
    QStringList users;
    QValueList<int> widths;
    KConfig *config = kapp->config();
    QHeader *hdr = m_photolist->header();

    // set the group
    config->setGroup("KFlickrWidget");

    // save list column widths
    for( int i = 0; i < hdr->count(); ++i )
        widths.append( hdr->sectionSize( i ) );
    config->writeEntry( "column_widths", widths );

    // save users
    for( int i = 0; i < m_users->count(); ++i )
        users += m_users->text( i );
    config->writeEntry( "users", users );

    // save user tokens
    config->writeEntry( "user_nsids", m_userNSIDs );
    config->writeEntry( "user_tokens", m_userTokens );

    // save current user
    config->writeEntry( "current_user", m_users->currentText() );

    // delete stuff
    if( m_progressDlg != 0L )
        delete m_progressDlg;
}

void kflickrWidget::addPhoto(const KURL &url)
{
    PhotoListViewItem *item = new PhotoListViewItem(m_photolist, url);

    if( item )
    {
        updateAvailableActions();
    }
}

void kflickrWidget::addPhoto(const KURL &url,
                             const QString &title,
                             const QStringList &tags,
                             const QString &desc)
{
    // Add the photo to the list
    PhotoListViewItem *item = new PhotoListViewItem(m_photolist, url);

    if( item )
    {
        if( title != QString::null )    item->photo().title(title);
        if( tags.count() > 0 )          item->photo().tags(tags);
        if( desc != QString::null )     item->photo().description(desc);

        updateAvailableActions();
    }
}

void kflickrWidget::addPhotos()
{
    // allow user to select some jpeg files to add
    KURL::List urls =  KFileDialog::getOpenURLs( ":OpenPhoto", "*.jpg *.png *.gif|Photo Files",
                                       this, i18n("Select Image(s)") );

    addPhotos(urls);
}

void kflickrWidget::addPhotos(const KURL::List &urls)
{
    if( !urls.empty() )
    {
        KURL::List::const_iterator it;

        // add each selection to the photo list
        for( it = urls.begin(); it != urls.end(); ++it )
        {
            addPhoto(*it);
        }
    }
}

void kflickrWidget::uploadPhotos()
{
    if( m_photolist->childCount() > 0 )
    {
        m_uploadInProgress = true;
        m_photolist->clearSelection();

        // display the progress dialog
        showUploadProgress(m_photolist->childCount());

        // upload the first photos one at a time
        uploadNextPhoto();
    }
}


/*!
    \fn kflickrWidget::uploadNextPhoto()
 */
void kflickrWidget::uploadNextPhoto()
{
    // get last photo in list
    PhotoListViewItem *item = dynamic_cast<PhotoListViewItem*>(m_photolist->lastItem());

    if( item )
    {
        m_comm.sendPhoto(m_userTokens[m_currentUserIndex], &(item->photo()));
        updateUploadProgress(i18n("Sending Photo - %1").arg(item->photo().URL().filename()));
    }
    else
    {
        // The progress dialog will sometime pop back up if
        // it is left with steps to do, so update to make %100
        updateUploadProgress(i18n("Done"));
        m_uploadInProgress = false;
        hideUploadProgress();

        m_photoProps->clearAndDisable();
        updateAvailableActions();
    }
}


/*!
    \fn kflickrWidget::photoUploadedOK(const QString &photoID)
 */
void kflickrWidget::photoUploadedOK(const QString &photoID)
{

    // update unused bandwidth
    m_comm.sendUpStatusRequest(m_userTokens[m_currentUserIndex]);

    // We know that the last picture uploaded was the last photograph in the list.
    /// @todo Not actually true if the previous file failed to upload this may not be the case
    PhotoListViewItem *item = dynamic_cast<PhotoListViewItem*>(m_photolist->lastItem());
    if( item )
    {
        // Add photo to a photoset
        if( item->photo().photoset() != i18n("<photostream only>") )
        {
           m_comm.addPhoto2Photoset(m_userTokens[m_currentUserIndex], item->photo().photoset(), photoID);
        }

        // Set the license for the photograph
        m_comm.setPhotoLicense(m_userTokens[m_currentUserIndex], item->photo().license(), photoID);

        // Remove the photograph as it is now uploaded
        delete item;
    }

    // upload the next one
    uploadNextPhoto();
}


/*!
    \fn kflickrWidget::cancelUpload()
 */
void kflickrWidget::cancelUpload()
{
    m_uploadInProgress = false;
    m_comm.abortCurrentRequest();
    hideUploadProgress();
}


/*!
    \fn kflickrWidget::slotAuthorizeNewUser()
 */
void kflickrWidget::authorizeNewUser()
{
    AuthQuestionDlg dlg( this );

    // inform the user and check if we should continue
    if( dlg.exec() != QDialog::Accepted )
    {
        return;
    }

    // Get our FROB from flickr.com
    m_comm.sendFROBRequest();
}


/*!
    \fn kflickrWidget::doUserAuthentication( const QString &frob )
 */
void kflickrWidget::doUserAuthentication( const QString &frob )
{
    AuthCompleteDlg dlg( this );

    // Open browser etc... for web authentication
    m_comm.doWebAuthentication( frob );

    // Wait for user to login and such at Flickr.com
    if( dlg.exec() != QDialog::Accepted )
    {
        return;
    }

    // Request our TOKEN from flickr.com
    m_comm.sendTokenRequest( frob );
}

void kflickrWidget::newBandwidthValue(const QString &val)
{
    emit bandwidthValue(val);
}

void kflickrWidget::updateCount(QListViewItem*)
{
    m_timer.start(500, true);
}

void kflickrWidget::refreshCount()
{
    emit photoCount(m_photolist->childCount());
}

void kflickrWidget::editSelected()
{
    // If we have a single photo selected populate the properties widget
    // with the selected photo's properties. If we have more then one photo
    // selected do a batch edit in the properties widget. If we have no
    // photos selected disable the properties widget widgets.
    if( m_photolist->numSelected() == 1 )
    {
        QListViewItemIterator it(m_photolist, QListViewItemIterator::Selected);
        m_photoProps->editSinglePhoto(static_cast<PhotoListViewItem*>(it.current())->photo());
    }
    else if( m_photolist->numSelected() > 1 ) 
    {
        QPtrList<Photo> photographs;
        QPtrList<QListViewItem> selected = m_photolist->selectedItems();
        PhotoListViewItem* item = static_cast<PhotoListViewItem*>(selected.first());
        while( item )
        {
            photographs.append(&(item->photo()));
            item = static_cast<PhotoListViewItem*>(selected.next());
        }
        m_photoProps->editPhotoBatch(photographs);
    }
    else 
    {
        m_photoProps->clearAndDisable();
    }
}

void kflickrWidget::addUser( const QString &name, const QString &token, const QString &nsid )
{
    int i;

    // check if the user already exists
    for( i = 0; i < m_users->count(); ++i )
    {
        if( name == m_users->text( i ) )
        {
            break;
        }
    }

    // brand new user, existing users retain their current token
    if( i == m_users->count() )
    {
        m_userNSIDs.append( nsid );
        m_users->insertItem( name );
        m_userTokens.append( token );
    }

    // make the active user
    setActiveUser( name );

    // Notify user
    QMessageBox::information( this, i18n("New User"),
                i18n("A new user '%1' has been added successfully.").arg(name) );
}


/*!
    \fn kflickrWidget::setActiveUser( const QString & )
 */
void kflickrWidget::setActiveUser( const QString &name )
{
    // update combobox
    if( m_users->currentText() != name )
        m_users->setCurrentText( name );

    // update current user index, must always do this here
    m_currentUserIndex = m_users->currentItem();

    // request users tags
    m_comm.sendTagsRequest(m_userTokens[m_currentUserIndex], m_userNSIDs[m_currentUserIndex]);

    // request user photosets
    m_comm.sendPhotosetsRequest(m_userTokens[m_currentUserIndex], m_userNSIDs[m_currentUserIndex]);

    // update unused bandwidth
    m_comm.sendUpStatusRequest(m_userTokens[m_currentUserIndex]);

    // update the button states
    updateAvailableActions();
}

void kflickrWidget::removePhotos()
{
    // remove selected photos
    m_photolist->removeSelected();

    // update the button states
    updateAvailableActions();
}


/*!
    \fn kflickrWidget::dropSlot( QDropEvent *e, QListViewItem *after )
 */
void kflickrWidget::dropSlot( QDropEvent *e, QListViewItem *after )
{
    QString text;
    QStringList parts;
    PhotoListViewItem* afterItem = dynamic_cast<PhotoListViewItem*>(after);

    // make sure the drag is of the right type
    if ( QTextDrag::decode( e, text ) )
    {
        // there might be more than one file, split it up
        parts = QStringList::split( "\n", text );

        for ( QStringList::Iterator it = parts.begin(); it != parts.end(); ++it )
        {
            // turn into a URL
            KURL url( (*it).stripWhiteSpace() );

            // ensure it is a jpeg type file
            if( (url.protocol() == "file") &&
                ( (url.filename()).endsWith( ".jpg", false ) ||
                  (url.filename()).endsWith( ".png", false ) ||
                  (url.filename()).endsWith( ".gif", false ) ) )
            {
                // If afterItem is null we are dropping at the start of the list or in an empty
                // list. This will result in the dropped item appearing first in the list.
                new PhotoListViewItem(m_photolist, url, afterItem);
            }
        }
    }

    // update the button states
    updateAvailableActions();
}

void kflickrWidget::hideUploadProgress()
{
    // hide the progress dialog
    if( m_progressDlg != 0L && m_progressDlg->isVisible() )
    {
        m_progressDlg->hide();
    }
}

void kflickrWidget::showUploadProgress(int numPhotos)
{
    // create the progress dialog if it does not yet exist
    if( m_progressDlg == 0L )
    {
        m_progressDlg = new KProgressDialog(this, "m_progressDlg");
        m_progressDlg->progressBar()->setTextEnabled(false);
        m_progressDlg->setCaption(i18n("Upload Progress"));
        m_progressDlg->setAutoClose(false);
        m_progressDlg->setModal(true);

        // Progress dialog connections
        connect(m_progressDlg, SIGNAL(cancelClicked()), this, SLOT(cancelUpload()));
    }

    // tune the progress dialog accordingly
    m_progressDlg->progressBar()->setTotalSteps(numPhotos);
    m_progressDlg->progressBar()->setValue(0);
    m_progressDlg->show();
}

void kflickrWidget::updateUploadProgress(const QString &str)
{
    // set text
    m_progressDlg->setLabel(str);

    // move progress bar
    if( m_progressDlg->progressBar()->totalSteps() != m_photolist->childCount())
        m_progressDlg->progressBar()->advance(1);
}

void kflickrWidget::handleCommError(const QString &errstr)
{
    QMessageBox::critical(this, i18n("Error"), errstr);

    // Are we in the middle of an upload
    if( m_uploadInProgress )
    {
        // Reset upload things
        m_uploadInProgress = false;
        hideUploadProgress();

        // Inform the user as to what has happenned and how to continue
        QMessageBox::information(this, i18n("Information"),
            i18n("Due to an unexpected communications error your upload has been terminated.") +
            i18n("\nYou may press the \"Upload\" button to retry uploading the unsent photos.") +
            i18n("\nYou may also want to check your flickr site to ensure no partial photos exist."));
    }
}

void kflickrWidget::updateAvailableActions()
{
    // Upload button enabled if we have a valid user and at least one photo
    m_uploadBtn->setEnabled(FALSE);
    if( m_photolist->firstChild() && m_currentUserIndex != -1 )
    {
        m_uploadBtn->setEnabled(TRUE);
    }

    // remove action is only available if at least one photo selected
    m_removeBtn->setEnabled(FALSE);
    if( m_photolist->numSelected() > 0 )
    {
        m_removeBtn->setEnabled(TRUE);
    }

    // enable next and previous buttons according to the selection
    m_nextBtn->setEnabled(FALSE);
    m_previousBtn->setEnabled(FALSE);
    if( m_photolist->numSelected() == 1 )
    {
        QListViewItem* item = QListViewItemIterator(m_photolist, QListViewItemIterator::Selected).current();
        if( item != m_photolist->firstChild() )
        {
            m_previousBtn->setEnabled(TRUE);
        }
        if( item != m_photolist->lastItem() )
        {
            m_nextBtn->setEnabled(TRUE);
        }
    }
}


#include "kflickrwidget.moc"
