/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 22-11-2006
 * Description : Implements a list of photographs.
 *
 * Copyright 2005 by M. Asselstine
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */
#ifndef PHOTOLIST_H
#define PHOTOLIST_H

#include <kurl.h>

#include <qtimer.h>
#include <qobject.h>
#include <qstring.h>
#include <qpixmap.h>
#include <qvariant.h>
#include <qptrlist.h>
#include <qstringlist.h>

// Forward declarations
class Photo;
class KFileItem;

// Update bitflag defines
#define PHOTO_URL           1
#define PHOTO_PERMISSIONS   2
#define PHOTO_ROTATION      4
#define PHOTO_DESCRIPTION   8
#define PHOTO_SIZE          16
#define PHOTO_TITLE         32
#define PHOTO_LICENSE       64
#define PHOTO_PHOTOSET      128
#define PHOTO_TAGS          256
#define PHOTO_SELECTION     512
#define PHOTO_PREVIEW       1024

/**
  * Implements the class which is used to represent each of our
  * photographs. All photo properties are stored and accessed via
  * this class.
  * @author M. Asselstine <asselsm@gmail.com>
  */
class Photo : public QObject
{
Q_OBJECT

public:
    /** Default constructor. Creates a Photo object with usable defaults. */
    Photo();

    /** Constructor with url. Construct a Photo object with usable
      * defaults and the passed url for the photograph file path
      * @param url URL of the photograph to associate this object to
      */
    Photo(const KURL& url);

    /** Copy constructor. */
    Photo(const Photo& photo);

    ~Photo();

    /** Rotates the photo by 90 degrees clockwise */
    void rotate();

    /** Overload the equals operator. We currently assume that if the URL for
      * the photograph is the same the photograph itself is the same.
      * @param other The other Photo to compare
      * @return TRUE if the Photo are considered to be equal
      */
    bool operator==(const Photo& other) const;

    /** Use to access the preview image of the photograph.
      * @return Reference to the preview image
      */
    const QPixmap& preview() const {return m_preview;}

    /** Use to set the preview image of the photograph.
      * @param Reference to the preview image
      */
    void preview(const QPixmap& pm);

    /** URL get accessor.
      * @return The url of the file associated with this photograph
      */
    const KURL& URL() const {return m_URL;}

    /** URL set accessor.
      * @param u The URL of the file to associated with this photograph to
      */
    void URL(const KURL& u);

    /** "Private" get accessor.
      * @return True if this photograph is marked as private
      */
    bool exposed() const {return m_exposed;}

    /** "Private" set accessor.
      * @parma b True if this photograph is to be marked as private
      */
    void exposed(bool b);

    /** "Family" get accessor.
      * @return True if this photograph is marked as family only
      */
    bool family() const {return m_family;}

    /** "Family" set accessor.
      * @param b True if this photograph is to be marked as family only
      */
    void family(bool b);

    /** "Friends" get accessor.
      * @return True if this photograph is marked as friends only
      */
    bool friends() const {return m_friends;}

    /** "Friends" set accessor.
      * @param b True if this photograph is to be marked as friends only
      */
    void friends(bool b);

    /** Rotation get accessor.
      * @return The rotation in degrees of this photograph
      */
    int rotation() const {return m_rotation;}

    /** Rotation set accessor.
      * @param r The new rotation in degrees of this photograph
      */
    void rotation(int r);

    /** Description get accessor.
      * @return The description of this photograph
      */
    const QString& description() const {return m_desc;}

    /** Description set accessor.
      * @param d The new description of this photograph
      */
    void description(const QString& d);

    /** Size get accessor.
      * @return The desired upload size of this photograph
      */
    const QString& size() const {return m_size;}

    /** Size set accessor.
      * @param s The new desired upload size of this photograph
      */
    void size(const QString& s);

    /** Title get accessor.
      * @return The title of this photograph
      */
    const QString& title() const {return m_title;}

    /** Title set accessor.
      * @param t The new title of this photograph
      */
    void title(const QString& t);

    /** License get accessor.
      * @return The license for this photograph
      */
    const QString& license() const {return m_license;}

    /** License set accessor.
      * @param l The new license for this photograph
      */
    void license(const QString& l);

    /** Photoset get accessor.
      * @return The photoset which this photograph will belong to
      */
    const QString& photoset() const {return m_photoset;}

    /** Photoset set accessor.
      * @param ps The new photoset this photograph will belong to
      */
    void photoset(const QString& ps);

    /** Tags get accessor.
      * @return The tags associated to this photograph
      */
    const QStringList& tags() const {return m_tags;}

    /** Tags set accessor.
      * @param t The new tags list to associated to this photograph
      */
    void tags(const QStringList& t);

    /** Selected get accessor.
      * @return The selected state of this photograph
      */
    bool selected() const {return m_selected;}

    /** Selected set accessor.
      * @param s The new selected state of this photograph
      */
    void selected(bool s);

    /**
     * Used to check if the photograph is in portrait (tall and skinny) or not.
     * @return True if in portrait, false if in landscape or undetermined.
     */
    bool isPortrait();

signals:
    /** Emit when something has been changed. Objects that are interested
      * in staying up to date with the details of this photo will listen
      * to this signal and as such be notified that something has changed
      */
    void update(Q_UINT16);

private:
    KURL m_URL;                 /// The URL to the photograph file
    bool m_exposed;             /// If true the photo is public, private otherwise
    bool m_family;              /// If true the photo will be accessable by family
    bool m_friends;             /// If true the photo will be accessible by friends
    int m_rotation;             /// The amount of rotation to apply to the photo
    QString m_desc;             /// The photo's description
    QString m_size;             /// The desired upload size
    QString m_title;            /// The photo's title
    QString m_license;          /// The photo's license
    QString m_photoset;         /// The photo's photoset name
    QStringList m_tags;         /// The photo's tags

    bool m_selected;            /// True if the photo is currently selected
    QPixmap m_preview;          /// The preview image for the photo
};

#endif
