/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
#ifndef KFTPNETWORKERRORHANDLER_H
#define KFTPNETWORKERRORHANDLER_H

#include "kftpextensions.h"

#include <qobject.h>
#include <qmap.h>

namespace KFTPNetwork {

/*
 * Error handling:
 * ---------------
 *
 * There are two types of error codes. Handled and unhandled. Handled
 * errors require no interaction from higher-level methods (eg. they
 * are just notifications that something went wrong and that it was
 * aborted). Most of the errors are handled, since imediate action is
 * required.
 *
 * Unhandled error codes are used for "errors", where current operation
 * was interrupted and will not continue unless a higher-level method
 * restarts the operation. The high-level method handling the error
 * notification should imediately execute ErrorHandler::handlingError(err, id),
 * so that error is marked as handled and the signal won't be emitted for
 * the same error again (eg. when retrying).
 *
 * If the higher-level method wishes to ignore that the error has been
 * handled (eg. because the user pressed cancel), it should call the
 * error handler object using ErrorHandler::abandonHandler(id).
 *
 * Errors are dispatched in the following order:
 *
 *  +---------------------+         +-------------+         +-----+
 *  | KFTPNetwork::Socket |-------->| KFTPSession |-------->| GUI |
 *  +---------------------+         +-------------+         +-----+
 *                                        |
 *                                        |
 *                                        ˇ
 *                              +---------------------+
 *                              | KFTPQueue::Transfer |
 *                              +---------------------+
 *
 */

#define FTP_EXCEPTION throw new KFTPNetwork::Exception()
#define FTP_EXCEPTION_TRY try
#define FTP_EXCEPTION_CATCH catch (...) { return -100; }
 
class ErrorHandler;
class Socket;
 
enum ErrorCode {
  /* Low-level (handled) */
  EC_READ_ERR = -1,
  EC_WRITE_ERR = -2,
  EC_TIMEOUT = -3,
  EC_DATA_CONN_ERR = -4,
  EC_FXP_SSL_ERR = -5,
  EC_FD_ERR = -6,

  /* Higher level (handled) */
  EC_PERMISSION_DENIED = -25,
  EC_FILE_NOT_FOUND = -26,
  EC_UNABLE_TO_ENTER_DIR = -27,
  
  EC_SSH_PKEY_PASSWD = -30,

  /* Higher level (unhandled) */
  EC_FILE_EXISTS_DOWNLOAD = -50,
  EC_FILE_EXISTS_UPLOAD = -51,
  EC_FILE_EXISTS_FXP = -52
};

class ErrorData {
public:
    ErrorData()
      : m_filename(QString::null)
    {
    }
    
    ErrorData(const QString &filename, FTPDirectoryItem entry1 = FTPDirectoryItem(), FTPDirectoryItem entry2 = FTPDirectoryItem())
      : m_filename(filename), m_stat1(entry1.m_ftpEntry), m_stat2(entry2.m_ftpEntry)
    {
    }
    
    QString filename() { return m_filename; }
    FTPEntry stat1() { return m_stat1; }
    FTPEntry stat2() { return m_stat2; }
private:
    QString m_filename;
    
    FTPEntry m_stat1;
    FTPEntry m_stat2;
};

inline bool operator==(ErrorData d1, ErrorData d2)
{
  return d1.filename() == d2.filename();
}

class Error {
friend class ErrorHandler;
public:
    Error();
    Error(ErrorHandler *handler, ErrorCode code, ErrorData data);
    
    ErrorHandler *handler() { return m_handler; }
    ErrorData data() { return m_data; }
    ErrorCode code() { return m_code; }
    int returnVal() { return m_returnVal; }
private:
    ErrorHandler *m_handler;
    ErrorCode m_code;
    ErrorData m_data;
    int m_returnVal;
};

class ErrorHandler : public QObject {
Q_OBJECT
public:
    ErrorHandler(Socket *socket);
    ~ErrorHandler();
    
    Error getError(long id) { return m_errorMap[id]; }
    void setHandlerEnabled(bool value) { m_enabled = value; }
    
    long handlingError(Error error);
    void handlerReturnCode(long id, int code);
    void abandonHandler(long id);
    
    /* This should only be called by the socket */
    bool dispatchError(ErrorCode code, ErrorData data = ErrorData());
    int returnCode(ErrorCode code, ErrorData data = ErrorData());
    void errorDone(ErrorCode code, ErrorData data = ErrorData());
private:
    QMap<long, Error> m_errorMap;
    Socket *m_socket;
    long m_lastId;
    bool m_enabled;
private slots:
    void slotSocketDisconnected();
};

class Exception {
public:
    Exception()
    {
      // This does nothing :)
    }
};

}

#endif
