/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "browser/actions.h"
#include "browser/view.h"
#include "browser/listview.h"
#include "browser/propsplugin.h"

#include "kftpquickconnectdialog.h"
#include "kftpbookmarks.h"
#include "kftpqueue.h"
#include "misc.h"
#include "kftpapi.h"
#include "kftpsession.h"
#include "misc/config.h"
#include "verifier.h"

#include <kglobal.h>
#include <kcharsets.h>
#include <kapplication.h>
#include <kmainwindow.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kinputdialog.h>
#include <kpropertiesdialog.h>
#include <kio/job.h>
#include <kshred.h>
#include <klineedit.h>
#include <kfiledialog.h>
#include <kstandarddirs.h>

#include <qdir.h>
#include <qclipboard.h>

using namespace KFTPGrabberBase;

namespace KFTPWidgets {

namespace Browser {

Actions::Actions(QObject *parent, const char *name)
 : QObject(parent, name)
{
  m_dirView = static_cast<View*>(parent);
}

KActionCollection *Actions::actionCollection()
{
  return KFTPAPI::getInstance()->mainWindow()->actionCollection();
}

void Actions::initActions()
{
  // Create all the actions;
  m_goUpAction = KStdAction::up(this, SLOT(slotGoUp()), actionCollection(), "");
  m_goBackAction = KStdAction::back(this, SLOT(slotGoBack()), actionCollection(), "");
  m_goForwardAction = KStdAction::forward(this, SLOT(slotGoForward()), actionCollection(), "");
  m_goHomeAction = KStdAction::home(this, SLOT(slotGoHome()), actionCollection(), "");

  m_reloadAction = KStdAction::redisplay(this, SLOT(slotReload()), actionCollection(), "");
  m_reloadAction->setText(i18n("&Reload"));
  
  m_abortAction = new KAction(i18n("&Abort"), "stop", KShortcut(), this, SLOT(slotAbort()), actionCollection());
  m_toggleTreeViewAction = new KToggleAction(i18n("&Show Tree View"), "view_tree", KShortcut(), this, SLOT(slotShowHideTree()), actionCollection());
  
  m_transferAction = new KAction(i18n("&Transfer"), KShortcut(), this, SLOT(slotTransfer()), actionCollection());
  m_queueTransferAction = new KAction(i18n("&Queue Transfer"), "queue", KShortcut(), this, SLOT(slotQueueTransfer()), actionCollection());
  m_createDirAction = new KAction(i18n("&Create Directory..."), "folder_new", KShortcut(), this, SLOT(slotCreateDir()), actionCollection());
  m_renameAction = new KAction(i18n("&Rename"), KShortcut(), this, SLOT(slotRename()), actionCollection());
  m_deleteAction = new KAction(i18n("&Delete"), "editdelete", KShortcut(), this, SLOT(slotDelete()), actionCollection());
  m_propsAction = new KAction(i18n("&Properties"), KShortcut(), this, SLOT(slotProps()), actionCollection());
  m_setFilterAction = new KAction(i18n("&Set Filter..."), "filter", KShortcut(), this, SLOT(slotSetFilter()), actionCollection());
  m_clearFilterAction = new KAction(i18n("Clear Filter"), KShortcut(), this, SLOT(slotClearFilter()), actionCollection());
  m_shredAction = new KAction(i18n("&Shred"), "editshred", KShortcut(), this, SLOT(slotShred()), actionCollection());
  m_fileEditAction = new KAction(i18n("&Open file"), "fileopen", KShortcut(), this, SLOT(slotFileEdit()), actionCollection());
  m_verifyAction = new KAction(i18n("&Verify..."), "ok", KShortcut(), this, SLOT(slotVerify()), actionCollection());

  m_copyAction = KStdAction::copy(this, SLOT(slotCopy()), actionCollection(), "");
  m_pasteAction = KStdAction::paste(this, SLOT(slotPaste()), actionCollection(), "");
  
  populateEncodings();
  
  m_moreActions = new KActionMenu(i18n("&More Actions"), "configure", this);
  m_rawCmdAction = new KAction(i18n("Send &Raw Command..."), "openterm", KShortcut(), this, SLOT(slotRawCmd()), actionCollection());
  m_exportListingAction = new KAction(i18n("&Export Directory Listing..."), "", KShortcut(), this, SLOT(slotExportListing()), actionCollection());
  
  m_moreActions->insert(m_rawCmdAction);
  m_moreActions->insert(m_changeEncodingAction);
  m_moreActions->insert(m_exportListingAction);
  
  m_moreActions->setStickyMenu(true);
  m_moreActions->setDelayed(false);

  m_siteChangeAction = new KActionMenu(i18n("&Change Site"), "goto", this);
  m_quickConnectAction = new KAction(i18n("&Quick Connect..."), "connect_creating", KShortcut(), this, SLOT(slotQuickConnect()), actionCollection());
  m_connectAction = new KActionMenu(i18n("&Connect To"), this);
  m_disconnectAction = new KAction(i18n("&Disconnect"), "connect_no", KShortcut(), this, SLOT(slotDisconnect()), actionCollection());
  m_localAction = new KAction(i18n("View &Local"), "gohome", KShortcut(), this, SLOT(slotLocal()), actionCollection());
  m_offlineAction = new KToggleAction(i18n("&Offline Mode"), KShortcut(), this, SLOT(slotOffline()), actionCollection());

  m_siteChangeAction->insert(m_quickConnectAction);
  m_siteChangeAction->insert(m_connectAction);
  m_siteChangeAction->insert(m_disconnectAction);
  m_siteChangeAction->popupMenu()->insertSeparator();
  m_siteChangeAction->insert(m_localAction);
  m_siteChangeAction->insert(m_offlineAction);
  m_siteChangeAction->setStickyMenu(true);
  m_siteChangeAction->setDelayed(false);

  // Populate bookmarks
  FTPBookmarkManager->guiPopulateBookmarksMenu(m_connectAction, QDomNode(), false, m_dirView->m_session);
}

void Actions::populateEncodings()
{
  // Charsets
  m_changeEncodingAction = new KActionMenu(i18n("Change Remote &Encoding"), "charset", actionCollection(), "changeremoteencoding");
  m_changeEncodingAction->setDelayed(false);
  
  KPopupMenu *menu = m_changeEncodingAction->popupMenu();
  menu->clear();
  
  QStringList charsets = KGlobal::charsets()->descriptiveEncodingNames();
  int count = 0;
  for (QStringList::iterator i = charsets.begin(); i != charsets.end(); ++i)
    menu->insertItem(*i, this, SLOT(slotCharsetChanged(int)), 0, ++count);
  
  menu->insertSeparator();
  menu->insertItem(i18n("Default"), this, SLOT(slotCharsetReset(int)), 0, ++count);
  menu->setItemChecked(count, true);
  
  m_defaultCharsetOption = count;
  m_curCharsetOption = count;
}

void Actions::updateActions()
{
  // Enable/disable actions
  m_goUpAction->setEnabled(m_dirView->m_list->canMoveTo(ListView::Up));
  m_goBackAction->setEnabled(m_dirView->m_list->canMoveTo(ListView::Back));
  m_goForwardAction->setEnabled(m_dirView->m_list->canMoveTo(ListView::Forward));
  m_goHomeAction->setEnabled(m_dirView->m_list->canMoveTo(ListView::Home));

  m_reloadAction->setEnabled(m_dirView->m_list->canMoveTo(ListView::Reload));
  m_abortAction->setEnabled(m_dirView->m_ftpClient->getClient()->isBusy());
  m_toggleTreeViewAction->setEnabled(true);

  m_localAction->setEnabled(!m_dirView->m_list->m_curURL.isLocalFile());
  m_quickConnectAction->setEnabled(m_dirView->m_list->m_curURL.isLocalFile() && !m_offlineAction->isChecked());
  m_connectAction->setEnabled(true);
  m_disconnectAction->setEnabled(m_dirView->m_ftpClient->getClient()->isConnected() && !m_offlineAction->isChecked());
  m_offlineAction->setEnabled(true);
  
  if (m_dirView->m_list->getCurrentSelection().count() == 1) {
    m_fileEditAction->setEnabled(!m_dirView->m_list->getCurrentSelectionLI().at(0)->m_directory);
    m_verifyAction->setEnabled(m_dirView->m_list->m_curURL.isLocalFile() && m_dirView->m_list->getCurrentSelectionLI().at(0)->m_url.fileName().lower().right(3) == "sfv");
  } else {
    m_fileEditAction->setEnabled(false);
    m_verifyAction->setEnabled(false);
  }

  // We can queue a transfer if we canTransfer() or we are in offline mode and have a remote URL
  // open or "the other" is in offline mode and has a remote URL open
  if (m_dirView->m_list->canTransfer() || (m_dirView->m_list->m_offlineMode && !m_dirView->m_list->m_curURL.isLocalFile()) ||
      (m_dirView->m_list->m_companion && (m_dirView->m_list->m_companion->m_offlineMode &&
      !m_dirView->m_list->m_companion->m_curURL.isLocalFile())))
  {
    m_queueTransferAction->setEnabled(true);
  } else {
    m_queueTransferAction->setEnabled(false);
  }

  m_transferAction->setEnabled(m_dirView->m_list->canTransfer());
  m_createDirAction->setEnabled(m_dirView->m_list->canLocalActions());
  m_renameAction->setEnabled(m_dirView->m_list->canLocalActions());
  m_deleteAction->setEnabled(m_dirView->m_list->canLocalActions());
  m_propsAction->setEnabled(!m_dirView->m_list->m_offlineMode);
  m_shredAction->setEnabled(m_dirView->m_list->m_curURL.isLocalFile());

  m_changeEncodingAction->setEnabled(!m_dirView->m_list->m_curURL.isLocalFile());
  m_exportListingAction->setEnabled(true);
  m_rawCmdAction->setEnabled(!m_dirView->m_list->m_curURL.isLocalFile() && 
                             m_dirView->m_ftpClient->getClient()->getFeatures() & SF_RAW_COMMAND);
  
  m_setFilterAction->setEnabled(true);
  m_clearFilterAction->setEnabled(!m_dirView->m_list->m_filter.isNull());
}

void Actions::slotQuickConnect()
{
  // Create/get the new dialog
  KFTPQuickConnectDialog* qcDialog = new KFTPQuickConnectDialog(m_dirView, "qc dialog");

  if (qcDialog->exec() == KDialogBase::Accepted) {
    // Get the url and connect
    if (m_dirView->m_ftpClient->getClient()->isConnected()) {
      if (KFTPCore::Config::confirmDisconnects() && KMessageBox::warningYesNo(0, i18n("Do you want to drop current connection?")) == KMessageBox::No)
        return;

      m_dirView->m_ftpClient->KFTPClientThr::disconnect();
    }

    m_dirView->m_list->setHomeURL(KURL());
    m_dirView->m_session->setSite(0);
    
    qcDialog->setupClient(m_dirView->m_ftpClient);
    m_dirView->m_ftpClient->connectToHost(qcDialog->getServerURL());
  }

  delete qcDialog;
}

void Actions::slotDisconnect()
{
  if (m_dirView->m_ftpClient->getClient()->isConnected()) {
    if (KFTPCore::Config::confirmDisconnects() && KMessageBox::warningYesNo(0, i18n("Do you want to drop current connection?")) == KMessageBox::No)
      return;

    m_dirView->m_ftpClient->KFTPClientThr::disconnect();
  }
}

void Actions::slotLocal()
{
  if (m_dirView->m_ftpClient->getClient()->isConnected()) {
    if (KFTPCore::Config::confirmDisconnects() && KMessageBox::warningYesNo(0, i18n("Do you want to drop current connection?")) == KMessageBox::No)
      return;

    m_dirView->m_ftpClient->KFTPClientThr::disconnect();
  }

  m_dirView->m_list->openURL(KURL(KFTPCore::Config::defLocalDir()));

  if (m_offlineAction->isChecked()) {
    m_offlineAction->setChecked(false);
    m_dirView->m_list->m_offlineMode = false;
    updateActions();
    
    // Trick the session into thinking that the "remote client" has disconnected
    FTPSessionManager->find(m_dirView)->slotClientDisconnected();
  }
}

void Actions::slotOffline()
{
  if (m_dirView->m_ftpClient->getClient()->isConnected()) {
    if (KFTPCore::Config::confirmDisconnects() && KMessageBox::warningYesNo(0, i18n("Do you want to drop current connection?")) == KMessageBox::No) {
      m_offlineAction->setChecked(false);
      m_dirView->m_list->m_offlineMode = false;
      return;
    }

    m_dirView->m_ftpClient->KFTPClientThr::disconnect();
    m_dirView->m_list->openURL(KURL(KFTPCore::Config::defLocalDir()));
  }

  m_dirView->m_list->m_offlineMode = m_offlineAction->isChecked();
  updateActions();

  if (!m_offlineAction->isChecked()) {
    // Go back to local dir
    m_dirView->m_list->openURL(KURL(KFTPCore::Config::defLocalDir()));
    
    // Trick the session into thinking that the "remote client" has disconnected
    FTPSessionManager->find(m_dirView)->slotClientDisconnected();
  }
}

void Actions::slotGoUp()
{
  m_dirView->m_list->moveTo(ListView::Up);
}

void Actions::slotGoBack()
{
  m_dirView->m_list->moveTo(ListView::Back);
}

void Actions::slotGoForward()
{
  m_dirView->m_list->moveTo(ListView::Forward);
}

void Actions::slotReload()
{
  m_dirView->m_list->moveTo(ListView::Reload);
}

void Actions::slotGoHome()
{
  m_dirView->m_list->moveTo(ListView::Home);
}

void Actions::slotTransfer()
{
    // Queue a transfer
  ListItems selection = m_dirView->m_list->getCurrentSelectionLI();

  // Determine transfer type
  bool c_local = m_dirView->m_list->m_companion->m_curURL.isLocalFile();
  bool w_local = m_dirView->m_list->m_curURL.isLocalFile();
  KFTPQueue::TransferType transType = KFTPQueue::Download;

  if (w_local && !c_local) /* Local --> Remote (UPLOAD) */
    transType = KFTPQueue::Upload;

  if (!w_local && c_local) /* Remote --> Local (DOWNLOAD) */
    transType = KFTPQueue::Download;

  if (!w_local && !c_local) /* Remote --> Remote (FXP TRANSFER) */
    transType = KFTPQueue::FXP;

  KFTPQueue::Transfer *m_transfer = 0L;

  for (unsigned int i = 0; i < selection.count(); i++) {
    ListViewItem *item = selection.at(i);
    KURL destURL = m_dirView->m_list->m_companion->m_curURL;
    destURL.addPath(item->m_dirObject.name());

    m_transfer = KFTPQueue::Manager::self()->spawnTransfer(
      item->m_url,
      destURL,
      item->m_dirObject.size(),
      item->m_directory,
      transType,
      selection.count() == 1,
      true,
      0L,
      true
    );
  }

  // Execute transfer
  if (m_transfer)
    static_cast<KFTPQueue::Site*>(m_transfer->parentObject())->delayedExecute();
}

void Actions::slotQueueTransfer()
{
  // Queue a transfer
  ListItems selection = m_dirView->m_list->getCurrentSelectionLI();

  // Determine transfer type
  bool c_local = m_dirView->m_list->m_companion->m_curURL.isLocalFile();
  bool w_local = m_dirView->m_list->m_curURL.isLocalFile();
  KFTPQueue::TransferType transType = KFTPQueue::Download;

  if (w_local && !c_local) /* Local --> Remote (UPLOAD) */
    transType = KFTPQueue::Upload;

  if (!w_local && c_local) /* Remote --> Local (DOWNLOAD) */
    transType = KFTPQueue::Download;

  if (!w_local && !c_local) /* Remote --> Remote (FXP TRANSFER) */
    transType = KFTPQueue::FXP;

  for (unsigned int i = 0; i < selection.count(); i++) {
    ListViewItem *item = selection.at(i);
    KURL destURL = m_dirView->m_list->m_companion->m_curURL;
    destURL.addPath(item->m_dirObject.name());

    KFTPQueue::Manager::self()->spawnTransfer(
      item->m_url,
      destURL,
      item->m_dirObject.size(),
      item->m_directory,
      transType,
      selection.count() == 1,
      true,
      0L,
      i != 0
    );
  }
}

void Actions::slotCreateDir()
{
  // Create new directory
  bool ok;
  QString newDirName = KInputDialog::getText(i18n("Create Directory"), i18n("Directory name:"), "", &ok);
  
  if (ok) {
    if (m_dirView->m_list->m_curURL.isLocalFile()) {
      KIO::mkdir(KURL(m_dirView->m_list->m_curURL.path(1) + newDirName));
    } else {
      KURL newDir = m_dirView->m_list->m_curURL;
      newDir.addPath(newDirName);
      m_dirView->m_ftpClient->mkdir(newDir);
    }
  }
}

void Actions::slotShred()
{
  // Shred the file
  if (KMessageBox::warningContinueCancel(0, i18n("Are you sure you want to SHRED this file?"), i18n("Shred File"),KGuiItem(i18n("&Shred","editshred"))) == KMessageBox::Cancel)
    return;
  
  KURL::List selection = m_dirView->m_list->getCurrentSelection();
  KShred::shred((*selection.at(0)).path());
  
  slotReload();
}

void Actions::slotFileEdit()
{
  // Get Selection
  ListItems selection = m_dirView->m_list->getCurrentSelectionLI();

  bool isLocal = m_dirView->m_list->m_curURL.isLocalFile();
  ListViewItem *item = selection.at(0);
  
  if (!item->m_directory) {
    if (isLocal) {
      QString mimeType = KMimeType::findByURL(item->m_url, 0, true, true)->name();
      KService::Ptr offer = KServiceTypeProfile::preferredService(mimeType, "Application");
      
      if (offer)
        KRun::run(offer->exec(), item->m_url, offer->name(), offer->icon());
    } else {
      // Create a new transfer to download the file and open it
      KFTPQueue::TransferFile *transfer = new KFTPQueue::TransferFile(KFTPQueue::Manager::self());
      transfer->setSourceUrl(item->m_url);
      transfer->setDestUrl(KURL(KGlobal::dirs()->saveLocation("tmp") + QString("%1-%2").arg(KApplication::randomString(7)).arg(item->m_url.fileName())));
      transfer->addSize(item->m_dirObject.size());
      transfer->setTransferType(KFTPQueue::Download);
      transfer->setOpenAfterTransfer(true);
      KFTPQueue::Manager::self()->insertTransfer(transfer);
      
      // Execute the transfer
      transfer->delayedExecute();
    }
  }
}

void Actions::slotRename()
{
  // Rename the first file in the current selection
  m_dirView->m_list->rename(m_dirView->m_list->selectedItems().at(0), 0);
  
  // Enhanced rename: Don't highlight the file extension. (from Konqueror)
  KLineEdit *le = m_dirView->m_list->renameLineEdit();
  
  if (le) {
    const QString txt = le->text();
    QString pattern;
    KMimeType::diagnoseFileName(txt, pattern);
    if (!pattern.isEmpty() && pattern.at(0) == '*' && pattern.find('*',1) == -1)
      le->setSelection(0, txt.length()-pattern.stripWhiteSpace().length()+1);
    else {
      int lastDot = txt.findRev('.');
      if (lastDot > 0)
        le->setSelection(0, lastDot);
    }
  }
}

void Actions::slotDelete()
{
  // Delete a file or directory
  KURL::List selection = m_dirView->m_list->getCurrentSelection();

  if (KMessageBox::warningContinueCancel(0, i18n("Are you sure you want to delete this file(s)?"), i18n("Delete File"),KGuiItem(i18n("&Delete"),"editdelete")) == KMessageBox::Cancel)
    return;

  // Go trough all files and delete them
  if (selection[0].isLocalFile()) {
    KIO::del(selection);
  } else {
    KURL::List::Iterator end( selection.end() );
    for (KURL::List::Iterator i( selection.begin() ); i != end; ++i) {
      if (!(*i).isLocalFile()) {
        m_dirView->m_ftpClient->remove(KURL((*i).url()));
      }
    }
  }
}

void Actions::slotCopy()
{
  QClipboard *cb = QApplication::clipboard();
  
  cb->setData(m_dirView->m_list->dragObject(), QClipboard::Clipboard);
}

void Actions::slotPaste()
{
  // Decode the data and try to init transfer
  KIO::MetaData p_meta;
  KURL::List p_urls;
  
  if (KURLDrag::decode(QApplication::clipboard()->data(), p_urls, p_meta)) {
    // Add destination url and call the QueueManager
    p_meta.insert("DestURL", m_dirView->m_list->m_curURL.url());
    KURLDrag *drag = new KURLDrag(p_urls, p_meta, m_dirView->m_list, name());
    KFTPQueue::Manager::self()->insertTransfer(drag);
  }
}

void Actions::slotProps()
{
  // Show file properties
  KURL::List selection = m_dirView->m_list->getCurrentSelection();
  KURL itemURL = *selection.at(0);

  if (selection.count() == 0) {
    itemURL = m_dirView->m_list->m_curURL;
    
    if (!itemURL.isLocalFile())
      return;
  }

  // Show the dialog
  if (itemURL.isLocalFile()) {
    KPropertiesDialog *propsDialog = new KPropertiesDialog(new KFileItem(itemURL, 0, 0));
    propsDialog->exec();
  } else {
    ListItems selItems = m_dirView->m_list->getCurrentSelectionLI();
    KFTPSession *session = FTPSessionManager->find(m_dirView);
    FTPDirectoryItem i = selItems.at(0)->m_dirObject;
    
    KPropertiesDialog *propsDialog = new KPropertiesDialog(itemURL.fileName());
    propsDialog->insertPlugin(new PropsPlugin(propsDialog, i, itemURL));
    propsDialog->insertPlugin(new PermissionsPropsPlugin(propsDialog, i, session, itemURL));
    propsDialog->exec();
  }
}

void Actions::slotSetFilter()
{
  bool ok;
  QString filterStr = KInputDialog::getText(i18n("Filter"), i18n("Enter filter:"), m_dirView->m_list->m_filter, &ok);

  if (ok) {
    m_dirView->m_list->setFilter(filterStr);
  }
}

void Actions::slotClearFilter()
{
  m_dirView->m_list->setFilter(QString::null);
}

void Actions::slotAbort()
{
  KFTPSession *session = FTPSessionManager->find(m_dirView);

  // Abort the session
  if (session) {
    session->abort();
  }
}

void Actions::slotRawCmd()
{
  bool ok;
  QString rawCmd = KInputDialog::getText(i18n("Send Raw Command"), i18n("Command:"), "", &ok);

  if (ok) {
    m_dirView->m_ftpClient->rawCommand(rawCmd);
  }
}

void Actions::slotShowHideTree()
{
  m_dirView->setTreeVisible(m_toggleTreeViewAction->isChecked());
}

void Actions::slotCharsetChanged(int id)
{
  if (!m_changeEncodingAction->popupMenu()->isItemChecked(id)) {
    QStringList charsets = KGlobal::charsets()->descriptiveEncodingNames();
    QString charset = KGlobal::charsets()->encodingForName(charsets[id - 1]);
    
    // Set the current socket's charset
    m_dirView->m_ftpClient->getClient()->changeEncoding(charset);
    
    // Update checked items
    m_changeEncodingAction->popupMenu()->setItemChecked(id, true);
    m_changeEncodingAction->popupMenu()->setItemChecked(m_curCharsetOption, false);
    m_curCharsetOption = id;
  }
}

void Actions::slotCharsetReset(int id)
{
  // Revert to default charset if possible
  KFTPBookmarks::Site *site = m_dirView->m_session->getSite();
  
  if (site) {
    // Set the current socket's charset
    m_dirView->m_ftpClient->getClient()->changeEncoding(site->getProperty("encoding"));
    
    // Update checked items
    m_changeEncodingAction->popupMenu()->setItemChecked(id, true);
    m_changeEncodingAction->popupMenu()->setItemChecked(m_curCharsetOption, false);
    m_curCharsetOption = id;
  }
}

void Actions::slotExportListing()
{
  QString savePath = KFileDialog::getSaveFileName(QString::null, "*.txt|Directory Dump", 0, i18n("Export Directory Listing"));
  
  if (!savePath.isEmpty()) {
    QFile file(savePath);
    
    if (!file.open(IO_WriteOnly))
      return;
      
    QTextStream stream(&file);
      
    QListViewItem *item = m_dirView->getListView()->firstChild();
    while (item) {
      ListViewItem *i = static_cast<ListViewItem*>(item);
      
      stream << i->m_dirObject.permissions() << "\t";
      stream << i->m_dirObject.owner() << "\t" << i->m_dirObject.group() << "\t";
      stream << i->m_dirObject.dateAsString() << "\t";
      stream << i->m_dirObject.name() << "\t";
      stream << "\n";
      
      item = item->nextSibling();
    }
      
    file.flush();
    file.close();
  }
}

void Actions::slotVerify()
{
  KFTPWidgets::Verifier *verifier = new KFTPWidgets::Verifier();
  verifier->setFile(m_dirView->m_list->getCurrentSelection()[0].path());
  verifier->exec();
  
  delete verifier;
}

}

}

#include "actions.moc"
