/***************************************************************************
 *   Copyright (C) 2007, 2008 by Marcel Hasler                             *
 *   mahasler@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "configuration.h"

#include <kglobal.h>

#include <map>
#include <iostream>
using namespace std;

ConfigurationHandler::ConfigurationHandler()
{
	m_config = KGlobal::config();
	m_wallet = new Wallet();

	// Read general options
	m_config->setGroup("General");
	m_stringOptions["Username"] = m_config->readEntry("Username", "username@gmail.com");
	m_boolOptions["SavePassword"] = m_config->readBoolEntry("SavePassword", false);
	m_stringOptions["PasswordMethod"] = m_config->readEntry("PasswordMethod", "Wallet");
	m_numOptions["CheckInterval"] = m_config->readNumEntry("CheckInterval", 60);
	m_stringOptions["BrowserPath"] = m_config->readEntry("BrowserPath", "/usr/bin/firefox");

	if (m_boolOptions["SavePassword"] && m_stringOptions["PasswordMethod"] == "Wallet")
		m_stringOptions["Password"] = m_wallet->readPassword();
	else
		m_stringOptions["Password"] = m_config->readEntry("Password", "");

	// Read appearance options
	m_config->setGroup("Appearance");
	m_numOptions["HorizontalOffset"] = m_config->readNumEntry("HorizontalOffset", 0);
	m_numOptions["VerticalOffset"] = m_config->readNumEntry("VerticalOffset", 0);
	m_numOptions["PopupTimeout"] = m_config->readNumEntry("PopupTimeout", 5);
	m_boolOptions["PopupAnimated"] = m_config->readBoolEntry("PopupAnimated", true);
	m_numOptions["AnimationDelay"] = m_config->readNumEntry("AnimationDelay", 5);
	m_stringOptions["PopupDirection"] = m_config->readEntry("PopupDirection", "Up");
	m_boolOptions["NumberOnSystray"] = m_config->readBoolEntry("NumberOnSystray", false);

	// Read notification options
	m_config->setGroup("Notification");
	m_boolOptions["ShowPopup"] = m_config->readBoolEntry("ShowPopup", true);
	m_boolOptions["SoundEnabled"] = m_config->readBoolEntry("SoundEnabled", false);
	m_stringOptions["SoundFile"] = m_config->readEntry("SoundFile", "");
	m_boolOptions["LedEnabled"] = m_config->readBoolEntry("LedEnabled", false);
	m_stringOptions["LedOnCommand"] = m_config->readEntry("LedOnCommand", "");
	m_stringOptions["LedOffCommand"] = m_config->readEntry("LedOffCommand", "");
	m_boolOptions["SilentNotification"] = m_config->readBoolEntry("SilentNotification", false);

	// Read advanced option
	m_config->setGroup("Advanced");
	m_boolOptions["LockOnPrimaryScreen"] = m_config->readBoolEntry("LockOnPrimaryScreen", true);
	m_boolOptions["Autostart"] = m_config->readBoolEntry("Autostart", true);

	// Read standard "Notification Messages" options
	m_config->setGroup("Notification Messages");
	m_stringOptions["QuitDontAskAgain"] = m_config->readEntry("QuitDontAskAgain", "");
}


ConfigurationHandler::~ConfigurationHandler()
{
	if (m_wallet)
		delete m_wallet;

	if (m_config)
		delete m_config;
}


QString ConfigurationHandler::readEntry(const QString &option) const
{
	map<QString, QString>::const_iterator it = m_stringOptions.find(option);
	if (it == m_stringOptions.end())
	{
		cout << "Invalid option in ConfigurationHandler::readStrOption() (\"" << option << "\")!" << endl;
		return QString("");
	}
	else
		return it->second;
}


int ConfigurationHandler::readNumEntry(const QString &option) const
{
	map<QString, int>::const_iterator it = m_numOptions.find(option);
	if (it == m_numOptions.end())
	{
		cout << "Invalid option in ConfigurationHandler::readNumOption() (\"" << option << "\")!" << endl;
		return 0;
	}
	else
		return it->second;
}


bool ConfigurationHandler::readBoolEntry(const QString &option) const
{
	map<QString, bool>::const_iterator it = m_boolOptions.find(option);
	if (it == m_boolOptions.end())
	{
		cout << "Invalid option in ConfigurationHandler::readBoolOption() (\"" << option << "\")!" << endl;
		return false;
	}
	else
		return it->second;
}


void ConfigurationHandler::writeEntry(const QString &option, QString value)
{
	if (m_stringOptions.find(option) == m_stringOptions.end())
		cout << "Invalid string option in ConfigurationHandler::writeEntry() (\"" << option << "\")!" << endl;
	else
		m_stringOptions[option] = value;
}


void ConfigurationHandler::writeEntry(const QString &option, int value)
{
	if (m_numOptions.find(option) == m_numOptions.end())
		cout << "Invalid numeric option in ConfigurationHandler::writeEntry() (\"" << option << "\")!" << endl;
	else
		m_numOptions[option] = value;
}


void ConfigurationHandler::writeEntry(const QString &option, bool value)
{
	if (m_boolOptions.find(option) == m_boolOptions.end())
		cout << "Invalid boolean option in ConfigurationHandler::writeEntry() (\"" << option << "\")!" << endl;
	else
		m_boolOptions[option] = value;
}


void ConfigurationHandler::writeToDisk()
{
	// Write general options
	m_config->setGroup("General");
	m_config->writeEntry("Username", m_stringOptions["Username"]);
	m_config->writeEntry("SavePassword", m_boolOptions["SavePassword"]);
	m_config->writeEntry("PasswordMethod", m_stringOptions["PasswordMethod"]);
	m_config->writeEntry("CheckInterval", m_numOptions["CheckInterval"]);
	m_config->writeEntry("BrowserPath", m_stringOptions["BrowserPath"]);

	if (m_boolOptions["SavePassword"] && m_stringOptions["PasswordMethod"] == "Wallet")
	{
		m_config->writeEntry("Password", "");
		m_wallet->writePassword(m_stringOptions["Password"]);
	}
	else if (m_boolOptions["SavePassword"] && m_stringOptions["PasswordMethod"] == "File")
	{
		m_config->writeEntry("Password", m_stringOptions["Password"]);
		m_wallet->deletePassword();
	}
	else
	{
		m_config->writeEntry("Password", "");
		m_wallet->deletePassword();
	}

	// Write appearance options
	m_config->setGroup("Appearance");
	m_config->writeEntry("HorizontalOffset", m_numOptions["HorizontalOffset"]);
	m_config->writeEntry("VerticalOffset", m_numOptions["VerticalOffset"]);
	m_config->writeEntry("PopupTimeout", m_numOptions["PopupTimeout"]);
	m_config->writeEntry("PopupAnimated", m_boolOptions["PopupAnimated"]);
	m_config->writeEntry("AnimationDelay", m_numOptions["AnimationDelay"]);
	m_config->writeEntry("PopupDirection", m_stringOptions["PopupDirection"]);
	m_config->writeEntry("NumberOnSystray", m_boolOptions["NumberOnSystray"]);

	// Write notification options
	m_config->setGroup("Notification");
	m_config->writeEntry("ShowPopup", m_boolOptions["ShowPopup"]);
	m_config->writeEntry("SoundEnabled", m_boolOptions["SoundEnabled"]);
	m_config->writeEntry("SoundFile", m_stringOptions["SoundFile"]);
	m_config->writeEntry("LedEnabled", m_boolOptions["LedEnabled"]);
	m_config->writeEntry("LedOnCommand", m_stringOptions["LedOnCommand"]);
	m_config->writeEntry("LedOffCommand", m_stringOptions["LedOffCommand"]);
	m_config->writeEntry("SilentNotification", m_boolOptions["SilentNotification"]);

	// Write advanced option
	m_config->setGroup("Advanced");
	m_config->writeEntry("LockOnPrimaryScreen", m_boolOptions["LockOnPrimaryScreen"]);
	m_config->writeEntry("Autostart", m_boolOptions["Autostart"]);

	// Write standard "Notification Messages" options
	m_config->setGroup("Notification Messages");
	m_config->writeEntry("QuitDontAskAgain", m_stringOptions["QuitDontAskAgain"]);

	// Write to hard disk
	m_config->sync();
}

// Global public friend function
ConfigurationHandler& Configuration()
{
	static ConfigurationHandler config;
	return config;
};
