/***************************************************************************
 *   Copyright (C) 2007-2009 by Marcel Hasler                              *
 *   mahasler@gmail.com                                                    *
 *                                                                         *
 *   This file is part of KGmailNotifier.                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation, either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program. If not, see <http://www.gnu.org/licenses/>.  *
 ***************************************************************************/

#include "configuration.h"
#include "popupwindow.h"

#include <QApplication>
#include <QDesktopWidget>
#include <QTextStream>

#include <KLocale>
#include <KStandardDirs>

static QTextStream cout(stdout, QIODevice::WriteOnly);

// ------------------------------------------------------------------------------------------------------------------- //
//    Construction & Destruction                                                                                       //
// ------------------------------------------------------------------------------------------------------------------- //

PopupWindow::PopupWindow()
    : QWidget(0, Qt::Window | Qt::WindowStaysOnTopHint | Qt::FramelessWindowHint | Qt::X11BypassWindowManagerHint),
      m_width(200),
      m_height(150)
{
    m_font = new QFont();
    m_font->setPixelSize(11);

    // Add the background image
    m_image = new QLabel(this);
    m_image->setPixmap(KStandardDirs::locate("appdata", "pics/gmailLabel.png"));
    m_image->setGeometry(1, 1, m_width, m_height);
    m_image->show();

    // Define new mail label
    m_newMailLabel = new KUrlLabel(m_image);
    m_newMailLabel->setGeometry(15, 50, 170, 20);
    m_newMailLabel->setAlignment(Qt::AlignCenter);
    m_newMailLabel->setWordWrap(false);
    m_newMailLabel->setFont(*m_font);
    m_newMailLabel->setSelectedColor(QColor("blue"));
    m_newMailLabel->setText(i18n("No new mail"));
    m_newMailLabel->show();
    connect(m_newMailLabel, SIGNAL(leftClickedUrl()), this, SLOT(slotProcessUrlClicked()));

    // Define sender label
    m_senderLabel = new QLabel(m_image);
    m_senderLabel->setGeometry(15, 70, 170, 35);
    m_senderLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    m_senderLabel->setWordWrap(false);
    m_senderLabel->setFont(*m_font);
    m_senderLabel->setText("");
    m_senderLabel->hide();

    // Define message title label
    m_subjectLabel = new QLabel(m_image);
    m_subjectLabel->setGeometry(15, 105, 170, 35);
    m_subjectLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    m_subjectLabel->setWordWrap(false);
    m_subjectLabel->setFont(*m_font);
    m_subjectLabel->setText("");
    m_subjectLabel->hide();

    // Define connection failed label
    m_connectionFailedLabel = new QLabel(m_image);
    m_connectionFailedLabel->setGeometry(10, 50, 180, 20);
    m_connectionFailedLabel->setAlignment(Qt::AlignCenter);
    m_connectionFailedLabel->setWordWrap(false);
    m_connectionFailedLabel->setFont(*m_font);
    m_connectionFailedLabel->setText(i18n("Connection failed!"));
    m_connectionFailedLabel->hide();

    // Set up popup timer
    m_closeTimer = new QTimer();
    m_closeTimer->setSingleShot(true);
    connect(m_closeTimer, SIGNAL(timeout()), this, SLOT(slotTimeout()));

    // Set up animation timer
    m_animationTimer = new QTimer();
    m_animationTimer->setSingleShot(false);
    connect(m_animationTimer, SIGNAL(timeout()), this, SLOT(slotProcessAnimation()));

    // Set states
    m_isAnimated = false;
    m_isUp = false;
    m_isOpening = false;

    this->updateConfiguration();
}

// ------------------------------------------------------------------------------------------------------------------- //

PopupWindow::~PopupWindow()
{
    if (m_font)
        delete m_font;

    if (m_image)
        delete m_image;

    if (m_newMailLabel)
        delete m_newMailLabel;

    if (m_senderLabel)
        delete m_senderLabel;

    if (m_subjectLabel)
        delete m_subjectLabel;

    if (m_connectionFailedLabel)
        delete m_connectionFailedLabel;

    if (m_closeTimer)
        delete m_closeTimer;

    if (m_animationTimer)
        delete m_animationTimer;
}


// ------------------------------------------------------------------------------------------------------------------- //
//    Public methods                                                                                                   //
// ------------------------------------------------------------------------------------------------------------------- //

void PopupWindow::toggle()
{
    if (m_isUp || m_isOpening)
        this->slotClose();
    else
        this->slotOpen();
}

// ------------------------------------------------------------------------------------------------------------------- //

void PopupWindow::updateConfiguration()
{
    m_closeTimeout = Configuration::config()->popupTimeout();
    m_animationDelay = 100 / Configuration::config()->animationSpeed();
    m_popupDirection = (PopupDirection)Configuration::config()->popupDirectionUp();
    m_isAnimated = Configuration::config()->popupAnimated();
    m_hOffset = Configuration::config()->horizontalOffset();
    m_vOffset = Configuration::config()->verticalOffset();
    m_lockOnPrimaryScreen = Configuration::config()->lockOnPrimaryScreen();

    this->updatePosition();
}

// ------------------------------------------------------------------------------------------------------------------- //

void PopupWindow::updatePosition()
{
    QDesktopWidget *d = QApplication::desktop();

    if (m_lockOnPrimaryScreen)
    {
        QRect rect = d->screenGeometry(d->primaryScreen());
        m_xPos = rect.width() - m_hOffset - m_width;
        m_yPos = rect.height() - m_vOffset - m_height;
    }
    else
    {
        m_xPos = d->width() - m_hOffset - m_width;
        m_yPos = d->height() - m_vOffset - m_height;
    }

    this->setGeometry(m_xPos, m_yPos, m_width, m_height);
}

// ------------------------------------------------------------------------------------------------------------------- //

void PopupWindow::setMessageCount(unsigned int number)
{
    if (number == 0)
    {
        m_connectionFailedLabel->hide();
        m_senderLabel->hide();
        m_subjectLabel->hide();

        m_newMailLabel->setText(i18n("No new mail"));
        m_newMailLabel->setAlignment(Qt::AlignCenter);
        m_newMailLabel->show();
    }
    else
    {
        m_connectionFailedLabel->hide();

        m_newMailLabel->setText(i18n("You have new mail") + QString(" (%1)").arg(number));
        m_newMailLabel->setAlignment(Qt::AlignLeft);
        m_newMailLabel->show();
    }
}

// ------------------------------------------------------------------------------------------------------------------- //

void PopupWindow::setSender(const QString &sender)
{
    QString s = sender;

    if (s == "")
        s = i18n("(No sender)");
    else if (s.length() > 19)
    {
        s.truncate(16);
        s += QString(" ...");
    }

    m_senderLabel->setText("<b>" + i18n("From:") + "</b><br>" + s);

    m_senderLabel->show();
}

// ------------------------------------------------------------------------------------------------------------------- //

void PopupWindow::setSubject(const QString &subject)
{
    QString s = subject;

    if (s == "")
        s = i18n("(No subject)");
    else if (s.length() > 28)
    {
        s.truncate(25);
        s += QString(" ...");
    }

    m_subjectLabel->setText("<b>" + i18n("Subject:") + "</b><br>" + s);

    m_subjectLabel->show();
}

// ------------------------------------------------------------------------------------------------------------------- //

void PopupWindow::setConnectionFailed()
{
    m_newMailLabel->hide();
    m_senderLabel->hide();
    m_subjectLabel->hide();

    m_connectionFailedLabel->show();
}


// ------------------------------------------------------------------------------------------------------------------- //
//    Private slots                                                                                                    //
// ------------------------------------------------------------------------------------------------------------------- //

void PopupWindow::slotOpen()
{
    this->updatePosition();

    if (m_isAnimated)
    {
        m_isOpening = true;
        if (m_popupDirection == PopupDirectionUp)
            m_currentImagePos = m_height;
        else
            m_currentImagePos = -m_height;
        m_currentHeight = 0;

        m_image->move(0, m_currentImagePos);
        m_animationTimer->start(m_animationDelay);
    }
    else
    {
        m_image->move(0, 0);
        this->setMask(QRegion(0, 0, m_width, m_height));
        this->show();
        m_closeTimer->start(m_closeTimeout*1000);
        m_isUp = true;
    }
}

// ------------------------------------------------------------------------------------------------------------------- //

void PopupWindow::slotTimeout()
{
    // Keep the popup window open if it has mouse focus
    if (this->underMouse())
    {
        this->m_closeTimer->stop();
        this->m_closeTimer->start(1000);
    }
    else
        this->slotClose();
}

// ------------------------------------------------------------------------------------------------------------------- //

void PopupWindow::slotClose()
{
    m_closeTimer->stop();

    m_isUp = false;

    if (m_isAnimated)
    {
        m_isOpening = false;
        m_animationTimer->start(m_animationDelay);
    }
    else
        this->hide();
}

// ------------------------------------------------------------------------------------------------------------------- //

void PopupWindow::slotProcessAnimation()
{
    if (m_isOpening)
    {
        if (m_popupDirection == PopupDirectionUp)
            m_currentImagePos -= 1;
        else
            m_currentImagePos += 1;
        m_currentHeight += 1;
        
        if (m_currentHeight >= m_height)
        {
            m_animationTimer->stop();
            m_closeTimer->start(m_closeTimeout*1000);
            m_isUp = true;
        }
    }

    else
    {
        if (m_popupDirection == PopupDirectionUp)
            m_currentImagePos += 1;
        else
            m_currentImagePos -= 1;
        m_currentHeight -= 1;

        if (m_currentHeight <= 0)
        {
            m_animationTimer->stop();
            this->hide();
            return;
        }
    }

    m_image->move(0, m_currentImagePos);
    this->setMask(QRegion(0, m_currentImagePos, m_width, m_currentHeight));
    this->show();
}

// ------------------------------------------------------------------------------------------------------------------- //

void PopupWindow::slotProcessUrlClicked()
{
    emit urlClicked();
    this->slotClose();
}

// ------------------------------------------------------------------------------------------------------------------- //

#include "popupwindow.moc"
