/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

//Own
#include "password.h"

//Qt
#include <qprocess.h>

//KDE
#include <kmessagebox.h>

//KGRUBEditor
#include "kgrubeditor.h"
#include "core/filetransactions.h"

PasswordAssistant::PasswordAssistant( GRUB::ComplexCommand::Password *password, QWidget *parent ) : KAssistantDialog( parent )
{
//SETUP UI
	setCaption( i18n( "Password Assistant" ) );
	setWindowIcon( KIcon( "dialog-password" ) );
	setAttribute( Qt::WA_DeleteOnClose );
	setInitialSize( QSize( 650, 450 ) );
	showButton( KDialog::Help, false );

	QWidget *intro = new QWidget;
	ui_intro.setupUi( intro );
	addPage( intro, i18n( "Introduction" ) );
	QWidget *step1 = new QWidget;
	ui_step1.setupUi( step1 );
	passwordPage = addPage( step1, i18n( "Step 1/2 - Password and MD5 Encryption" ) );
	QWidget *step2 = new QWidget;
	ui_step2.setupUi( step2 );
	addPage( step2, i18n( "Step 2/2 - New Configuration File" ) );
	QWidget *summary = new QWidget;
	ui_summary.setupUi( summary );
	summaryPage = addPage( summary, i18n( "Summary" ) );
//PROPERTY SETUP
	m_password = password;
//SHOW PASSWORD (IF ANY - ONLY USED IN EDIT MODE)
	ui_step1.klineedit_password->setText( password->password() );
	ui_step1.checkBox_encrypt->setChecked( password->md5crypted() );
	ui_step2.kurlrequester_config->setPath( password->configFile() );
//MISC UI SETUP
	setValid( passwordPage, false );
	setPasswordPageValid();
//SETUP CONNECTIONS
	connect( ui_step1.klineedit_password, SIGNAL( cursorPositionChanged( int, int ) ), SLOT( setPasswordPageValid() ) );
	connect( ui_step1.kpushbutton_encrypt, SIGNAL( pressed() ), SLOT( encryptPassword() ) );
	connect( ui_step2.kurlrequester_config, SIGNAL( urlSelected( const KUrl ) ), SLOT( convertConfigPath( const KUrl ) ) );
	connect( this, SIGNAL( user2Clicked() ), SLOT( nextClicked() ) );
	connect( this, SIGNAL( user1Clicked() ), SLOT( finishClicked() ) );
}
void PasswordAssistant::setPasswordPageValid()
{
	( !ui_step1.klineedit_password->text().isEmpty() ? setValid( passwordPage, true ) : setValid( passwordPage, false ) );
}
void PasswordAssistant::encryptPassword()
{
	if ( ui_step1.klineedit_password->text().trimmed().isEmpty() )
		return;

	QByteArray password;
	password.append( ui_step1.klineedit_password->text() + QString( "\n" ) );

	QProcess md5;
	md5.start( "grub-md5-crypt" );
	md5.waitForStarted();
	md5.write( password );
	md5.write( password );
	md5.closeWriteChannel();
	md5.waitForFinished();

	KMessageBox::information( this, i18n( "The password you entered will now be transformed into an inconceivable line of numbers, letters and special characters. When prompted by GRUB to enter the password, you mustn't enter this line. All you have to enter is the initial password, which is:\n('%1')", ui_step1.klineedit_password->text() ), i18n( "Important Information" ), i18n( "Don't remind me of this again." ) );
	ui_step1.checkBox_encrypt->setChecked( true );
	ui_step1.klineedit_password->setText( QString( md5.readAll() ).section( "\n", 2 ).trimmed() );
}
void PasswordAssistant::convertConfigPath( const KUrl path )
{
	if ( path.isEmpty() )
		return;

	ui_step2.kurlrequester_config->setPath( FileTransactions::convertToGRUBPath( path.path(), KGRUBEditor::DeviceList ) );
}
void PasswordAssistant::nextClicked()
{
	if ( currentPage() == summaryPage )
	{
		ui_summary.label_password->setText( ui_step1.klineedit_password->text() );
		ui_summary.label_md5->setText( ui_step1.checkBox_encrypt->isChecked() ? i18n( "Enabled" ) : i18n( "Disabled" ) );
		ui_summary.label_config->setText( ui_step2.kurlrequester_config->url().path().isEmpty() ? i18n( "<none>" ) : ui_step2.kurlrequester_config->url().path() );
	}
}
void PasswordAssistant::finishClicked()
{
	m_password->setPassword( ui_step1.klineedit_password->text() );
	m_password->setMd5crypted( ui_step1.checkBox_encrypt->isChecked() );
	m_password->setConfigFile( ui_step2.kurlrequester_config->url().path() );
}
