/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA 02110-1301, USA.                                          *
 ***************************************************************************/

//Own
#include "datatypes.h"

//Qt
#include <qregexp.h>

//KDE
#include <klocale.h>

namespace GRUB
{
	namespace ComplexCommand
	{
		//
		//Map
		//
		Map::Map()
		{
			clear();
		}
		Map::Map( const QString toDrive, const QString fromDrive )
		{
			clear();
			m_toDrive = toDrive;
			m_fromDrive = fromDrive;
		}
		Map::Map( const QString map )
		{
			clear();
			resolve( map );
		}
		void Map::clear()
		{
			m_toDrive.clear();
			m_fromDrive.clear();
		}
		void Map::resolve( const QString map )
		{
			m_toDrive = map.section( QRegExp( "\\s+" ), 0, 0 );
			m_fromDrive = map.section( QRegExp( "\\s+" ), 1 );
		}
		const QString Map::result() const
		{
			return i18n( "%1 is mapped to %2", m_fromDrive, m_toDrive );
		}

		//
		//Color
		//
		QMap<int, QString> Color::m_colorMap = Color::mappedColors();
		Color::Color()
		{
			clear();
		}
		Color::Color( const QString color )
		{
			clear();
			resolve( color );
		}
		void Color::clear()
		{
			m_blinkNormal = false;
			m_normalBackground.clear();
			m_normalForeground.clear();
			m_blinkHighlighted = false;
			m_highlightedBackground.clear();
			m_highlightedForeground.clear();
		}
		void Color::resolve( QString color )
		{
			m_blinkNormal = color.section( QRegExp( "\\s+" ), 0, 0 ).section( "/", 0, 0 ).startsWith( "blink-" );
			m_normalBackground = color.section( QRegExp( "\\s+" ), 0, 0 ).section( "/", 1 );
			m_normalForeground = color.section( QRegExp( "\\s+" ), 0, 0 ).section( "/", 0, 0 ).remove( "blink-" );
			m_blinkHighlighted = color.section( QRegExp( "\\s+" ), 1 ).section( "/", 0, 0 ).startsWith( "blink-" );
			m_highlightedBackground = color.section( QRegExp( "\\s+" ), 1 ).section( "/", 1 );
			m_highlightedForeground = color.section( QRegExp( "\\s+" ), 1 ).section( "/", 0, 0 ).remove( "blink-" );
		}
		const QString Color::result() const
		{
			QString result = 
			( m_blinkNormal ? "blink-" : QString() ) + m_normalForeground + ( normalIsEnabled() ? "/" : QString() ) + m_normalBackground 
			+ " " 
			+ ( m_blinkHighlighted ? "blink-" : QString() ) + m_highlightedForeground + ( highlightedIsEnabled() ? "/" : QString() ) + m_highlightedBackground;
			return result;
		}
		const QMap<int, QString> Color::mappedColors()
		{
			QMap<int, QString> colorMap;
			colorMap[0] = "black";
			colorMap[1] = "blue";
			colorMap[2] = "green";
			colorMap[3] = "cyan";
			colorMap[4] = "red";
			colorMap[5] = "magenta";
			colorMap[6] = "brown";
			colorMap[7] = "light-gray";

			colorMap[8] = "dark-gray";
			colorMap[9] = "light-blue";
			colorMap[10] = "light-green";
			colorMap[11] = "light-cyan";
			colorMap[12] = "light-red";
			colorMap[13] = "light-magenta";
			colorMap[14] = "yellow";
			colorMap[15] = "white";
			return colorMap;
		}
		const QString Color::previewOptimisedColor( const QString color ) const
		{
			if ( color == "magenta" )
				return QString( "darkmagenta" );
			else if ( color == "light-magenta" )
				return QString( "magenta" );
			else if ( color == "light-red" )
				return QString( "orangered" );
			else
				return QString( color ).remove( "-" );
		}

		//
		//Password
		//
		Password::Password()
		{
			clear();
		}
		Password::Password( const bool md5crypted, const QString password, const QString configFile )
		{
			clear();
			m_md5crypted = md5crypted;
			m_password = password;
			m_configFile = configFile;			
		}
		Password::Password( const QString password )
		{
			clear();
			resolve( password );
		}
		void Password::clear()
		{
			m_md5crypted = false;
			m_password.clear();
			m_configFile.clear();
		}
		void Password::resolve( QString password )
		{
			if ( password.startsWith( "--md5" ) )
			{
				m_md5crypted = true;
				m_password = password.section( QRegExp( "\\s+" ), 1, 1 );
				m_configFile = password.section( QRegExp( "\\s+" ), 2, 2 );
			}
			else
			{
				m_md5crypted = false;
				m_password = password.section( QRegExp( "\\s+" ), 0, 0 );
				m_configFile = password.section( QRegExp( "\\s+" ), 1, 1 );
			}
		}
		const QString Password::result() const
		{
			return QString( ( m_md5crypted ? "--md5 " : QString() ) + m_password + ( !m_configFile.isEmpty() ? " " + m_configFile : QString() ) );
		}

		//
		//Kernel
		//
		Kernel::Kernel()
		{
			clear();
		}
		Kernel::Kernel( const QString kernel )
		{
			clear();
			resolve( kernel );
		}
		Kernel::Kernel( const QString kernel, const QString arguments )
		{
			m_kernel = kernel;
			m_arguments = arguments;
		}
		void Kernel::clear()
		{
			m_kernel.clear();
			m_arguments.clear();
		}
		void Kernel::resolve( QString kernel )
		{
			m_kernel = kernel.section( QRegExp( "\\s+" ), 0, 0 );
			m_arguments = kernel.section( QRegExp( "\\s+" ), 1 );
		}
		const QString Kernel::result() const
		{
			return QString( m_kernel + " " + m_arguments );
		}
	}

	namespace ConfigFile
	{
		//
		//Entry
		//
		Entry::Entry()
		{
			clear();
		}
		void Entry::clear()
		{
			m_title.clear();
			m_lock = false;
			m_password.clear();
			m_root.clear();
			m_kernel.clear();
			m_initrd.clear();
			m_color.clear();
			m_maps.clear();
			m_chainLoader.clear();
			m_saveDefault = false;
			m_makeActive = false;
		}
		
		//
		//Settings
		//
		Settings::Settings()
		{
			clear();
		}
		void Settings::clear()
		{
			m_splashImage.clear();
			m_gfxMenu.clear();
			m_timeout = -1;
			m_default = -1;
			m_fallback = -1;
			m_hiddenMenu = false;
			m_maps.clear();
			m_color.clear();
			m_password.clear();

			m_automagic.clear();
		}
	}

	namespace Misc
	{
		//
		//Device
		//
		Device::Device()
		{
			clear();
		}
		Device::Device( const QString device, const QString partition, const QString mountpoint )
		{
			clear();
			m_device = device;
			m_partition = partition;
			m_mountPoint = mountpoint;
		}
		void Device::clear()
		{
			m_device.clear();
			m_partition.clear();
			m_grubDevice.clear();
			m_grubPartition.clear();
			m_mountPoint.clear();
			m_uuid.clear();
		}
		//
		//Automagic
		//
		Automagic::Automagic()
		{
			clear();
		}
		void Automagic::clear()
		{
			m_comments.clear();
			m_firstEntry = -1;
			m_lastEntry = -1;
		}
	}
}
