/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA 02110-1301, USA.                                          *
 ***************************************************************************/

#ifndef KGRUBEDITOR_H
#define KGRUBEDITOR_H

//KDE
#include <kcmodule.h>
class KDirWatch;
namespace ThreadWeaver
{
class Job;
}

//KGRUBEditor
class SplashModel;

//Ui
#include "ui/ui_kgrubeditor.h"
#include "ui_settings_paths.h"

/**
 * @short Main window.
 *
 * Handles everything that is GUI related.
 */
class KGRUBEditor : public KCModule, private Ui::KGRUBEditor
{
Q_OBJECT
public:
	/**
	* Default constructor.
	*
	* First, it creates the UI.
	* Secondly, it reads file input.
	* Thirdly, it show the input collected.
	*
	* The last 2 steps are done using refresh().
	*
	* @param parent QWidget parent. Self explanatory.
	*/
	KGRUBEditor( QWidget *parent = 0, const QVariantList &list = QVariantList() );
// protected:
	/**
	* Called before the window is closed, either by the user
	* or indirectly by the session manager.
	*
	* The purpose of this function is to prepare the window
	* in a way that it is safe to close it, i.e. without
	* the user losing some data.
	*/
// 	bool queryClose();
private:
//SETUP
	/**
	* Sets up special objects such as timers etc.
	*/
	void setupObjects();
	/**
	* Connects signals to slots.
	*/
	void setupConnections();
//SHOW FUNCTIONS
//entries
	/**
	* Shows all GRUB entries (with icons for every entry, radio
	* buttons for default etc.).
	*/
	void showEntries( const int entryRow = -1 );
//settings
	/**
	* Shows the hidden menu option.
	*/
	void showHiddenMenu();
	/**
	* Shows the timeout option.
	*/
	void showTimeout();
	/**
	* Shows the password.
	*/
	void showPassword();
private slots:
	/**
	* Shows the selected splash image and selects its preview appropriately.
	*/
	void showSplashImage();
private:
	/**
	* Shows the selected gfxmenu.
	*/
	void showGfxMenu();
	/**
	* Shows the colors with their preview.
	*/
	void showColors();
	/**
	* Shows the maps.
	*/
	void showMaps( const int mapListRow = 0 );
//general
	/**
	* Shows the input collected from all files.
	* Firsty it shows the GRUB Entries in the GRUB Entries page and
	* secondly it shows the GRUB Settings in the GRUB Settings page.
	*
	* It is only called through refresh() so every parameter filtering should be done there.
	*
	* @param entryListRow This row will be selected in the entry listbox after 
        *                     the GRUB Items are shown.
	*                     Defaults to -1 (no entry will be selected if not set).
	* @param mapListRow This row will be selected in the map listbox after 
        *                   the GRUB maps are shown.
	*                   Defaults to 0 (first entry will be selected if not set).
	* @param focusWidget The widget that will receive input focus. 0 = none.
	*/
	void showInput( const int entryRow = -1, const int mapListRow = 0, QWidget* focusWidget = 0 );
//OTHER
	/**
	* Enables or disables some entry-specific action which should
	* be available(=enabled) only when an entry is selected.
	*/
	void enableEntryActions( const bool state );

	/**
	* Moves an entry from source to target in the QVector which
	* contains the entries.
	*
	* @param source The initial position of the entry in the QVector.
	* @param target The final position of the entry in the QVector.
	*/
	void moveEntry( const int source, const int target );

	/**
	* Monitors the specified directory for new splash images
	* and for deletion of existing splash images.
	*
	* @param path The directory to monitor.
	*/
	void monitorSplashDirectory( const QString &directory );
	/**
	* Appends the specified directory to the list of monitored splash
	* directories.
	*
	* The directory will from now on be monitored for new splash images
	* and for deletion of existing splash images.
	*
	* @param path The directory to append.
	*
	* @see monitorSplashDirectory()
	*/
	void appendSplashDirectory( const QString &directory );
private slots:
	/**
	* An entry has been marked as default. Let's save this
	* to the GRUB settings.
	*/
	void defaultUpdated();

	/**
	* A new splash image in the monitored folders was created.
	* Let's append a thumbnail to the list.
	*/
	void splashImageCreated( const QString &path );
	/**
	* A splash image in the monitored folders was deleted.
	* Let's remove its thumbnail from the list.
	*/
	void splashImageDeleted( const QString &path );
	/**
	* A splash image's preview is done.
	*/
	void splashImagePreviewDone( ThreadWeaver::Job *job );

	/**
	* Called when selection changes (either by selecting a new
	* entry or by clicking on empty space thus unselecting the 
	* current entry), in order to enable/disable some actions.
	* (eg. Edit/Remove Entry etc.)
	*
	* @see enableEntryActions(bool).
	*/
	void selectionChanged();
//ENTRIES
	/**
	* Moves the current entry one seat up.
	*/
	void moveUp();
	/**
	* Moves the current entry one seat down.
	*/
	void moveDown();
	/**
	* It launches the Entry Assistant in order to create
	* a new entry.
	*/
	void add();
	/**
	* It launches the QuickEditor in order to quickly-edit
	* the current entry.
	*/
	void quickEdit();
	/**
	* It launches the Entry Assistant in order to fully-edit
	* the current entry.
	*
	* If no entry is selected, it does nothing.
	*/
	void fullEdit();
	/**
	* Removes the current entry after prompting the user.
	*
	* If no entry is selected, it does nothing.
	*/
	void remove();
	/**
	* Shows the attributes of the currently selected entry 
	* using Rich Text formatting.
	*/
	void showDetails();
	/**
	* Switches over to the GRUB Entries List page and sets the 
	* focus to the entry list.
	*/
	void backToList();
//SETTINGS
	/**
	* Called when the hiddenMenu checkbox is clicked,
	* in order to update the file's content.
	*/
	void checkBox_hiddenMenu_clicked();
	/**
	* Called when the timeout checkbox is clicked,
	* in order to enable/disable intSpinBox_timeout.
	*/
	void checkBox_timeout_clicked();
	/**
	* Called in order to update the file's content
	* according to intSpinBox_timeout state and value.
	*/
	void updateTimeout();
	/**
	* It launches the Password Assistant in order to create
	* a new password.
	*/
	void createPassword();
	/**
	* It launches the Password Assistant in order to edit
	* the current password.
	*/
	void editPassword();
	/**
	* It deletes the current password (the user is prompted first).
	*/
	void deletePassword();
	/**
	* Updates the splash image according to the currently selected
	* splash preview.
	*/
	void updateSplashImage( const int previewRow );
	/**
	* Removes the splash image command from the config file.
	*/
	void clearSplashImage();
	/**
	* Opens a file dialog to browse for a GRUB splash image.
	*/
	void browseSplashImage();
	/**
	* Creates a new splash image.
	*/
	void createSplashImage();
	/**
	* Shows the Get-Hot-New-Stuff dialog in order to download
	* new splash images.
	*/
	void getNewSplashImages();
	/**
	* Previews the splash image specified.
	*/
	void previewSplashImage();
	/**
	* Updates the gfxmenu according to the path provided.
	*/
	void updateGfxmenu( const QString &path );
	/**
	* Removes the gfxmenu command from the config file.
	*/
	void clearGfxmenu();
	/**
	* Shows the Get-Hot-New-Stuff dialog in order to download
	* new gfxboots.
	*/
// 	void getNewGfxboot();
	/**
	* Enables the use of colors. Triggered when the 'Normal' check box is checked.
	*/
	void enableNormal( const bool state );
	/**
	* Enables the use of highlighted color. Triggered when the 'Highlight' check box is checked.
	*/
	void enableHighlight( const bool state );
	/**
	* Resets blinking properties to avoid asynchronous blinking.
	*/
	void blinkReset();
	/**
	* Updates the colors used in GRUB.
	*/
	void updateColors();
	/**
	* It launches the Map Assistant in order to add a new map.
	*/
	void addMap();
	/**
	* It launches the Map Assistant in order to edit the current map.
	*/
	void editMap();
	/**
	* It deletes the currently selected map (the user is prompted first).
	*/
	void removeMap();
//TOOLS
	/**
	* Manages Backups using an Assistant.
	*/
	void backup();
	/**
	* Installs GRUB using an Assistant.
	*/
	void install();
	/**
	* Creates a KDialog that shows all Device-relates input collected,
	* such as partition mountpoint, UUID, grub naming etc.
	*
	* For more info chech the Device class:
	* @see GRUB::Misc::Device
	*/
	void viewDevices();
	/**
	* Creates a KDialog that shows the contents of the 2 files, which
	* are read by the application.
	*/
	void viewFiles();
	/**
	* Shows the Configuration dialog.
	*/
	void preferences();
//OTHER
	/**
	* Calls showInput() to update the info shown, after having called 
	* Core::FileIO::readFileInput().
	* Should be called whenever we want to retrieve new file input
	* and update the info currently shown.
	*
	* @see KGRUBEditor::showInput()
	*/
	void refresh( int entryRow = -1, int mapListRow = 0, QWidget* focusWidget = 0 );
	/**
	* This is the same as refresh() but should be called only when the
	* application starts execution. It makes a queued call to
	* createSplashPreviews() after it finishes so as to prevent
	* doubled password dialogs.
	*
	* Since it is called only at the begining, it takes no parameters.
	*/
	void load();
	/**
	* Saves all the changes done to the GRUB Items/Settings
	* to GRUB's configuration file using Core::FileIO::writeFileOutput().
	*/
	void save();
	/**
	* Checks if Settings::menulst() + "_original" exists and if so,
	* it tries to restore from it.
	*/
	void defaults();
private:
	Ui::settings_paths ui_settings_paths;

	KDirWatch *m_splashDirWatchCreation; //monitors splash dirs for new files
	KDirWatch *m_splashDirWatchDeletion; //monitors all splash images in the splash dirs for deletion
	KDirWatch *m_gfxmenuDirWatchDeletion; //monitors the gfxmenu for deletion

	QTimer *timer_blink;
	SplashModel *m_splashModel;

	GRUB::ConfigFile::Settings GrubSettings;
	QVector<GRUB::ConfigFile::Entry> GrubEntries;
};

#endif
