/*
 * Multi-threaded GUI player for audio+video. 
 * Demonstrates the QGstPlay classes.
 */

#include "guiplayer.h"
#include <kde/gst/gstreamer.h>
#include <kde/gstplay/play.h>

#include <qapplication.h>
#include <qthread.h>

using namespace KDE::GST;
using namespace KDE::GSTPlay;

GuiPlayer::GuiPlayer(QApplication *a, const QString &filename) :
	QWidget(0,"GuiPlayer"),
	m_app(a) {

	QBoxLayout *topLayout = new QVBoxLayout(this, 8);
	QBoxLayout *buttonLayout = new QHBoxLayout(topLayout, 8);
	QBoxLayout *sliderLayout = new QHBoxLayout(topLayout, 8);
	QBoxLayout *videoLayout = new QHBoxLayout(topLayout, 8);
	
	m_playButton = new QPushButton("Play", this);
	connect(m_playButton, SIGNAL(clicked()), SLOT(play()));
	m_playButton->setEnabled(true);
	buttonLayout->addWidget(m_playButton);

	m_pauseButton = new QPushButton("Pause", this);
	connect(m_pauseButton, SIGNAL(clicked()), SLOT(pause()));
	m_pauseButton->setEnabled(false);
	buttonLayout->addWidget(m_pauseButton);

	m_stopButton = new QPushButton("Stop", this);
	connect(m_stopButton, SIGNAL(clicked()), SLOT(stop()));
	m_stopButton->setEnabled(false);
	buttonLayout->addWidget(m_stopButton);

	m_slider = new QSlider(0, 1000, 50, 0, Qt::Horizontal, this);
	connect(m_slider, SIGNAL(sliderPressed()), SLOT(pause()));
	connect(m_slider, SIGNAL(sliderReleased()), SLOT(seek()));
	m_slider->setEnabled(false);
	sliderLayout->addWidget(m_slider);

	m_player = new Play(Play::PIPE_VIDEO, this, "Play");
	m_videoScreen = new VideoScreen(m_player, this);
	videoLayout->addWidget(m_videoScreen);

	connect(m_player, SIGNAL(timeTick(long long)), 
		SLOT(updateSlider(long long)));
	connect(m_player, SIGNAL(streamLength(long long)), 
		SLOT(setLength(long long)));
	connect(m_player, SIGNAL(streamEnd()), SLOT(stop()));
	connect(m_player, SIGNAL(haveVideoSize(int,int)), 
		SLOT(haveSize(int,int)));
	m_player->setLocation(filename);
}

GuiPlayer::~GuiPlayer() {
}

/* XVideoSink's have_size callback (called from video thread) */
void GuiPlayer::haveSize(int width, int height) {
	QSize s = sizeHint();
	resize((s.width() > width) ? s.width() : width,
	       s.height() + height);
}

void GuiPlayer::setLength(long long d)
{
	m_duration = d;
}

/* Read position from QSlider and seek... (after slider has been moved) */
void GuiPlayer::seek()
{
	long long real = m_slider->value() * m_duration / 1000LL;

	m_player->seekToTime(real);
}

void GuiPlayer::play()
{
	if (m_player->getState() != Element::STATE_PLAYING)
		m_player->setState(Element::STATE_PLAYING);

	m_playButton->setEnabled(false);
	m_pauseButton->setEnabled(true);
	m_stopButton->setEnabled(true);
	m_slider->setEnabled(true);
}

void GuiPlayer::pause()
{
	if (m_player->getState() != Element::STATE_PAUSED)
		m_player->setState(Element::STATE_PAUSED);

	m_playButton->setEnabled(true);
	m_pauseButton->setEnabled(false);
	m_stopButton->setEnabled(true);
	m_slider->setEnabled(true);
}

void GuiPlayer::stop()
{
	if (m_player->getState() != Element::STATE_READY) 
		m_player->setState(Element::STATE_READY);

	m_playButton->setEnabled(true);
	m_pauseButton->setEnabled(false);
	m_stopButton->setEnabled(false);
	m_slider->setValue(0);
	m_slider->setEnabled(false);
}

void GuiPlayer::updateSlider(long long timeNs) 
{
	if (m_duration > 0)
		m_slider->setValue(timeNs * 1000.0 / m_duration);
}

int main(int argc, char **argv)
{
	if (argc != 2) {
		qDebug("usage: %s <filename>\n", argv[0]);
		return -1;
	}

	// this initialization is optional for Play but recommended
	GStreamer::init(&argc, &argv);

	QApplication a(argc, argv);

	GuiPlayer player(&a, argv[1]);
	player.setCaption("QGstPlay Example");
	a.setMainWidget(&player);
	player.show();

	return a.exec();
}

#include "guiplayer.moc"
