/*
 *  Interface/wrapper for GStreamer GstBin
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_BIN_H
#define KDE_GST_BIN_H

#include <kde/gst/element.h>

namespace KDE {
namespace GST {
/**
 *
 * Bin is the simplest of the container elements, allowing elements to
 * become children of itself.  Pads from the child elements can be ghosted to
 * the bin, making the bin itself look transparently like any other element,
 * allowing for deep nesting of predefined sub-pipelines.
 *
 * Use a @ref Pipeline if you want to create a toplevel bin because a normal 
 * bin doesn't have a scheduler of its own.
 *
 * After the bin has been created you will typically add elements to it with
 * @ref #add(). You can remove elements with @ref #remove().
 *
 * An element can be retrieved from a bin with #ref #getByBame(), using the
 * elements name. @ref #getByNameRecurseUp() is mainly used for internal
 * purposes and will query the parent bins when the element is not found in 
 * the current bin.
 * 
 * The list of elements in a bin can be retrieved with @ref #getList().
 *
 * After the bin has been set to the PLAYING state (with 
 * @ref #setState()), @ref #iterate() is used to process the elements in the 
 * bin.
 *
 * To control the selection of the clock in a bin, you can use the 
 * following methods:
 * @ref #autoClock() to let the bin select a clock automatically, 
 * @ref #getClock() to get the current clock of the bin and 
 * @ref #useClock() to specify a clock explicitly.
 *
 * @short Base container element
 */
	class Bin : public Element {
	Q_OBJECT
	private:
                friend class GStreamer;
                static Wrapper* wrapperFactory(void *real);
		void *reserved;
		
	public:

		enum BinFlags {
		  /* this bin is a manager of child elements, */
		  /* i.e. a pipeline or thread */
		  FLAG_MANAGER		= Element::FLAG_LAST,
		  /* this bin iterates itself */
		  FLAG_SELF_SCHEDULABLE,
		  /* we prefer to have cothreads when its an option, */
		  /* over chain-based */
		  FLAG_PREFER_COTHREADS,
		  FLAG_FIXED_CLOCK,
		  /* padding */
		  FLAG_LAST		= Element::FLAG_LAST + 5
		};
		
/**
 * Creates a new Bin that wrapps the given GstBin. Usually
 * you really don't want to call this, use @ref #wrap instead.
 * You must not create a object of this class on the stack, always 
 * use new.
 * @param real the GstBin to be wrapped
 */
	        Bin(void *real);
	        virtual ~Bin();

/**
 * Creates a new Bin that wrapps the given GstBin. Unlike the
 * Bin constructor this returns the appropriate sub-class for the 
 * GstBin, so when you give a GstPipeline you will get a 
 * QGst::Pipeline instead of a QGst::Bin. And if there is already a 
 * wrapper object it will be re-used. 
 *
 * @param real the GstBin to be wrapped
 * @return the wrapped Bin (or derivative)
 */
		static Bin* wrap(void *real);

/**
 * Create a new bin with given name.
 *
 * @param name name of new bin (or QString::null for auto naming)
 */
		Bin(const QString &name = QString::null);

/**
 * Adds one or mote elements to the bin. Set the elements parent, and thus
 * add a reference.
 *
 * @param elem1 first Element to add to bin
 * @param elem2 second Element to add to bin (can be 0)
 * @param elem3 third Element to add to bin (can be 0)
 * @param elem4 fourth Element to add to bin (can be 0)
 * @param elem5 fifth Element to add to bin (can be 0)
 */
		void add(Element *elem1,
			 Element *elem2 = 0,
			 Element *elem3 = 0,
			 Element *elem4 = 0,
			 Element *elem5 = 0);

/**
 * Remove the element from its associated bin, unparenting as well.
 * The element will also be unreferenced so there's no need to call
 * @ref Object#unref on it.
 * If you want the element to still exist after removing, you need to call
 * @ref Object#ref before removing it from the bin.
 *
 * @param element Element to remove
 */
		void remove(Element *element);

/**
 * Get the element with the given name from this bin.
 *
 * @param name the element name to search for
 * @return the element with the given name
 */
		Element *getByName(const QString &name);

/**
 * Get the element with the given name from this bin. If the
 * element is not found, a recursion is performed on the parent bin.
 *
 * @param name the element name to search for
 * @return the element with the given name
 */
		Element *getByNameRecurseUp(const QString &name);

/**
 * Get the list of elements in this bin.
 *
 * @return a QValueVector with pointers to the elements (shallow copy)
 */
		QValueVector<Element*> getList();

/**
 * Let the bin select a clock automatically.
 */
		void autoClock();

/**
 * Gets the current clock of the (scheduler of the) bin.
 *
 * @return the Clock of the bin
 */
		Clock* getClock();

/**
 * Force the bin to use the given clock. Use NULL to 
 * force it to use no clock at all.
 *
 * @param clock the clock to use.
 */
		void useClock(Clock *clock);

/**
 * Iterates over the elements in this bin.
 *
 * @return TRUE if the bin did something usefull. This value
 *          can be used to determine it the bin is in EOS.
 */
		bool iterate();

/**
 * Attaches a callback which will be run before every iteration of the bin
 * The callback is executed in the Bin's thread.
 *
 * @param func callback function to call
 * @param cookie private data to put in the function call
 */
		void setPreIterateFunction(void (*func)(Bin *bin, void *data), 
					   void *cookie);


/**
 * Attaches a callback which will be run after every iteration of the bin.
 * The callback is executed in the Bin's thread.
 *
 * @param func callback function to call
 * @param cookie private data to put in the function call
 */
		void setPostIterateFunction(void (*func)(Bin *bin, void *data), 
					    void *cookie);
	};
}
}

#endif
