/*
 *  Interface/wrapper for GStreamer GstElementFactory
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_ELEMENTFACTORY_H
#define KDE_GST_ELEMENTFACTORY_H

#include <kde/gst//wrapper.h>

namespace KDE {
namespace GST {
	class Element;
	class Caps;
	class PadTemplate;

	class ElementFactory : public Wrapper {
	Q_OBJECT
	private:
                friend class GStreamer;
                static Wrapper* wrapperFactory(void *real);
		void *reserved;
		
	public:

		static const unsigned int RANK_PRIMARY    = 256;
		static const unsigned int RANK_SECONDARY  = 128;
		static const unsigned int RANK_MARGINAL   = 64;
		static const unsigned int RANK_NONE       = 0;

/**
 * Creates a new ElementFactory that wrapps the given 
 * GstElementFactory. Usually you really don't want to call this, 
 * use @ref #wrap instead.
 * You must not create a object of this class on the stack, always 
 * use new.
 * @param real the GstElementFactory to be wrapped
 */
	        ElementFactory(void *real);
	        virtual ~ElementFactory();

/**
 * Creates a new ElementFactort that wrapps the given GstElementFactory. 
 * If there is already a wrapper object it will be re-used. 
 *
 * @param real the GstElementFactory to be wrapped
 * @return the wrapped ElementFactory
 */
		static ElementFactory* wrap(void *real);

/**
 * Search for an element factory of the given name.
 *
 * @param name name of factory to find
 * @return ElementFactory if found, NULL otherwise
 */
		static ElementFactory* find(const QString &name);

/**
 * Add the given padtemplate to this elementfactory.
 * @param templ the padtemplate to add
 */
		void addPadTemplate(PadTemplate *templ);


/**
 * Checks if the factory can source the given capability.
 *
 * @param caps the caps to check
 * @return true if it can src the capabilities
 */
		bool canSrcCaps(Caps *caps);

/**
 * Checks if the factory can sink the given capability.
 *
 * @param caps the caps to check
 * @return true if it can src the capabilities
 */
		bool canSinkCaps(Caps *caps);

/**
 * Create a new element of the type defined by the given elementfactory.
 * It will be given the name supplied, since all elements require a name as
 * their first argument.
 *
 * @param name name of new element
 * @return the new Element
 */
		Element *create(const QString &name);

/**
 * Create a new element of the type defined by the given element factory.
 * If name is QString::null, then the element will receive a guaranteed unique name,
 * consisting of the element factory name and a number.
 * If name is given, it will be given the name supplied.
 *
 * @param factoryname a named factory to instantiate
 * @param name name of new element or QString::null to auto-naming
 * @return new #GstElement (or NULL if unable to create element)
 */
		static Element* make(const QString &factoryname, 
				     const QString &name = QString::null);

/**
 * Specifies a rank for the element so that autoplugging uses the 
 * most appropriate elements. See also the RANK constants.
 *
 * @param rank rank value - higher number means more priority rank
 */
		void setRank(unsigned short rank);

	};
}
}

#endif
